/*
 * Copyright (c) 1995 PFU Limited.
 *	author Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.lang.CodeReader;
import dejava.lang.DescriptionException;
import java.awt.*;

class Editor extends FocusDialog {
    private static int shiftWidth = 4;
    private static int tabWidth = 8;

    private static Editor shiftWidthDialog;
    private static TextField shiftWidthField;
    private static TextField tabWidthField;
    private static Button okButton;
    private static Button cancelButton;

    private final static String TABWIDTH = "tab";
    private final static String SHIFTWIDTH = "shift";
    private final static String OK = "Ok";
    private final static String CANCEL = "Cancel";

    // region and column
    private static final int START  = 0;
    private static final int END    = 1;
    private static final int REGION_SIZE = 2;
    private static final int COLUMN = 2;
    private static final int REGION_COLUMN_SIZE = 3;

    public Editor(Frame f, boolean modal) {
	super(f, modal);
    }

    private static String space(int size) {
	String space = "";
	for (; size >= tabWidth; size -= tabWidth) {
	    space += "\t";
	}
	for (; size > 0; size--) {
	    space += " ";
	}
	return space;
    }

    private static int spacesAtBOL(String text, int pos)[] {
	// search beginning of line
	int bol = pos;
	while (--bol >= 0 && text.charAt(bol) != '\n') {
	}
	bol++;
	// search non space charactor and calculate column
	int column = 0;
	int nonspace;
	int eol = text.length();
	for (nonspace = bol; nonspace < eol; nonspace++)  {
	    char c = text.charAt(nonspace);
	    if (c == ' ') {
		column++;
	    } else if (c == '\t') {
		column += 8 - (column % 8);
	    } else {
		if (c == '\n') {
		    column = -shiftWidth;
		}
		break;
	    }
	}
	int rval[] = new int[REGION_COLUMN_SIZE];
	rval[START]  = bol;
	rval[END]    = nonspace;
	rval[COLUMN] = column;
	return rval;
    }

    private static void shiftLine(TextArea textArea, int shift) {
	String text = textArea.getText();
	int start = textArea.getSelectionStart();
	int end = textArea.getSelectionEnd();
	if (end > 0 && text.charAt(end - 1) == '\n') end--;
	while (start <= end) {
	    end = shiftLine(textArea, text, shift, end) - 1;
	}
    }

    private static int shiftLine(TextArea textArea, String text, int shift, int pos) {
	int spacesAtBOL[] = spacesAtBOL(text, pos);
	textArea.replaceText(space(spacesAtBOL[COLUMN] + shift),
			     spacesAtBOL[START],
			     spacesAtBOL[END]);
	return spacesAtBOL[START];
    }

    public static void shiftLineLeft(TextArea textArea) {
	shiftLine(textArea, -shiftWidth);
    }

    public static void shiftLineRight(TextArea textArea) {
	shiftLine(textArea, shiftWidth);
    }

    private static int blockRegion(CodeReader codeReader, String text, int pos)[] throws DescriptionException {
	int nest;

	// skip sentences
	nest = codeReader.nextDescription(false);
	if (nest < 0) return null;	// no more block

	// found block, adjust block start to beginning of line
	int start = codeReader.startPoint();
	while (--start >= 0 && text.charAt(start) != '\n') {
	}
	start++;

	int end = text.length();

	for ( ; ; ) {
	    // search inside start of block
	    int insideStart = codeReader.endPoint();
	    while (--insideStart >= 0 && text.charAt(insideStart) != '{') {
	    }
	    insideStart++;

	    // search block end
	    nest = codeReader.nextDescription(false);

	    if (nest < 0) return null;	// no more block

	    // found block end
	    end = codeReader.endPoint();

	    // adjust block end to end of line
	    int endLimit = end;
	    while (--end >= start && text.charAt(end) != '}') {
	    }
	    int insideEnd = end; // found inside block end
	    while (++end < endLimit && text.charAt(end) != '\n') {
	    }
	    if (insideStart <= pos && pos <= insideEnd) {
		// ajust inside block start to beginning of next line
		while (insideStart < insideEnd && text.charAt(insideStart) != '\n') {
		    insideStart++;
		}
		insideStart++;
		// adjust inside block end to end of previous line
		while (--insideEnd > insideStart && text.charAt(insideEnd) != '\n') {
		}
		int rval[] = new int[REGION_SIZE];
		rval[START] = insideStart;
		rval[END] = insideEnd;
		return rval;
	    }

	    if (codeReader.endPoint() > end) break;

	    // check next
	    nest = codeReader.nextDescription();
	    if (nest < 0) return null;
	    if (nest == 0) break;
	}

	int rval[] = new int[REGION_SIZE];
	if (!(start <= pos && pos <= end)) {
	    start = -1;
	    end = -1;
	}
	rval[START] = start;
	rval[END] = end;
	return rval;
    }

    private static int blockRegion(TextArea textArea, int pos)[] {
	return blockRegion(textArea.getText(), pos);
    }

    private static int blockRegion(String text, int pos)[] {
	int start = 0;
	int end = text.length();
	CodeReader codeReader = new CodeReader(text, 1, true);
	int block[];
	try {
	    while ((block = blockRegion(codeReader, text, pos)) != null) {
		if (block[START] == 0) break;	// found in block
		if (block[START] > 0) {		// found in inside of block
		    end = start + block[END];
		    start += block[START];
		    pos -= block[START];
		    text = text.substring(block[START], block[END]);
		    codeReader = new CodeReader(text, 1, true);
		}
	    }
	} catch (DescriptionException e) {
	}
	int rval[] = new int[REGION_SIZE];
	rval[START] = start;
	rval[END] = end;
	return rval;
    }

    private static void shiftBlock(TextArea textArea, int shift) {
	String text = textArea.getText();
	int selectionStart = textArea.getSelectionStart();
	int selectionEnd   = textArea.getSelectionEnd();
	if (selectionEnd > 0 && text.charAt(selectionEnd - 1) == '\n') selectionEnd--;

	int region[] = blockRegion(text, selectionStart);
	int start = region[START];
	int end   = region[END];
	if (end < selectionEnd) {
	    region = blockRegion(text, selectionEnd);
	    if (start > region[START]) start = region[START];
	    if (  end < region[END]  )   end = region[END];
	}
	while (start <= end) {
	    end = shiftLine(textArea, text, shift, end) - 1;
	}
	textArea.getText();
    }

    public static void shiftBlockRight(TextArea textArea) {
	shiftBlock(textArea, shiftWidth);
    }

    public static void shiftBlockLeft(TextArea textArea) {
	shiftBlock(textArea, -shiftWidth);
    }

    // this method did not work correctry becaus AWT may be wrong
    public static void selectBlock(TextArea textArea) {
	String text = textArea.getText();
	int start = textArea.getSelectionStart();
	int end   = textArea.getSelectionEnd();
	int region[] = blockRegion(text, start);
	start = region[START];
	if (start != end) {
	    region = blockRegion(text, end);
	}
	end = region[END];
	textArea.select(start, end);
    }

    public static void beginningOfBlock(TextArea textArea) {
	int pos = blockRegion(textArea, textArea.getSelectionStart())[START];
	textArea.select(pos, pos);
    }

    public static void endOfBlock(TextArea textArea) {
	int pos = blockRegion(textArea, textArea.getSelectionEnd())[END];
	textArea.select(pos, pos);
    }

    public static void setShiftWidth(Frame f) {
	if (shiftWidthDialog != null) {
	    shiftWidthDialog.show();
	    Rectangle parent = f.bounds();
	    Dimension me = shiftWidthDialog.size();
	    shiftWidthDialog.move(parent.x + (parent.width  - me.width ) / 2,
				  parent.y + (parent.height - me.height) / 2);
	    return;
	}
	shiftWidthDialog = new Editor(f, false);
	shiftWidthDialog.setLayout(new BorderLayout());

	shiftWidthField = new TextField(String.valueOf(shiftWidth));
	tabWidthField   = new TextField(String.valueOf(tabWidth));

	Panel panel = new Panel();
	panel.setLayout(new GridLayout(2, 2));
	panel.add(new Label("shift width:"));
	panel.add(shiftWidthField);
	panel.add(new Label("  TAB width:"));
	panel.add(tabWidthField);

	Panel buttons = new Panel();
	buttons.setLayout(new BorderLayout());
	buttons.add("West", new DejavaButton(OK));
	buttons.add("East", new DejavaButton(CANCEL));

	shiftWidthDialog.add("Center", panel);
	shiftWidthDialog.add("South",  buttons);
	shiftWidthDialog.pack();
	shiftWidthDialog.show();
    }

    public boolean action(Event ev, Object arg) {
	if (ev.target instanceof Button) {
	    String button = (String)arg;
	    if (button.equals(OK)) {
		Editor.shiftWidth = Integer.parseInt(Editor.shiftWidthField.getText());
		Editor.tabWidth   = Integer.parseInt(Editor.tabWidthField.getText());
		Editor.shiftWidthDialog.hide();
		return true;
	    } else if (button.equals(CANCEL)) {
		Editor.shiftWidthDialog.hide();
		return true;
	    }
	} else if (ev.target == shiftWidthField) {
	    return false;
	} else if (ev.target == tabWidthField) {
	    return false;
	}
	return super.action(ev, arg);
    }
}
