/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import dejava.util.*;
import java.awt.*;
import java.util.*;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.File;

public class GrepFrame extends ListingCommand {
    private CodeField inputField;
    private Panel locationArea;
    private String command[];
    private static final int PATTERN = 3;
    private static final String egrepChars = ".*^$[]+?|";
    Grep grep;

    static void openOn(CodePath cp, String pattern, String title)
    throws DejavaException {
	GrepFrame frame = new GrepFrame(cp, pattern, title);
	frame.open();
	frame.start();
    }

    public GrepFrame(CodePath cp, String pattern, String title)
    throws DejavaException {
	super(cp, title, "grep");
	initialize(cp, pattern, title);
    }

    protected void initialize(CodePath cp, String pattern, String title)
    throws DejavaException {
	if (title == null) setTitle("Occurence of: '" + pattern + "'");
	grep = new Grep(cp, pattern);
	inputField.setText(pattern);
    }

    protected void createDisplayItems() {
	super.createDisplayItems();
	locationArea = new Panel();
	locationArea.setLayout(new BorderLayout());
	locationArea.add(WEST, new Label("egrep pattern:"));
	locationArea.add(CENTER, inputField = new CodeField());
	listingArea.add(NORTH, locationArea);
    }

    protected void showPattern(boolean show) {
	if (show) {
	    locationArea.show();
	} else {
	    locationArea.hide();
	}
	pack();
    }

    public boolean action(Event ev, Object arg) {
	if (ev.target == inputField) {
	    String pattern = inputField.getText();
	    if (pattern.length() > 0) {
		setTitle("Occurence of: '" + pattern + "'");
		grep.setPattern(pattern);
		restart(grep);
	    }
	    return true;
	} else {
	    boolean ret = super.action(ev, arg);
	    return ret;
	}
    }

    public void start() {
	start(grep);
    }

    protected void doCommand(DataInputStream matches) throws IOException {
	while (commandThread != null) {
	    String line = matches.readLine();
	    if (line == null) return;

	    int first  = line.indexOf(':');
	    if ((SystemManager.onWin32) &&
		first == 1 && (line.charAt(2) == Grep.separatorChar ||
			       line.charAt(2) == File.separatorChar)) {
		char c = line.charAt(0);
		if ('A' <= c && c <= 'Z' || 'a' <= c && c <= 'z') {
		    first = line.indexOf(':', 3);
		}
	    }
	    int second = line.indexOf(':', first + 1);
	    if (first < 0 && second <= first) {
		SystemManager.information(line);
		continue;
	    }

	    String fileName = line.substring(0, first);
	    if (Grep.separator != null) {
		fileName = fileName.replace(Grep.separatorChar, File.separatorChar);
	    }
	    int lineNumber = 0;
	    try {
		lineNumber = parseNumber(line, first + 1, second);
	    } catch (NumberFormatException e) {
		addMessage(line + "\n");
		continue;
	    }

	    int offset;
	    String ss = trimRegexp(grep.pattern());
	    if (ss.charAt(ss.length() - 1) == '$') {
		offset = line.lastIndexOf(ss);
	    } else {
		offset = line.indexOf(ss, second + 1);
	    }
	    if (offset <= second) {
		offset = 0;
	    } else {
		offset -= second + 1;
	    }

	    int separator = line.lastIndexOf(Grep.separatorChar, first);
	    String listString =
		   (separator <= 0 ? line : line.substring(separator + 1));

	    try {
		CodeLocation item = new CodeLocation(fileName, lineNumber,
					    listString, offset, ss.length());
		if (addItem(item)) {
		    list.reshape();
		}
	    } catch (CodeLocationException e) {
		addMessage("fail to search location " +
			   fileName + ":" + lineNumber + ", sorry.\n");

	    }
	}
    }

    protected String trimRegexp(String original) {
	int ssStart, ssEnd;
	String ss = original;
	if ((ssStart = skipEgrepChars(ss, 0)) >= 0) {
	    if ((ssEnd = indexOfEgrepChar(ss, ssStart)) < 0)
		ssEnd = ss.length();
	    ss = ss.substring(ssStart, ssEnd);
	}
	return ss;
    }

    private static int parseNumber(String str, int pos, int end) {
	int number = 0;
	while (pos < end) {
	    char c = str.charAt(pos);
	    if (Character.isDigit(c)) {
		number = number * 10 + (c - '0');
	    }
	    pos++;
	}
	return number;
    }

    private static int isEgrepChar(String str, int start) {
	int len = str.length();
	if (start >= len) return -1;
	char c = str.charAt(start);
	if (c == '\\') {
	    int end = start + 1;
	    if (end >= len) return 1;
	    if (str.charAt(end) != '(') return 2;
	    end++;
	    for (;;) {
		end = str.indexOf('\\', end) + 2;
		if (end < 2 || end >= len) return len - start;
		if (str.charAt(end - 1) == ')') return end - start;
	    }
	}
	if (c == '[') {
	    int end = start + 2;
	    if (end >= len) return len - start;
	    for (;;) {
		end = str.indexOf(']', end) + 1;
		if (end <= 0 || end >= len) return len - start;
		if (str.charAt(end - 2) != '\\') return end - start;
	    }
	}
	if (egrepChars.indexOf(c) >= 0) return 1;
	return 0;
    }

    static private int indexOfEgrepChar(String str, int pos) {
	int max = str.length();
	while (pos < max) {
	    int length = isEgrepChar(str, pos);
	    if (length > 0) return pos;
	    if (length < 0) return -1;
	    pos++;
	}
	return -1;
    }

    static private int skipEgrepChars(String str, int pos) {
	int max = str.length();
	while (pos < max) {
	    int length = isEgrepChar(str, pos);
	    if (length < 0) return -1;
	    if (length == 0) return pos;
	    pos += length;
	}
	return -1;
    }

    void replace(TextArea editArea) {
	replace(editArea, trimRegexp(grep.pattern()));
    }

    protected ListingFrame copyCreate(CodePath cp) {
	try {
	    return new GrepFrame(cp, grep.pattern(), "Same elements for: " + cp.printString());
	} catch (DejavaException e) {
	    DejavaDialog.information(this, "fail to copy frame: " + e.getMessage());
	    return null;
	}
    }
}
