/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import dejava.util.*;

import java.awt.*;
import java.util.Vector;
import java.util.Enumeration;

public class ListingFrame extends DejavaFrame {
    protected Panel listingArea;
    protected DejavaList list;
    private Vector items = new Vector();
    protected int selection = -1;
    private StringBuffer messages = new StringBuffer();
    private ListingReplacer listingReplacer = null;

    ListingFrame(CodePath cp, String title) {
	super(cp, title);
    }

    protected DejavaMenuBar createMenuBar() {
	return new ListingMenuBar();
    }

    protected void createDisplayItems() {
	super.createDisplayItems();

	int listLines = DejavaProperties.getInteger("attribute.listing.lines");
	list = new DejavaList(listLines, false);
	list.setFont(DejavaProperties.getFont("font.list"));

	listingArea = new Panel();
	listingArea.setLayout(new BorderLayout());
	listingArea.add(CENTER, list.makePanel());

	add(NORTH, listingArea);
    }

    /**
     * return current CodePath
     */
    public CodePath codePath() {
	return new CodePath(current);
    }

    public boolean handleEvent(Event ev) {
	if (ev.target == list) {
	    int selected = list.getSelectedIndex();
	    if (ev.id == Event.ACTION_EVENT) {
		if (isMarked(selected)) {
		    unmark(selected);
		} else {
		    mark(selected);
		}
		if (selection != selected) select(selected);
	    } else {
		select(selected);
		if (SystemManager.onWin32) {
		    codeView.requestFocus();
		}
	    }
	    return true;
	}
	return super.handleEvent(ev);
    }

    protected boolean select(int newSelection) {
	CodePath cp;
	if (newSelection < 0 || newSelection == selection) {
	    cp = CodePath.nullCodePath;
	    newSelection = -1;
	} else {
	    ListItem item = itemAt(newSelection);
	    cp = item.codePath();
	    if (selection != -1 && item.isSameCode(itemAt(selection))) {
		// same code, no need to set code, move cursor only
		selection = newSelection;
		list.selectAndVisible(selection);
		int cursor[] = item.cursorPos();
		setCursorPos(cursor[0], cursor[1]);
		return true;
	    }
	}
	if (acceptWithConfirm()) {
	    selection = newSelection;
	    current = cp;
	    list.selectAndVisible(selection);
	    updateCurrent();
	    return true;
	} else {
	    if (selection < 0) {
		list.deselect(list.getSelectedIndex());
	    } else {
		list.select(selection);
	    }
	    return false;
	}
    }

    /**
     * update
     */
    protected void update() {
	if (selection < 0) {
	    int selected = list.getSelectedIndex();
	    if (selected > 0) list.deselect(selected);
	} else {
	    if (!list.isSelected(selection)) list.selectAndVisible(selection);
	}
	super.update();
    }

    protected void updateCode() {
        if (selection == -1) {
            setCode(messages.toString());
	} else {
	    DejavaList save = list;
	    list = null;
	    super.updateCode();
	    list = save;
	    ListItem item = (ListItem)itemAt(selection);
	    int cursor[] = item.cursorPos();
	    setCursorPos(cursor[0], cursor[1]);
	}
    }

    protected void updateButtonActivity() {
	if (selection == -1) {
	    updateButtonActivity(false, false);
	} else {
	    ListItem item = (ListItem)itemAt(selection);
	    updateButtonActivity(item.isChanged() || codeView.isEditable(),
				codeView.isEditable());
	}
    }

    /**
     * set mark to specified ListItem
     */
    void mark(int pos) {
	ListItem item = itemAt(pos);
	if (!item.isMarked()) {
	    item.setMark(true);
	    updateList();
	}
	update();
    }

    /**
     * reset mark to specified ListItem
     */
    void unmark(int pos) {
	ListItem item = itemAt(pos);
	if (item.isMarked()) {
	    item.setMark(false);
	    updateList();
	}
	update();
    }

    /**
     * return marked or not of specified ListItem
     */
    boolean isMarked(int pos) {
	return itemAt(pos).isMarked();
    }

    /**
     * set mark to all ListItem
     */
    void markAll() {
	for (int pos = 0; pos < items.size(); pos++) {
	    ListItem item = itemAt(pos);
	    if (!item.isMarked()) {
		item.setMark(true);
	    }
	}
	updateList();
	update();
    }

    /**
     * reset mark to all ListItem
     */
    void unmarkAll() {
	for (int pos = 0; pos < items.size(); pos++) {
	    ListItem item = itemAt(pos);
	    if (item.isMarked()) {
		item.setMark(false);
	    }
	}
	updateList();
	update();
    }

    void updateList() {
	list.clear();
	for (int pos = 0; pos < items.size(); pos++) {
	    ListItem item = itemAt(pos);
	    list.addItem((item.isMarked() ? "M " : "  ") + item.listString());
	}
	list.reshape();
	if (selection != -1) {
	    list.selectAndVisible(selection);
	}
    }

    protected Enumeration itemElements() {
	return items.elements();
    }

    protected boolean addItem(ListItem item) {
	items.addElement(item);
	list.addItem((item.isMarked() ? "M " : "  ") + item.listString());
	return true;
    }

    protected ListItem itemAt(int pos) {
	return (ListItem)items.elementAt(pos);
    }

    void spawnMatchingItems(CodePath cp, boolean exact) {
	ListingFrame newFrame = copyCreate(cp);
	boolean open = false;
	for (int pos = 0; pos < items.size(); pos++) {
	    ListItem item = itemAt(pos);
	    if (item.codePath().matches(cp, exact)) {
		item = item.cloneItem();
		if (item != null) {
		    newFrame.addItem(item);
		    open = true;
		}
	    }
	}
	if (open) newFrame.open();
    }

    void markMatchingItems(CodePath cp, boolean exact, boolean mark) {
	int pos;
	for (pos = 0; pos < items.size(); pos++) {
	    ListItem item = itemAt(pos);
	    if (item.codePath().matches(cp, exact)) {
		item.setMark(mark);
	    }
	}
	updateList();
    }

    void spawnMarkedItems(boolean marked) {
	ListingFrame newFrame = copyCreate(startup);
	boolean open = false;
	for (int pos = 0; pos < items.size(); pos++) {
	    ListItem item = itemAt(pos);
	    if (item.isMarked() == marked) {
		item = item.cloneItem();
	        if (item != null) {
		    newFrame.addItem(item);
		    open = true;
		}
	    }
	}
	if (open) newFrame.open();
    }

    public synchronized void start(Enumeration items) {
	while (items.hasMoreElements()) {
	    addItem((ListItem)items.nextElement());
	}
	list.reshape();
    }

    protected boolean syncAndClearListItems() {
        if (!select(-1)) {
            SystemManager.information("Could not restart compilation:\n" + startup.printString());
            return false;
        }
        startup.getManager().sync();
	CodePath syncedCp = startup;
	Enumeration cpList = items.elements();
	while (cpList.hasMoreElements()) {
	    CodePath cp = ((ListItem)cpList.nextElement()).codePath();
	    if (cp != null && !cp.matches(syncedCp)) {
		cp.getManager().sync();
		syncedCp = cp;
	    }
        }
	list.clear();
	items.setSize(0);
	messages.setLength(0);
	update();
	return true;
    }

    protected synchronized void restart(Enumeration newItems) {
	if (syncAndClearListItems()) {
	    start(newItems);
	}
    }

    protected String getMessages() {
	return messages.toString();
    }

    protected void addMessage(String message) {
	messages.append(message);
	if (selection == -1) {
            setCode(messages.toString());
	}
    }

    void close() {
	if (listingReplacer != null) {
	    listingReplacer.hide();
	    listingReplacer.dispose();
	    listingReplacer = null;
	}
	super.close();
    }

    protected ListingFrame copyCreate(CodePath cp) {
	return new ListingFrame(cp, "Same elements for: " + cp.printString());
    }

    void replace(TextArea editArea) {
	replace(editArea, null);
    }

    void replace(TextArea editArea, String searchString) {
	if (listingReplacer == null) {
	    listingReplacer = new ListingReplacer(this, editArea);
	}
	if (searchString != null) {
	    listingReplacer.setSearchString(searchString);
	}
	listingReplacer.show();
    }
}
