/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.gui;

import dejava.sys.*;
import java.util.*;
import java.io.*;
import java.awt.*;

public class Transcript extends DelegateEventFrame implements Runnable {

    private static Transcript transcript;

    private static Thread messageThread;
    private static Vector messages = new Vector();

    private static TranscriptMenuBar menuBar;
    private static TextArea textArea;
    private static MemoryMonitor memoryMonitor;
    private static byte buf[] = new byte[100];
    public static String titleString = "System Transcript";
    static {
	transcript = new Transcript(titleString);
	transcript.start();
    }

    public static Transcript getTranscript() {
	return transcript;
    }

    public static TranscriptMenuBar getTranscriptMenuBar() {
	return (TranscriptMenuBar)transcript.getMenuBar();
    }

    public static Frame getDefaultFrame() {
	Frame f = DejavaFrame.getCurrentFrame();
	return f != null ? f : transcript;
    }

    static void raise() {
	transcript.show();
	transcript.requestFocus();
    }

    private Transcript(String title) {
	super(title);

	setFont(DejavaProperties.getFont("font.default"));
	setBackground(Color.lightGray);
	menuBar = new TranscriptMenuBar();
	menuBar.addFrame(this);
	setMenuBar(menuBar);

	textArea = new CodeArea(15, 80);
	textArea.setBackground(Color.lightGray);
	textArea.setEditable(true);
	add("Center", textArea);

	pack();
    }

    private void start() {
	show();
	messageThread = new Thread(this);
	messageThread.setPriority(Thread.MAX_PRIORITY);
	messageThread.start();
    }

    private synchronized void stop() {
	messageThread.stop();
	messageThread = null;
    }

    void close() {

	if (!DejavaDialog.confirm(transcript,
		  "You're going to quit dejava system. Really?",
		  "Quit", "Cancel")) {
	    return;
	}

	Enumeration en = DejavaFrame.allFrames();
	while (en.hasMoreElements()) {
	    DejavaFrame frame = (DejavaFrame)en.nextElement();
	    if (!frame.tryClose()) {
		DejavaDialog.information(this, "Could not close some browsers.\nQuit aborted.");
		return;
	    }
	}

	Enumeration changes = SourceFileManager.changedSources();
	if (changes.hasMoreElements()) {
	    StringBuffer files = new StringBuffer("");
	    do {
		SourceFileManager sfm = (SourceFileManager)changes.nextElement();
		files.append("  ");
		files.append(sfm.absolutePath());
		files.append('\n');
	    } while (changes.hasMoreElements());
	    int chosen =  DejavaDialog.choose(transcript,
		"Following files are changed:\n" + files.toString() +
		"Sync them?", "Sync and quit", "No sync and quit", "Cancel");
	    if (chosen == 3) return;
	    if (chosen == 1) SourceFileManager.syncAllChangedSources();
	}

	stop();
	System.exit(0);
    }

    public boolean handleEvent(Event e) {
	if (e.id == Event.WINDOW_DESTROY) {
	    close();
	    return true;
	}
	return super.handleEvent(e);
    }

    public boolean action(Event ev, Object arg) {
	return menuBar.action(ev, arg, this);
    }

    public void message(String str) {
	putMessage(new StringBufferInputStream(str));
    }

    public void message(InputStream stream) {
	putMessage(stream);
    }

    public void error(String str) {
	SystemManager.beep();
	textArea.insertText(str, textArea.getText().length());
    }

    private synchronized void putMessage(InputStream stream) {
	messages.addElement(stream);
	messageThread.resume();
	Thread.yield();
    }

    private InputStream getMessage() {
	if (messages.size() == 0) {
	    return null;
	} else {
	    InputStream message = (InputStream)messages.firstElement();
	    messages.removeElementAt(0);
	    return message;
	}
    }

    public String getText() {
	return textArea.getText();
    }

    public void run() {
	Thread currentThread = Thread.currentThread();
	if (currentThread == messageThread) {
	    while (messageThread != null) {
		InputStream stream;
		while ((stream = getMessage()) != null) {
		    try {
			int len;
			while ((len = stream.read(buf)) != -1) {
			    String str = new String(buf, 0, 0, len);
			    textArea.insertText(str, textArea.getText().length());
			    Thread.sleep(10);
			}
		    } catch (IOException e) {
		    } catch (InterruptedException e) {
		    }
		    try {
			stream.close();
		    } catch (IOException e) {
		    }
		}
		messageThread.suspend();
	    }
	}
    }

    void memoryMonitor() {
	if (memoryMonitor == null) {
	    memoryMonitor = new MemoryMonitor();
	}
	memoryMonitor.show();
    }

    public static void addProjectToMenu(String projname) {
	transcript.menuBar.addProject(projname);
    }

    boolean removeProject(String projname) {
	ProjectManager proj = SystemManager.findProject(projname);
	Enumeration frames = DejavaFrame.allFrames();
	while (frames.hasMoreElements()) {
	    DejavaFrame frame = (DejavaFrame)frames.nextElement();
	    if (frame.current.projectManager() == proj) {
		DejavaDialog.information(this, "Project browser on " +
			    projname +
			    " is still opened.\nClose it and try again.");
		return false;
	    }
	}
	if (DejavaDialog.confirm(this, "remove " + projname + " project, really?")) {
	    proj.remove();
	    return true;
	}
	return false;
    }

    public static void main(String argv[]) {
	transcript.start();
    }
}

class TranscriptMenuBar extends MenuBar {

    DejavaMenu fileMenu;
    DejavaMenu projectMenu;
    DejavaMenu removeMenu;
    DejavaMenu renameMenu;
    DejavaMenu frameMenu;

    protected static final String MENU_FILE    = "File";
    protected static final String MENU_PROJECT = "Project";
    protected static final String MENU_FRAME   = "Frame";

    protected static final String MENU_OPEN_CHANGES   = "open changes";
    protected static final String MENU_MEMORY_MONITOR = "memory monitor";
    protected static final String MENU_QUIT           = "quit";
    protected static final String MENU_NEW_PROJECT    = "new project";
    protected static final String MENU_REMOVE_PROJECT = "remove project";
    protected static final String MENU_RENAME_PROJECT = "rename project";
    protected static final String MENU_SPAWN_PROJECT  = "spawn ";

    TranscriptMenuBar() {
	fileMenu    = createFileMenu();
	projectMenu = createProjectMenu();
	frameMenu  = createFrameMenu();
	add(fileMenu);
	add(projectMenu);
	add(frameMenu);
    }

    public DejavaMenu createFileMenu() {
	DejavaMenu menu = new DejavaMenu(MENU_FILE);
	menu.add(MENU_OPEN_CHANGES);
	menu.add(MENU_MEMORY_MONITOR);
	menu.add(MENU_QUIT);
	return menu;
    }

    public DejavaMenu createProjectMenu() {

	removeMenu = new DejavaMenu(MENU_REMOVE_PROJECT);
	removeMenu.disable();
	renameMenu = new DejavaMenu(MENU_RENAME_PROJECT);
	renameMenu.disable();

	DejavaMenu menu = new DejavaMenu(MENU_PROJECT);
	menu.add(MENU_NEW_PROJECT);
	menu.add(removeMenu);
	menu.add(renameMenu);
	menu.addSeparator();
	// spawn project menu will be added by SystemManager call addProject()
	return menu;
    }

    public DejavaMenu createFrameMenu() {
	return new DejavaMenu(MENU_FRAME);
    }

    void addProject(String projname) {
	projectMenu.add(MENU_SPAWN_PROJECT + projname);
	removeMenu.add(projname);
	removeMenu.enable();
	renameMenu.add(projname);
	renameMenu.enable();
    }

    public void renameProject(String name, String newName) {
	int numItems = projectMenu.countItems();
	for (int i = 3; i < numItems; i++) {
	    MenuItem item = projectMenu.getItem(i);
	    if (item.getLabel().endsWith(" " + name)) {
		item.setLabel(MENU_SPAWN_PROJECT + newName);
		break;
	    }
	}
	numItems = removeMenu.countItems();
	for (int i = 0; i < numItems; i++) {
	    MenuItem item = removeMenu.getItem(i);
	    if (item.getLabel().equals(name)) {
		item.setLabel(newName);
		break;
	    }
	}
	numItems = renameMenu.countItems();
	for (int i = 0; i < numItems; i++) {
	    MenuItem item = renameMenu.getItem(i);
	    if (item.getLabel().equals(name)) {
		item.setLabel(newName);
		break;
	    }
	}
    }

    void addFrame(Frame frame) {
	frameMenu.add(new FrameRaiseMenuItem(frame));
    }

    void removeFrameAt(int index) {
	frameMenu.remove(index + 1);	// +1 for Transcript
    }

    public boolean action(Event ev, Object arg, Transcript transcript) {
	if (ev.target instanceof MenuItem) {
	    Menu menu = (Menu)((MenuItem)ev.target).getParent();
	    String label = (String)arg;
	    if (menu == fileMenu) {
		if (label.equals(MENU_OPEN_CHANGES)) {
		    File changesFile = SystemManager.changesFile();
		    String dirname = changesFile.getParent();
		    String fname = changesFile.getName();
		    fname = SystemManager.requestFile("Changes file to open", dirname, fname);
		    if (fname != null) {
			ChangeListFrame.openOn(fname);
		    }
		} else if (label.equals(MENU_MEMORY_MONITOR)) {
		    transcript.memoryMonitor();
		} else if (label.equals(MENU_QUIT)) {
		    transcript.close();
		}
		return true;
	    } else if (menu == projectMenu) {
		if (label.equals(MENU_NEW_PROJECT)) {
		    NewProjectDialog newProject = new NewProjectDialog(transcript);
		    if (!newProject.confirm()) return true;
		    SystemManager.createProject(newProject.projectName(),
					     newProject.projectPath(),
					     newProject.rebuildNow());
		} else if (label.startsWith(MENU_SPAWN_PROJECT)) {
		    String proj = label.substring(MENU_SPAWN_PROJECT.length());
		    CodePath cp = new CodePath(null,
					    SystemManager.findProject(proj),
					    null, null, null);
		    Browser.openOn(cp);
		}
		return true;
/*
	    } else if (menu == frameMenu) {
		int colon = label.indexOf(':');
		int index = Integer.parseInt(label.substring(0, colon));
		if (index == 0) {
		    Transcript.raise();
		} else {
		    DejavaFrame.frameAt(index - 1).show();
		}
		return true;
 */
	    } else if (menu == removeMenu) {
		if (transcript.removeProject(label)) {
		    removeMenu.remove((MenuComponent)ev.target);
		    renameMenu.remove(renameMenu.indexOf((String)arg));
		    projectMenu.remove(projectMenu.indexOf("spawn " + (String)arg));
/*
		    while (--n >= 0) {
			if (label.equals(removeMenu.getItem(n).getLabel())) {
			    projectMenu.remove(projectMenu.indexOf("spawn " + );
			    removeMenu.remove(n);
			    break;
			}
		    }
*/
		    if (removeMenu.countItems() == 0) removeMenu.disable();
		}
		return true;
	    } else if (menu == renameMenu) try {
		SystemManager.findProject(label).rename();
		return true;
	    } catch (DejavaException e) {
	    }
	}
	return false;
    }
}
