/*
 * Copyright (c) 1995 PFU Limited.
 *	author Kazuhisa Shirakami
 */
package dejava.lang;

public class AccessFlags {

    private int flags = 0;

    public static final int PUBLIC       = 0x0001;
    public static final int PRIVATE      = 0x0002;
    public static final int PROTECTED    = 0x0004;
    public static final int STATIC       = 0x0008;
    public static final int FINAL        = 0x0010;
    public static final int SYNCHRONIZED = 0x0020;
    public static final int THREADSAFE   = 0x0040;
    public static final int TRANSIENT    = 0x0080;
    public static final int NATIVE       = 0x0100;
    public static final int INTERFACE    = 0x0200;
    public static final int ABSTRACT     = 0x0400;
    public static final int NON_STATIC = (STATIC << 16);
    public static final int FRIENDLY   = ((PUBLIC | PROTECTED | PRIVATE) << 16);

    public static final String STR_PUBLIC       = "public";
    public static final String STR_PRIVATE      = "private";
    public static final String STR_PROTECTED    = "protected";
    public static final String STR_STATIC       = "static";
    public static final String STR_FINAL        = "final";
    public static final String STR_SYNCHRONIZED = "synchronized";
    public static final String STR_THREADSAFE   = "threadsafe";
    public static final String STR_TRANSIENT    = "transient";
    public static final String STR_NATIVE       = "native";
    public static final String STR_ABSTRACT     = "abstract";

    int signatures() {
	return flags & ~INTERFACE;
    }

    String signaturesString() {
	return signaturesString(flags);
    }

    static String signaturesString(int flags) {
	String signatures = "";
	if ((flags & PUBLIC      ) != 0) signatures += STR_PUBLIC       + " ";
	if ((flags & PRIVATE     ) != 0) signatures += STR_PRIVATE      + " ";
	if ((flags & PROTECTED   ) != 0) signatures += STR_PROTECTED    + " ";
	if ((flags & FINAL       ) != 0) signatures += STR_FINAL        + " ";
	if ((flags & NATIVE      ) != 0) signatures += STR_NATIVE       + " ";
	if ((flags & SYNCHRONIZED) != 0) signatures += STR_SYNCHRONIZED + " ";
	if ((flags & THREADSAFE  ) != 0) signatures += STR_THREADSAFE   + " ";
	if ((flags & TRANSIENT   ) != 0) signatures += STR_TRANSIENT    + " ";
	if ((flags & STATIC      ) != 0) signatures += STR_STATIC       + " ";
	if ((flags & ABSTRACT    ) != 0) signatures += STR_ABSTRACT     + " ";
	return signatures;
    }

    int getAccessFlags() {
	return flags;
    }

    void setAccessFlags(int flags) {
	this.flags = flags;
    }

    public void enableAccessFlags(int flags) {
	this.flags |= flags;
    }

    void disableAccessFlags(int flags) {
	this.flags &= ~flags;
    }

    boolean isEnableAccessFlags(int flag) {
	return (flags & flag) == flag;
    }

    boolean eachEnableAccessFlags(int flag) {
	return (flags & flag) != 0;
    }

    static int nameToFlag(String name) {
	if (name.equals(STR_PUBLIC)) {
	    return PUBLIC;
	} else if (name.equals(STR_PRIVATE)) {
	    return PRIVATE;
	} else if (name.equals(STR_PROTECTED)) {
	    return PROTECTED;
	} else if (name.equals(STR_STATIC)) {
	    return STATIC;
	} else if (name.equals(STR_FINAL)) {
	    return FINAL;
	} else if (name.equals(STR_SYNCHRONIZED)) {
	    return SYNCHRONIZED;
	} else if (name.equals(STR_THREADSAFE)) {
	    return THREADSAFE;
	} else if (name.equals(STR_TRANSIENT)) {
	    return TRANSIENT;
	} else if (name.equals(STR_NATIVE)) {
	    return NATIVE;
	} else if (name.equals(STR_ABSTRACT)) {
	    return ABSTRACT;
	}
	return 0;
    }

    boolean enableAccessFlag(String name) {
	int flag = nameToFlag(name);
	if (flag != 0) {
	    enableAccessFlags(flag);
	    return true;
	} else {
	    return false;
	}
    }

    boolean disableAccessFlag(String name) {
	int flag = nameToFlag(name);
	if (flag != 0) {
	    disableAccessFlags(flag);
	    return true;
	} else {
	    return false;
	}
    }

    boolean isInterface() {
	return (flags & INTERFACE) == INTERFACE;
    }

    protected boolean inhibit(int target) {
	if ((flags & target & 0xffff) != 0) return true;
	if ((flags & NON_STATIC) == NON_STATIC && (target & STATIC) == 0) return true;
	if ((flags & FRIENDLY) == FRIENDLY && (target & (PRIVATE|PROTECTED|PUBLIC)) == 0) return true;
	return false;
    }

    public String toString() {
	return getClass() + "[flags=" + flags
			  + ",signatures=" + signaturesString()
			  + "]";
    }
}
