/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.lang;

import java.util.StringTokenizer;

public class ClassSpec extends AccessFlags {
    private String signatures = null;
    private String className;
    private String superClassName;
    private String interfaceNames;
    private boolean isInterface;

    public static final String STR_CLASS      = "class";
    public static final String STR_INTERFACE  = "interface";
    public static final String STR_EXTENDS    = "extends";
    public static final String STR_IMPLEMENTS = "implements";

    private static final int SIGNATURES = 0;
    private static final int CLASSNAME = 1;
    private static final int EXTENDS_OR_IMPLEMENTS = 2;
    private static final int SUPERCLASSNAME = 3;
    private static final int IMPLEMENTS = 4;
    private static final int INTERFACES = 5;

    public static final String NO_CLASS_OR_INTERFACE =
				"'class' or 'interface' expected";
    private static final String MULTIPLE_CLASS_INTERFACE =
				"multiple defined 'class' and/or 'interface'";
    private static final String NO_CLASSNAME =
				"class name expected";
    private static final String NO_EXTENDS_OR_IMPLEMENTS =
				"'extends' or `implements' expected";
    private static final String ILLEGAL_EXTENDS =
				"illegal position of 'extends'";
    private static final String NO_SUPERCLASSNAME =
				"super class name expected";
    private static final String NO_IMPLEMENTS =
				"'implements' expected";
    private static final String ILLEGAL_IMPLEMENTS =
				"illegal position of 'implements'";
    private static final String NO_INTERFACES =
				"interfaces expected";
    private static final String SYNTAX_ERROR =
				"syntax error for class description";

    public static ClassSpec fromString(String description) {
	try {
	    return new ClassSpec(description);
	} catch (DescriptionException e) {
	    return null;
	}
    }

    public ClassSpec(String className, String superClassName, String interfaceNames, boolean isInterface) {
	super();
	this.className = className;
	this.superClassName = superClassName;
	this.interfaceNames = interfaceNames;
	this.isInterface = isInterface;
	if (className != null) {
	    int dot = className.lastIndexOf('.');
	    if (dot >= 0) this.className = className.substring(dot + 1);
	}
    }

    public ClassSpec(String description) throws DescriptionException {
	super();
	String error = setSpec(description);
	if (error != null) {
	    throw new DescriptionException(error);
	}
    }

    private String setSpec(String description) {
	setAccessFlags(0);
	className = null;
	superClassName = null;
	interfaceNames = null;
	isInterface = false;

	int setting = SIGNATURES;
	StringTokenizer st = new StringTokenizer(description);
	while (st.hasMoreTokens()) {
	    String token = st.nextToken();
	    // check keywords
	    if (token.equals(ClassSpec.STR_CLASS)) {
		if (setting != SIGNATURES) return MULTIPLE_CLASS_INTERFACE;
		isInterface = false;
		setting = CLASSNAME;
	    } else if (token.equals(ClassSpec.STR_INTERFACE)) {
		if (setting != SIGNATURES) return MULTIPLE_CLASS_INTERFACE;
		isInterface = true;
		setting = CLASSNAME;
	    } else if (token.equals(ClassSpec.STR_EXTENDS)) {
		if (setting < EXTENDS_OR_IMPLEMENTS) break;
		if (setting != EXTENDS_OR_IMPLEMENTS) return ILLEGAL_EXTENDS;
		setting = SUPERCLASSNAME;
	    } else if (token.equals(ClassSpec.STR_IMPLEMENTS)) {
		if (isInterface && setting == SUPERCLASSNAME &&
						    superClassName != null) {
		    setting = IMPLEMENTS;
		}
		if (setting < IMPLEMENTS &&
		    setting != EXTENDS_OR_IMPLEMENTS) break;
		if (setting >= INTERFACES) return ILLEGAL_IMPLEMENTS;
		setting = INTERFACES;
	    } else {
		if (token.indexOf('(') >= 0 ||
		    token.indexOf(')') >= 0) return SYNTAX_ERROR;
		switch (setting) {
		case SIGNATURES:
		    enableAccessFlag(token);
		    signatures = (signatures == null) ?
					    token :
					    (signatures + " " + token);
		    break;
		case CLASSNAME:
		    int dot = token.lastIndexOf('.');
		    className = token.substring(dot + 1);
		    setting = EXTENDS_OR_IMPLEMENTS;
		    break;
		case EXTENDS_OR_IMPLEMENTS:
		    return NO_EXTENDS_OR_IMPLEMENTS;
		case SUPERCLASSNAME:
		    if (isInterface) {
			if (superClassName == null) {
			    superClassName = token;
			} else {
			    if (superClassName.endsWith(",") || token.startsWith(",")) {
				superClassName += token;
			    } else {
				return SYNTAX_ERROR;
			    }
			}
		    } else {
			superClassName = token;
			setting = IMPLEMENTS;
		    }
		    break;
		case IMPLEMENTS:
		    return NO_IMPLEMENTS;
		case INTERFACES:
		    interfaceNames = (interfaceNames == null) ?
					    token :
					    (interfaceNames + token);
		    break;
		}
	    }
	}
	if (setting == SIGNATURES    ) return NO_CLASS_OR_INTERFACE;
	if (setting == CLASSNAME     ) return NO_CLASSNAME;
	if (setting == SUPERCLASSNAME && !isInterface) return NO_SUPERCLASSNAME;
	if (setting == INTERFACES &&
	    interfaceNames == null   ) return NO_INTERFACES;
	int illegal = JavaLexical.isIdentifier(className);
	if (illegal >=  0) return "illegal character '" +
				  className.charAt(illegal) +
				  "' for " + typeAsString() + " name";
	return null;
    }

    public void rename(String newName) {
	className = newName;
	int dot = className.lastIndexOf('.');
	if (dot >= 0) className = className.substring(dot + 1);
    }

    public boolean isAbstract() {
	return ((getAccessFlags() & ABSTRACT) == ABSTRACT);
    }

    public boolean isPublic() {
	return ((getAccessFlags() & PUBLIC) == PUBLIC);
    }

    public boolean isFinal() {
	return ((getAccessFlags() & FINAL) == FINAL);
    }

    public boolean isInterface() {
	return isInterface;
    }

    public void setPublic(boolean isPublic) {
	if (isPublic) {
	    enableAccessFlags(PUBLIC);
	} else {
	    disableAccessFlags(PUBLIC);
	}
    }

    public void setFinal(boolean isFinal) {
	if (isFinal) {
	    enableAccessFlags(FINAL);
	} else {
	    disableAccessFlags(FINAL);
	}
    }

    public void setAbstract(boolean isAbstract) {
	if (isAbstract) {
	    enableAccessFlags(ABSTRACT);
	} else {
	    disableAccessFlags(ABSTRACT);
	}
    }

    public String definition() {
	String buf = super.signaturesString();
	if (buf.length() != 0) buf += " ";
	buf += typeAsString() + " " + className;
	if (superClassName != null && superClassName.length() != 0) {
	    buf += " " + STR_EXTENDS + " " + superClassName;
	}
	if (interfaceNames != null && interfaceNames.length() != 0) {
	    buf += " " + STR_IMPLEMENTS + " " + interfaceNames;
	}
	return buf;
    }

    public int signatures() {
	return super.signatures();
    }

    public String className() {
	return className;
    }

    public String superClassName() {
	return superClassName;
    }

    public String interfaceNames() {
	return interfaceNames;
    }

    public String description() {
	return	signaturesString() +
		typeAsString() + " " + className +
		(superClassName == null ? "" :
					" " + STR_EXTENDS + " " + superClassName) +
		(interfaceNames == null ? "" :
				     " " + STR_IMPLEMENTS + " " + interfaceNames);
    }

    public String typeAsString() {
	return isInterface() ? STR_INTERFACE : STR_CLASS;
    }

    public void dispose() {
	signatures = null;
	className = null;
	superClassName = null;
	interfaceNames = null;
    }
}
