/*
 * Copyright (c) 1995 PFU Limited.
 *	author Kazuhisa Shirakami
 */
package dejava.lang;

import java.lang.*;
import java.io.*;
import java.util.*;

/**
 * code reader -- read a java source code with analyzing nest by brace block
 *	Method nextDescription() returns nest level of brace block after search
 *	statement or open/close brace charactor.
 *	Method description() returns found description
 *
 *	example (max nest leves is 2):
 *	--------------------
 *	package test;		-> nest is 0, description is "package test "
 *	class Test {		-> nest is 1, description is "class Test "
 *	    Test() {		-> nest is 2, description is "Test "
 *		super();	-> nest is 2, description is "super() "
 *		if (...) {	-> skip
 *		    :		-> skip
 *		}		-> skip
 *	    }			-> nest is 1
 *	}			-> nest is 0
 *	class Main {		-> nest is 1, description is "class Main "
 *	--------------------
 */
public class CodeReader {
    // 0: statements, 1: class or method, 2: class and method
    private int nestMax;

    // return statement (will return nest level same as last returned it)
    private boolean needSentence;

    private String code;
    private int codeLength;

    private int codePointer = 0;
    private int lineNumber = 1;
    private int beginningOfLinePointer = 0;
    private int descriptionPointer = 0;
    private StringBuffer description = new StringBuffer(160);

    private int startPoint = 0;	 	// start file pointer
    private int endPoint   = 0;		// next start file pointer
    private int startLine  = 1;		// start line number
    private int endLine    = 1;		// end line number

    private int nest = 0;		// nest level
    private boolean inComment = false;
    private char inQuote = '\0';

    public CodeReader(FileInputStream stream) throws IOException {
	this(stream, 2, true);
    }

    public CodeReader(FileInputStream stream, int max, boolean need_sentence)
    throws IOException {
	nestMax = max;		// 1:class or method, 2:class and method
	needSentence = need_sentence;
	codeLength = stream.available();
	byte byteBuf[] = new byte[codeLength];
	stream.read(byteBuf);
	code = new String(byteBuf, 0);
	stream.close();
    }

    public CodeReader(String str) { // for editing class or method description
	this(str, 1, true);
    }

    public CodeReader(String str, int max, boolean need_sentence) {
	nestMax = max;			// 1:class 2:class&method
	needSentence = need_sentence;
	code = str;
	codeLength = str.length();
    }

    public int nextDescription() throws DescriptionException {
	return nextDescription(needSentence);
    }

    public int nextDescription(boolean needSentence) throws DescriptionException {
	if (code == null) {
	    return -1; // already found end of code
	}
	resetDescription();
	int parenthesis = 0;
	while (codePointer < codeLength) {
	    char c = code.charAt(codePointer);
	    if (inComment) {
		if (code.startsWith("*/", codePointer)) {
		    codePointer += 2;
		    descriptionPointer = codePointer;
		    inComment = false;
		} else {
		    codePointer++;
		}
	    } else if (inQuote != '\0') {
		// in single quote or double quote
		if (c == '\\') {
		    codePointer += 2;
		} else {
		    codePointer++;
		    if (c == inQuote) {
			inQuote = '\0';
		    }
		}
	    } else if (code.startsWith("//", codePointer)) {
		setDescription();
		codePointer = code.indexOf('\n', codePointer);
		if (codePointer < 0) {
		    codePointer = codeLength;
		    beginningOfLinePointer = codeLength;
		}
		descriptionPointer = codePointer;
	    } else if (code.startsWith("/*", codePointer)) {
		setDescription();
		codePointer += 2;
		inComment = true;
	    } else if (c == '"' || c == '\'') {
		codePointer++;
		inQuote = c;
	    } else if (c == ';' && parenthesis == 0) {
		if (needSentence) setDescription();
		codePointer++;
		adjustEndPoint();
		if (needSentence && nest <= nestMax) {
		    //System.out.println("statement: " + toString());
		    return nest;
		}
		resetDescription();
	    } else if (c == '{') {
		if (parenthesis != 0) {
		    throw new DescriptionException("not close parenthesis");
		}
		setDescription();
		codePointer++;
		adjustEndPoint();
		if (++nest <= nestMax) {
		    //System.out.println("begin: " + toString());
		    return nest;
		}
		resetDescription();
	    } else if (c == '}') {
		if (parenthesis != 0) {
		    throw new DescriptionException("not close parenthesis");
		}
		startPoint = codePointer;
		startLine  = lineNumber;
		codePointer++;
		adjustEndPoint();
		description.setLength(0);
		if (--nest < 0) {
		    throw new DescriptionException("too many close brace");
		} else if (nest < nestMax) {
		    //System.out.println("end: " + toString());
		    return nest;
		}
	    } else if (c == '(') {
		codePointer++;
		parenthesis++;
	    } else if (c == ')') {
		codePointer++;
		parenthesis--;
		if (parenthesis < 0) {
		    throw new DescriptionException("too many close parenthesis");
		}
	    } else if (c <= ' ') {
		if (descriptionPointer < codePointer) setDescription();
		codePointer++;
		descriptionPointer = codePointer;
	    } else {
		codePointer++;
	    }
	    if (c == '\n') {
		beginningOfLinePointer = codePointer;
		lineNumber++;
	    }
	}

	// found end of code
	setDescription();
	code = null; // for free code area
	if (inComment) {
	    throw new DescriptionException("not found end of comment");
	} else if (inQuote == '\'') {
	    throw new DescriptionException("not found end of single quote");
	} else if (inQuote == '\"') {
	    throw new DescriptionException("not found end of double quote");
	} else if (nest != 0) {
	    throw new DescriptionException("not found matching close brace");
	} else if (description().length() != 0) {
	    throw new DescriptionException("syntax error near the end");
	}
	endPoint = codeLength;
	endLine = lineNumber;
	nest = -1;
	//System.out.println("eof: " + toString());
	return -1;
    }

    private void resetDescription() {
	startPoint = endPoint;
	startLine  = lineNumber;
	description.setLength(0);
    }

    private void setDescription() {
	if (descriptionPointer < codePointer && nest <= nestMax) {
	    String word = code.substring(descriptionPointer, codePointer).trim();
	    if (word.length() > 0) {
		description.append(word);
		description.append(' ');
	    }
	}
	descriptionPointer = codePointer;
    }

    private void adjustEndPoint() {
	boolean inLine = true;

	while (codePointer < codeLength) {
	    char c = code.charAt(codePointer);
	    if (inComment) {
		if (code.startsWith("*/", codePointer)) {
		    codePointer += 2;
		    inComment = false;
		} else {
		    codePointer++;
		}
	    } else if (inLine && code.startsWith("//", codePointer)) {
		// skip to end of line
		codePointer = code.indexOf('\n', codePointer);
		if (codePointer < 0) {
		    codePointer = codeLength;
		    beginningOfLinePointer = codeLength;
		}
	    } else if (inLine && code.startsWith("/*", codePointer)) {
		codePointer += 2;
		inComment = true;
	    } else if (c <= ' ' || c == ';') {
		codePointer++;
	    } else {
		if (inLine) break;
		descriptionPointer = codePointer;
		endPoint = beginningOfLinePointer;
		endLine  = lineNumber - 1;
		return;
	    }
	    if (c == '\n') {
		inLine = false;
		beginningOfLinePointer = codePointer;
		lineNumber++;
	    }
	}
	descriptionPointer = codePointer;
	endPoint = codePointer;
	endLine = lineNumber;
	return;
    }

    /**
     * return currrent description
     */
    public String description() {
	for (int i = description.length(); --i >= 0; ) {
	    if (description.charAt(i) < ' ') description.setCharAt(i, ' ');
	}
	return description.toString();
    }

    public int startPoint() {
	return startPoint;
    }

    public int endPoint() {
	return endPoint;
    }

    public int startLine() {
	return startLine;
    }

    public int endLine() {
	return endLine;
    }

    public String toString() {
	return "nest=" + nest +
		",point=(" + startPoint + "," + endPoint + ")" +
		",line=(" + startLine + "," + endLine + ")" +
		",description=\"" + description() + "\"";
    }
}
