/*
 * Copyright (c) 1995 PFU Limited.
 *	author Kazuhisa Shirakami
 */
package dejava.lang;

import java.util.StringTokenizer;

public class MethodSpec extends AccessFlags {
    private boolean isConstructor = false;
    private String returnType = null;
    private String methodName = null;
    private String arguments  = null;
    private String dimensions = "";

    public static MethodSpec fromString(String description, String className) {
	try {
	    return new MethodSpec(description, className);
	} catch (DescriptionException e) {
	    return null;
	}
    }

    public MethodSpec(String methodName) {
	super();
	this.methodName = methodName;
    }

    public MethodSpec(String description, String className) throws DescriptionException {
	super();
	String errorMessage = setSpec(description, className);
	if (errorMessage != null) {
	    throw new DescriptionException(errorMessage);
	}
    }

    private String setSpec(String description, String className) {
	if (description.indexOf('=') >= 0) {
	    return "description syntax error";
	}
	int len = description.length();
	int arg = description.indexOf('(');
	int arg_end = description.lastIndexOf(')');
	if (arg < 0 || arg_end < 0 || arg >= arg_end) {
	    return "arguments required";
	}
	// set signatures, type and method name
	StringTokenizer t = new StringTokenizer(description.substring(0, arg));
	String token = null;
	while (t.hasMoreTokens() && enableAccessFlag(token = t.nextToken())) {
	}
	if (token == null) {
	    return "return type or constractor name required";
	}
	if (t.hasMoreElements()) {
	    returnType = token; 
	    methodName = (String)t.nextElement();
	    isConstructor = false;
	} else {
	    if (!token.equals(className)) {
		return "return type required";
	    }
	    returnType = null;
	    methodName = token;
	    isConstructor = true;
	}
	int dot = methodName.lastIndexOf('.');
	if (dot >= 0) methodName = methodName.substring(dot + 1);
	int illegal = JavaLexical.isIdentifier(methodName);
	if (illegal >=  0) return "illegal character '" +
				  methodName.charAt(illegal) +
				  "' for method name";
	if (t.hasMoreElements()) {
	    return "';' required";
	}

	// set arguments
	arg++;
	String args = description.substring(arg, arg_end).trim();
	arg = 0;
	int arglen = args.length();
	while (arg < arglen) {
	    int space, delimiter, arrray, array_end;
	    char c;
	    // search end of word
	    if ((space = args.indexOf(' ', arg)) < 0) {
		space = arglen;
	    }
	    if (space >= arglen) {
		space = arglen;
	    }
	    // get argument type
	    dot = args.lastIndexOf('.', space) + 1;
	    if (arg < dot) arg = dot;
	    if (arguments == null) {
		arguments = args.substring(arg, space);
	    } else {
		arguments = arguments + "," + args.substring(arg, space);
	    }
	    // ignore to next ',', it is argument name
	    if (arg >= arglen ||
		(delimiter = args.indexOf(',', arg)) < 0) {
		delimiter = arglen;
	    }
	    for ( ; arg < delimiter; arg++) {
		c = args.charAt(arg);
		if (c == '[') {
		    arguments = arguments + "[";
		} else if (c == ']') {
		    arguments = arguments + "]";
		}
	    }
	    arg = delimiter + 1;
	    // skip spacees
	    for ( ; arg < arglen; arg++) {
		c = args.charAt(arg);
		if (c != ' ') {
		    break;
		}
	    }
	}
	if (arguments == null) arguments = "";

	if (returnType != null) {
	    int bracket = returnType.indexOf('[');
	    if (bracket == 0) return "return type required";
	    if (bracket > 0) {
		String rt = returnType;
		returnType = returnType.substring(0, bracket);
		do {
		    dimensions += "[]";
		} while ((bracket = rt.indexOf('[', bracket + 1)) > 0);
	    }
	}
	if (arg_end < len) {
	    while ((arg_end = description.indexOf('[', arg_end) + 1) > 0) {
		dimensions += "[]";
	    }
	}
	return null;
    }

    public void rename(String newName) {
	methodName = newName;
    }

    public String arguments() {
	return arguments;
    }

    public String returnType() {
	return returnType;
    }

    public String methodName() {
	return methodName;
    }

    public String identityName() {
	if (arguments == null) {
	    return methodName;
	} else {
	    return methodName + "(" + arguments + ")";
	}
    }

    public String prototype() {
	return (returnType == null ? "" : (returnType + " ")) +
		identityName() +
		dimensions;
    }

    public String description() {
	return signaturesString() + prototype();
    }

    public boolean inhibitedBy(AccessFlags hideFlags) {
	int me = signatures();
	if (isConstructor) me |= STATIC;
	return hideFlags.inhibit(me);
    }

    public boolean isConstructor() {
	return isConstructor;
    }

    public boolean isPublic() {
	return isEnableAccessFlags(PUBLIC);
    }

    public boolean isAbstract() {
	return isEnableAccessFlags(ABSTRACT);
    }

    public boolean ident(MethodSpec target) {
	return identityName().equals(target.identityName());
    }

    public boolean sameReturnType(MethodSpec target) {
	if (returnType == null) {
	    return target.returnType() == null;
	} else if (dimensions != null && !dimensions.equals(target.dimensions) ||
		   target.dimensions != null && !target.dimensions.equals(dimensions)) {
	    return false;
	} else {
	    return returnType.equals(target.returnType());
	}
    }

    public void dispose() {
	returnType = null;
	methodName = null;
	arguments = null;
	dimensions = null;
    }
}
