/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import dejava.lang.*;
import dejava.util.*;
import java.io.*;
import java.util.*;

public final class ClassManager extends Manager {
    ClassSpec spec;

    /**
     * holds Vector of SourceCodeRange or StringSourceCode or Changes)
     */
    Object code = null;

    /**
     * holds SourceCodeRange for this class
     */
    SourceCodeRange range = null;

    /**
     * holds super class
     */
    ClassManager superClasz = null;

    /**
     * id number for execute
     */
    private static int idNum = 0;

    public static boolean canExecute = false;
    private static String java = null;

    static {
	String bin = System.getProperty("java.home") +
		     File.separator + "bin" + File.separator;
	if (SystemManager.onWin32) {
	    java = bin + "java.exe";
	} else {
	    java = bin + "java";
	}
	if ((new File(java)).exists()) {
	    canExecute = true;
	}
    }

    /**
     * read a class description via CodeReader
     * @return ClassManager for the description
     */
    synchronized
    static ClassManager readFrom(SourceFileManager parent, CodeReader reader) {
	try {
	    Vector ranges = new Vector();
	    File file = parent.file();
	    ClassSpec spec = new ClassSpec(reader.description());
	    ClassManager cman = new ClassManager(parent, spec);
	    cman.addChild(new HeaderManager(cman));
	    int firstStartLine = reader.startLine();
	    int startPoint = reader.startPoint();
	    int startLine  = reader.startLine();
	    int endPoint = startPoint;
	    int endLine = startLine -1;
	    int nest;
	    while ((nest = reader.nextDescription()) >= 1) {
		endPoint = reader.startPoint();
		endLine  = reader.startLine() - 1;
		MethodManager mman = MethodManager.readFrom(cman, reader, nest);
		if (mman != null) {
		    cman.addChild(mman);
		    if (startPoint != -1) {
			ranges.addElement(
			    new SourceCodeRange(startPoint, endPoint,
						startLine, endLine));
		    }
		    startPoint = -1;
		    startLine = -1;
		} else if (startPoint == -1) {
		    startPoint = endPoint;
		    startLine = endLine + 1;
		}
	    }
	    if (startPoint == -1) {
		startPoint = reader.startPoint();
		startLine = reader.startLine();
	    }
	    endPoint = reader.endPoint();
	    endLine  = reader.endLine();
	    ranges.addElement(new SourceCodeRange(startPoint, endPoint,
						  startLine, endLine));
	    cman.code = ranges;
	    cman.range = new SourceCodeRange(-1, -1, firstStartLine, endLine);
	    return cman;
	} catch (DescriptionException e) {
	    SystemManager.exception(parent.absolutePath() + ": " + reader.startLine(), e);
	    return null;
	}
    }

    public ClassManager(Manager parent, String name) {
	super(parent, name);
	spec = null;
	code  = null;
    }

    ClassManager(Manager parent, ClassSpec spec) {
	super(parent, spec.className());
	this.spec = spec;
    }

    ProjectManager projectManager() {
	return (ProjectManager)parent().parent().parent();
    }

    PackageManager packageManager() {
	return (PackageManager)parent().parent();
    }

    /**
     * returns super class for me
     * resolve package name using header information
     */
    public ClassManager superClass() throws DejavaException {
	readSource();
	String sc = spec.superClassName();
	if (sc != null) {
	    PackageManager pm = packageManager();
	    int index = sc.lastIndexOf('.');
	    if (index > 0) {	// class name contains package name
		String pkgName = sc.substring(0, index);
		pm = PackageManager.findPackage(pkgName);
	        if (pm == null) {
		    SystemManager.information("Unknown package name: " + pkgName);
		    // throw new DejavaException("Unknown package name: " + pkgName);
		    return null;
		} else {
		    sc = sc.substring(index + 1);
		}
	    }
	    if ((superClasz = pm.findClass(sc)) == null) {
		if (index > 0) { // package doesn't contain such class
		    SystemManager.information("Super class " + sc + " for " + name + " not found.");
		} else {
		    Enumeration m = header().imported().elements();
		    while (m.hasMoreElements()) {
			Manager man = (Manager)m.nextElement();
			if (man instanceof ClassManager) {
			    if (man.name().equals(sc)) {
				superClasz = (ClassManager)man;
				break;
			    }
			} else {
			    if ((superClasz = ((PackageManager)man).findClass(sc)) != null) {
				break;
			    }
			}
		    }
		}
	    }
	}
	return superClasz;
    }

    /**
     * get HeaderManager for this class
     *  -> the first child is it
     */
    HeaderManager header() {
	if (child == null) {
	    addChild(new HeaderManager(this));
	}
	return (HeaderManager)child;
    }

    /**
     * get hierarchy list of this class
     */
    public Enumeration hierarchy(String methodName) {
	return new HierarchyEnumerator(this, methodName);
    }

    /**
     * get binary file for this class
     */
    File binaryFile(File directory) {
	return new File(directory, name + ".class");
    }

    /**
     * get class CodePath
     */
    public CodePath getCodePath() {
	if (parent != null) {
	    CodePath cp = parent.getCodePath();
	    cp.classManager(this);
	    return cp;
	} else {
	    return new CodePath(null, null, null, this, null);
	}
    }

    /**
     * whether this class is editable or not
     */
    public boolean isEditable() {
	if (parent != null) {
	    return parent.isEditable();
	} else {
	    return false;
	}
    }

    /**
     * whether this class is updated after last compilation
     */
    boolean isUpdatedAfterLastCompilation(long sourceFileLastModified, File directory) {
	if (isChanged()) return true;
	File binaryFile = binaryFile(directory);
	return !binaryFile.exists() ||
	    (binaryFile.lastModified() < sourceFileLastModified);
    }

    /**
     * set new code for this class
     */
    public Manager setCode(String newCode)
    throws DejavaException {
	String description = "";
	ClassSpec newSpec;
	CodeReader reader = new CodeReader(newCode);
	int nest = reader.nextDescription();
	if (nest <= 0) {	// -1:EOF 0:abstruct..? 1:class description
	    return tryForMethod(newCode);
	}
	description = reader.description();
	if (nest > 0) reader.nextDescription(false);
	if (reader.nextDescription() >= 0) {
	    throw new DescriptionException("too many description or not enough open brace");
	}
	try {
	    newSpec = new ClassSpec(description);
	} catch (DescriptionException ce) {
	    String errorMessage = ce.getMessage();
	    if (errorMessage.equals(ClassSpec.NO_CLASS_OR_INTERFACE) ||
		description.indexOf('(') >= 0 ||
		description.indexOf(')') >= 0) {
		return tryForMethod(newCode);
	    }
	    throw new DescriptionException(errorMessage);
	}
	if (!newSpec.className().equals(spec.className())) {
	    PackageManager pman = packageManager();
	    return pman.createClass(newSpec, newCode);
	}
	Manager ret = changed(newCode);
	if (!resolveDependancy(newSpec)) return null;
	spec = newSpec;
	return ret;
    }

    /**
     * check for method description and proceed if it is
     */
    private Manager tryForMethod(String newCode)
    throws DejavaException {
	return child.setCode(newCode);
    }

    /**
     * resolve dependancy of ClassSpec
     */
    boolean resolveDependancy() {
	return resolveDependancy(spec);
    }

    /**
     * resolve dependancy of ClassSpec
     */
    private boolean resolveDependancy(ClassSpec newSpec) {
	String className = newSpec.superClassName();
	Vector imported = header().imported();
	Vector imports = new Vector();
	if (className != null) {
	    if (newSpec.isInterface()) {
		StringTokenizer tokenizer =
		    new StringTokenizer(className + " ", ", \t\n");
		while (tokenizer.hasMoreTokens()) {
		    if (!resolveDependancy(tokenizer.nextToken(), imports, imported)) {
		       return false;
		    }
		}
	    } else {
		if (!resolveDependancy(className, imports, imported)) {
		    return false;
		}
	    }
	    if (imports.size() > 0) {
		superClasz = (ClassManager)imports.firstElement();
	    } else {
		superClasz = null;
	    }
	}
	String interfaceNames = newSpec.interfaceNames();
	if (interfaceNames != null) {
	    StringTokenizer tokenizer =
		new StringTokenizer(interfaceNames, ", \t\n");
	    while (tokenizer.hasMoreTokens()) {
		if (!resolveDependancy(tokenizer.nextToken(), imports, imported)) {
		   return false;
		}
	    }
	}
	header().resolveImports(imports, imported);
	return true;
    }

    /**
     * resolve class whom I depend on
     */
    private boolean resolveDependancy(String className, Vector imports, Vector imported) {
	ClassManager cm = null;
	boolean shouldImportClass = false;
	PackageManager pman = packageManager();
	int indexOfDot = className.lastIndexOf('.');
	if (indexOfDot != -1) {
	    // className has package part
	    String pkgName = className.substring(0, indexOfDot);
	    pman = PackageManager.findPackage(pkgName);
	    if (pman == null) {
		SystemManager.information("Package " + pkgName + " not found.\n" +
					  "Could not resolve dependancy for: " + className);
	    } else if ((cm = pman.findClass(className.substring(indexOfDot + 1))) == null) {
		SystemManager.information("Class " + className + " not found.\n" +
					  "Could not resolve dependancy for: " + className);
	    }
	} else {
	    Enumeration p = PackageManager.allPackages();
	    PackageManager pm = pman;
	    while (true) {
		cm = pm.findClass(className);
		if (cm != null) {
		    if (pm == pman || pm.name().equals("java.lang") ||
			imported.contains(cm) || imported.contains(pm)) {
		        break;
		    } else if (SystemManager.confirm(
					"Class " + className +
					" found in package " + pm.name() + ".\n" +
					"Import it?",
				 	"Yes", "No")) {
			shouldImportClass = true;
			break;
		    }
		    cm = null;
		}
		if (p.hasMoreElements()) {
		    pm = (PackageManager)p.nextElement();
		} else {
		    break;
		}
	    }
	}

	if (cm == null) {
	    return SystemManager.confirm(
			    "Class " + className + " has not been imported yet.\n" +
			    "Proceed it?", "Proceed", "Abort");
	} else {
	    if (!spec.isInterface()) resolveMethodsFrom(cm);
	    if (shouldImportClass) imports.addElement(cm);
	    return true;
	}
    }

    /**
     * resolve methods derived my dependancy
     */
    private void resolveMethodsFrom(ClassManager depend) {
	try {
	    boolean isInterface = depend.spec().isInterface();
	    Enumeration m = depend.methods();
	    while (m.hasMoreElements()) {
		MethodManager mm = (MethodManager)m.nextElement();
		MethodSpec mspec = mm.spec();
		if (isInterface || mspec.isAbstract()) {
		    if (findMethod(mspec.identityName()) == null) {
			CodeReader reader = new CodeReader(mm.getCode());
			reader.nextDescription();
			String desc = reader.description();
			if (mspec.isAbstract()) {
			    int head = desc.indexOf(AccessFlags.STR_ABSTRACT);
			    desc = desc.substring(0, head) +
				   desc.substring(head + AccessFlags.STR_ABSTRACT.length() + 1);
			}
			if (isInterface && !mspec.isPublic()) {
			    desc = AccessFlags.STR_PUBLIC + " " + desc;
			}

			MethodSpec newSpec = new MethodSpec(desc, name());
			MethodManager newMm = new MethodManager(this, newSpec);
			addChild(newMm);
			StringBuffer buf = new StringBuffer();
			buf.append("    /**\n");
			buf.append("     * " + depend.fullClassName() + " stuff\n");
			buf.append("     */\n");
			buf.append("    ");
			buf.append(desc);
			buf.append(" {\n");
			buf.append("    }\n");
			newMm.created(buf.toString());
		    }
		}
	    }
	} catch (DejavaException e) {
	}
    }

    /**
     * get class source code
     */
    protected String getCodeInternal() throws DejavaException {
	readSource();
	if (parent == null)
	    throw new DejavaNotFoundException("class " + name() + " seems to have been removed");
	File file = file();
	if (code instanceof Vector) {
	    StringBuffer buf = new StringBuffer();
	    for (Enumeration r = ((Vector)code).elements(); r.hasMoreElements(); ) {
		SourceCodeRange aRange = (SourceCodeRange)r.nextElement();
		buf.append(aRange.getCode(file));
	    }
	    String codeString = buf.toString();
	    int lineFeed = codeString.lastIndexOf('}') - 1;
	    if (lineFeed > 0 &&
		codeString.charAt(lineFeed) == '\n' &&
		codeString.charAt(lineFeed - 1) == '\n') {
		codeString = codeString.substring(0, lineFeed).trim() +
			     codeString.substring(lineFeed);
	    }
	    code = new StringSourceCode(codeString);
	}
	return ((SourceCodeRange)code).getCode(file);
    }

    /**
     * sync sources for the class
     */
    public void sync() {
	if (parent != null) {
	    parent.sync();
	}
    }

    /**
     * read source file
     */
    public void readSource() throws DejavaException {
	if (parent != null) {
	    ((SourceFileManager)parent).readSource();
	}
	if (spec == null) {
	    throw new DejavaNotFoundException("Class not found in the source file");
	}
    }

    /**
     * copy instance variables from another ClassManager
     */
    void copyFrom(ClassManager cman, boolean force, boolean verbose) throws DejavaException {
	Vector order = new Vector();
	Hashtable hash = new Hashtable();
	Enumeration m = cman.methods();
	while (m.hasMoreElements()) {
	    MethodManager cman_mm = (MethodManager)m.nextElement();
	    MethodManager mm = findMethod(cman_mm.name());
	    if (mm != null) {
		mm.copyFrom(cman_mm, force);
		order.addElement(mm);
		hash.put(mm.name(), mm);
	    } else {
		if (verbose)
		     SystemManager.messageln("    + " + cman_mm.name());
		cman_mm.parent = this;
		order.addElement(cman_mm);
		hash.put(cman_mm.name(), cman_mm);
	    }
	}

	m = children();
	String pname = null;
	while (m.hasMoreElements()) {
	    MethodManager mm = (MethodManager)m.nextElement();
	    if (!hash.containsKey(mm.name())) {
		if (mm.isChanged()) {
		    if (pname != null) {
			MethodManager prevm = (MethodManager)hash.get(pname);
			int index = order.indexOf(prevm);
			order.insertElementAt(mm, index + 1);
		    } else {
			// not reached
			SystemManager.exception(new DejavaException("dejava internal error"));
		    }
		    if (verbose) SystemManager.messageln("    ! " + mm.name());
		} else {
		    mm.parent = null;
		    if (verbose) SystemManager.messageln("    - " + mm.name());
		}
	    }
	    pname = mm.name();
	}

	m = order.elements();
	child = (Manager)m.nextElement();
	Manager prev = child;
	while (m.hasMoreElements()) {
	    prev.brother = (Manager)m.nextElement();
	    prev = prev.brother;
	}
	prev.brother = null;
	spec = cman.spec;
	code = cman.code;
	if (force || !(range instanceof Changes)) {
	    range = cman.range;
	}
    }

    /**
     * source file
     */
    public File file() {
	if (parent != null) {
	    return ((SourceFileManager)parent).file();
	} else {
	    return null;
	}
    }

    public Enumeration methods() throws DejavaException {
	readSource();
	return children();
    }

    public Enumeration methodNames() throws DejavaException {
	readSource();
	return childNames();
    }

    public Enumeration methodPrototypes(AccessFlags hideMethods)
    throws DejavaException {
	readSource();
	return new MethodPrototypeEnumerator(child, hideMethods);
    }

    public int numMethods() throws DejavaException {
	readSource();
	return numChildren();
    }

    public MethodManager findMethod(String name)
    throws DejavaClassNotFoundException {
	return (MethodManager)findChild(name);
    }

    public MethodManager findMethodByPrototype(String prototype)
    throws DejavaException {
	readSource();
	Enumeration m = methods();
	while (m.hasMoreElements()) {
	    MethodManager mm = (MethodManager)m.nextElement();
	    if (mm.spec().prototype().equals(prototype)) {
		return mm;
	    }
	}
	return null;
    }

    public void changeMethodsOrder(Enumeration methods) {
	if (methods.hasMoreElements()) {
	    try {
		setCode(getCodeInternal());
		child = (Manager)methods.nextElement();
		Manager prev = child;
		while (methods.hasMoreElements()) {
		    prev.brother = (Manager)methods.nextElement();
		    prev = prev.brother;
		}
		prev.brother = null;
	    } catch (DejavaException e) {
	    }
	}
    }

    public ClassSpec spec() throws DejavaException {
	readSource();
	return spec;
    }

    public void changed(Changes chg) {
	/*
	if (code instanceof Vector) {
	    chg.setPrevious(null);
	} else {
	    chg.setPrevious((SourceCodeRange)code);
	}
	*/
	code = chg;
	changed();
    }

    public boolean isChanged() {
	/*
	if (code == null) return false;
	if (code instanceof Vector) return false;
	 */
	return parent().isChanged();
    }

    public void changed() {
	parent.changed();
    }

    /**
     * write source code on PrintStream
     */
    public void writeOn(PrintStream ps, boolean synced) throws java.io.IOException {
	String codeString = getCode();
	int index = codeString.lastIndexOf('}');
	if (parent().child != this) ps.println("");
	ps.println(codeString.substring(0, index).trim());
	for (Enumeration m = child.brothers(); m.hasMoreElements(); ) {
	    ((MethodManager)m.nextElement()).writeOn(ps, synced);
	}
	ps.println(codeString.substring(index).trim());
	if (synced) {
	    ((SourceCodeRange)this.code).synced();
	}
    }

    /**
     * return offset from beggining of this class
     */
    public int lineOffset(int line) throws DejavaException {
	readSource();
	if (line <= range.endLine()) {
	    return line - (int)range.startLine();
	} else {
	    return line + 1;
	}
    }

    String fullClassName() {
	PackageManager pm = packageManager();
	if (pm instanceof TopDirectoryPackageManager) {
	    return name();
	} else {
	    return pm.name() + "." + name();
	}
    }

    public boolean hasMainMethod() {
	try {
	    if (findMethodByPrototype("void main(String[])") != null) {
		return true;
	    }
	} catch (DejavaException e) {
	}
	return false;
    }

    public boolean isApplet() {
	try {
	    String superClassName = spec().superClassName();
	    if (superClassName != null &&
		(superClassName.equals("Applet") ||
		 superClassName.equals("applet.Applet") ||
		 superClassName.equals("java.applet.Applet"))) {
		 return true;
	    }
	} catch (DejavaException e) {
	}
	return false;
    }

    public boolean isInterface() {
	try {
	    if (spec().isInterface()) {
		return true;
	    }
	} catch (DejavaException e) {
	}
	return false;
    }

    /**
     * execute the class
     */
    public void execute(String args) {
	if (args == null) args = "";
	String name = fullClassName();
	File f = new File(((SourceFileManager)parent).absolutePath());
	String classpath = f.getParent();
	File classFile = new File(classpath + File.separator + name() + ".class");
	if (!classFile.exists()) {
	    SystemManager.information("no class file\nneed to be compiled before execute");
	    return;
	}
	if (f.lastModified() > classFile.lastModified()) {
	    if (SystemManager.confirm("source file is newer than compiled class file\n" +
				      "should compile before execute",
				      "Abort to execute", "Execute")) {
		return;
	    }
	}
	if (header().packageName() != null) {
	    classpath = projectManager().absolutePath();
	} else {
	    name = this.name;
	}
	classpath = classpath + SystemManager.PATHSEPARATOR +
				SystemManager.CLASSPATH;
	try {
	    if (hasMainMethod()) {
		// execute main method
		if (java == null) {
		    SystemManager.information("No interpritor found. Abort.");
		    return;
		}
		String cmd;
		cmd = java + " -classpath " + classpath + " " + name + " " + args;
		(new Execute(name, cmd)).start();
		return;
	    }
	    // execute applet
	    ClassSpec spec;
	    try {
		spec = spec();
	    } catch (DejavaException e) {
		spec = null;
	    }
	    if (spec == null) {
		SystemManager.information("Class " + name + " has no spec", "Abort");
		return;
	    }
	    String superClassName = spec.superClassName();
	    if (isApplet()) {
		File tmpdir = new File(SystemManager.profileDirectory(), ".tmp");
		if (!tmpdir.exists()) {
		    if (tmpdir.mkdir()) {
			SystemManager.information("Created temporaly directory for applets in profile directory.");
		    } else {
			SystemManager.information("Could not create temporaly directory for applets in profile directory.\nAborted");
			return;
		    }
		}
		String fname = tmpdir.getPath() + File.separator +
			       name + "." + idNum++ + ".html";
		PrintStream html = new PrintStream(new FileOutputStream(fname));
		html.print("<APPLET CODE=\"");
		html.print(name);
		html.print("\" ");
		if (args.indexOf("width=")  < 0) html.print("width=100 ");
		if (args.indexOf("height=") < 0) html.print("height=100 ");
		html.print(args);
		html.print(">\n</APPLET>\n");
		html.close();
		String cmd;
		cmd = java + " -classpath " + classpath +
			     " sun.applet.AppletViewer file:///" + fname;
		(new Execute(name, cmd)).start();
	    } else {
		SystemManager.information(name + " is not executable.", "Abort");
	    }
	} catch (IOException e) {
	    SystemManager.information("Failed to execute " + name, "Abort");
	}
    }

    /**
     * execute the class
     */
    public void executeWithHTML(String html) {
	if (html == null) html = "";
	String name = fullClassName();
	File f = new File(((SourceFileManager)parent).absolutePath());
	String classpath = f.getParent();
	File classFile = new File(classpath + File.separator + name() + ".class");
	if (!classFile.exists()) {
	    SystemManager.information("no class file\nneed to compile before execute");
	    return;
	}
	if (f.lastModified() > classFile.lastModified()) {
	    if (SystemManager.confirm("source file is newer than compiled class file\n" +
				      "should compile before execute",
				      "Abort to execute", "Execute")) {
		return;
	    }
	}
	if (header().packageName() != null) {
	    classpath = projectManager().absolutePath();
	} else {
	    name = this.name;
	}
	classpath = classpath + SystemManager.PATHSEPARATOR +
				SystemManager.CLASSPATH;
	ClassSpec spec;
	try {
	    spec = spec();
	} catch (DejavaException e) {
	    spec = null;
	}
	if (spec == null) {
	    SystemManager.information("Class " + name + " has no spec", "Abort");
	    return;
	}
	String superClassName = spec.superClassName();
	if (isApplet()) {
	    String cmd = java + " -classpath " + classpath +
			 " sun.applet.AppletViewer file:///" + html;
	    (new Execute(name, cmd)).start();
	} else {
	    SystemManager.information(name + " is not applet.", "Abort");
	}
    }

    /**
     * rename the class
     */
    public Manager rename(String newName)
    throws DejavaException {
	if (packageManager().findClass(newName) != null) {
	    SystemManager.information("Class " + newName + " already exists. Aborted.");
	    return null;
	}
	if (spec.isPublic()) try {
	    parent.rename(newName + ".java");
	} catch (DejavaException e) {
	    SystemManager.information(e.getMessage() + ". Aborted");
	    return null;
	}
	Enumeration m = methods();
	while (m.hasMoreElements()) {
	    MethodManager mm = (MethodManager)m.nextElement();
	    if (mm.spec().isConstructor()) {
		mm.rename(newName);
	    }
	}
	String code = getCode();
	StringBuffer newCode = new StringBuffer();
	CodeReader reader = new CodeReader(code, 1, false);
	reader.nextDescription();
	int index = code.indexOf(ClassSpec.STR_CLASS, reader.startPoint());
	if (index != -1) {
	    index += ClassSpec.STR_CLASS.length();
	} else {
	    index = code.indexOf(ClassSpec.STR_INTERFACE, reader.startPoint()) +
			ClassSpec.STR_INTERFACE.length();
	}
	index = code.indexOf(name, index);
	newCode.append(code.substring(0, index));
	newCode.append(newName);
	newCode.append(code.substring(index + name.length()));
	spec.rename(newName);
	name = newName;
	return renamed(newCode.toString());
    }

    /**
     * dispose all information
     */
    protected void dispose() {
	//disposeDetails();
	super.dispose();
    }

    /**
     * dispose class specific data
     */
    protected void disposeDetails() {
	if (spec != null) {
	    spec.dispose();
	    spec = null;
	}
	if (code instanceof Vector) {
	    Enumeration e = ((Vector)code).elements();
	    while (e.hasMoreElements()) {
		SourceCodeRange c = (SourceCodeRange)e.nextElement();
		c.dispose();
	    }
	    code = null;
	} else if (code instanceof SourceCodeRange) {
	    ((SourceCodeRange)code).dispose();
	    code = null;
	}
	if (range != null) {
	    range.dispose();
	    range = null;
	}
	if (child != null && child.brother != null) {
	    ((MethodManager)child.brother).disposeDetails();
	}
	if (brother != null) {
	    ((ClassManager)brother).disposeDetails();
	}
    }
}
