/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh, Kazuhisa Shirakami
 */
package dejava.sys;

import java.lang.*;
import java.io.*;
import java.util.*;

/**
 * String replesentation for Code's location
 *
 * notation:
 *     //<host>/<project>/<package>.<class>#<method>
 *
 */
public class CodePath {
    private String		hostManager;	// not implemented
    private ProjectManager	projectManager;
    private PackageManager	packageManager;
    private SourceFileManager	sourceFileManager;
    private ClassManager	classManager;
    private MethodManager	methodManager;

    public static final int HOST    = 0;
    public static final int PROJECT = 1;
    public static final int PACKAGE = 2;
    public static final int CLASS   = 3;
    public static final int METHOD  = 4;
    public static final int NAMES   = 5;

    public static final CodePath nullCodePath =
				 new CodePath(null, null, null, null, null);

    /**
     * construct from each item
     */
    public CodePath(String hst,
		    ProjectManager prj,
		    PackageManager pkg,
		    ClassManager cls,
		    MethodManager mtd) {
	hostManager       = hst;
	projectManager    = prj;
	packageManager    = pkg;
	sourceFileManager = (cls == null) ? null : (SourceFileManager)cls.parent();
	classManager      = cls;
	methodManager     = mtd;
    }

    /**
     * construct from Manager
     */
    public CodePath(Manager manager) {
	methodManager     = null;
	classManager      = null;
	sourceFileManager = null;
	packageManager    = null;
	projectManager    = null;
	if (manager instanceof MethodManager) {
	    methodManager = (MethodManager)manager;
	} else if (manager instanceof ClassManager) {
	    classManager = (ClassManager)manager;
	} else if (manager instanceof SourceFileManager) {
	    sourceFileManager = (SourceFileManager)manager;
	} else if (manager instanceof PackageManager) {
	    packageManager = (PackageManager)manager;
	} else if (manager instanceof ProjectManager) {
	    projectManager = (ProjectManager)manager;
	}
	if (methodManager != null) {
	    classManager = (ClassManager)methodManager.parent();
	}
	if (classManager != null) {
	    sourceFileManager = (SourceFileManager)classManager.parent();
	}
	if (sourceFileManager != null) {
	    packageManager = (PackageManager)sourceFileManager.parent();
	}
	if (packageManager != null) {
	    projectManager = (ProjectManager)packageManager.parent();
	}
	if (projectManager != null) {
	    Manager m = (Manager)projectManager.parent();
	    hostManager = (m != null) ? m.name() : null;
	}
    }

    /**
     * construct from another CodePath
     */
    public CodePath(CodePath cp) {
	hostManager       = cp.hostManager;
	projectManager    = cp.projectManager;
	packageManager    = cp.packageManager;
	sourceFileManager = cp.sourceFileManager;
	classManager      = cp.classManager;
	methodManager     = cp.methodManager;
    }

    /**
     * construct from String replesentation of CodePath
     */
    public CodePath(String path) {
	this(pathToNames(path));
    }

    /**
     * construct from String replesentation of CodePath
     */
    public CodePath(String names[]) {
	hostManager = names[HOST];

	if (names[PROJECT] != null) {
	    projectManager = SystemManager.findProject(names[PROJECT]);
	} else {
	    projectManager = null;
	}

	if (projectManager != null && names[PACKAGE] != null) {
	    packageManager = projectManager.findPackage(names[PACKAGE]);
	} else {
	    packageManager = null;
	}

	if (packageManager != null && names[CLASS] != null) {
	    classManager = packageManager.findClass(names[CLASS]);
	} else {
	    classManager = null;
	}

	if (classManager != null) {
	    sourceFileManager = (SourceFileManager)classManager.parent();
	} else {
	    sourceFileManager = null;
	}

	if (classManager != null && names[METHOD] != null) {
	    try {
		methodManager = classManager.findMethodByPrototype(names[METHOD]);
		if (methodManager == null) {
		    methodManager = classManager.findMethod(names[METHOD]);
		}
	    } catch (DejavaException e) {
		methodManager = null;
	    }
	} else {
	    methodManager = null;
	}
    }

    /**
     * return names
     */
    public String names()[] {
	String names[] = new String[NAMES];
	names[HOST] = hostManager;
	names[PROJECT] = projectManager != null ? projectManager.name() : null;
	names[PACKAGE] = packageManager != null ? packageManager.name() : null;
	names[CLASS]   =   classManager != null ?   classManager.name() : null;
	names[METHOD]  =  methodManager != null ?  methodManager.name() : null;
	return names;
    }

    /**
     * set code path from String representation
     */
    public static String pathToNames(String path)[] {
	String names[] = new String[NAMES];
	int from = 0;
	int len = path.length();

	// get host name
	int to = path.indexOf("//");
	if (to > from) {
	    from = to + 2;
	    to = path.indexOf('/', from);
	    names[HOST] = path.substring(from, to);
	    from = to + 1;
	} else {
	    names[HOST] = null;
        }

	// get project name
	to = path.lastIndexOf('/');
	if (to > from) {
	    names[PROJECT] = path.substring(from, to);
	    from = to + 1;
	} else {
	    names[PROJECT] = null;
	}

	// get package name
	to = path.lastIndexOf('.');
	if (to > from) {
	    names[PACKAGE] = path.substring(from, to);
	    from = to + 1;
	} else if (from < len) {
	    names[PACKAGE] = path.substring(from);
	    from = len;
	} else {
	    names[PACKAGE] = null;
	}

	// get class name
	to = path.indexOf('#', from);
	if (to > from) {
	    names[CLASS] = path.substring(from, to);
	    from = to + 1;
	} else if (from < len) {
	    names[CLASS] = path.substring(from);
	    from = len;
	} else {
	    names[CLASS] = null;
	}

	// get method name
	if (from < len) {
	    names[METHOD] = path.substring(from);
	} else {
	    names[METHOD] = null;
	}

	return names;
    }

    /**
     * complete CodePath string
     */
    public static CodePath complete(String names[]) {

	boolean complete = true;

	ProjectManager prj = null;
	if (names[PROJECT] != null) {
	    if ((prj = SystemManager.findProject(names[PROJECT])) == null) {
		String name = complete(names[PROJECT], SystemManager.allProjectNames());
		if (name != null && (prj = SystemManager.findProject(name)) != null) {
		    names[PROJECT] = name;
		}
	    }
	    if (prj == null) complete = false;
	}

	PackageManager pkg = null;
	if (names[PACKAGE] != null) {
	    if (prj != null && (pkg = prj.findPackage(names[PACKAGE])) == null) {
		if (names[CLASS] != null) {
		    String name = names[PACKAGE] + "." + names[CLASS];
		    if ((pkg = prj.findPackage(name)) == null) {
			name = complete(name, prj.packageNames());
			if (name != null) pkg = prj.findPackage(name);
		    }
		    if (pkg != null) {
			names[PACKAGE] = name;
			names[CLASS] = null;
		    }
		}
		if (pkg == null) {
		    String name = complete(names[PACKAGE], prj.packageNames());
		    if (name != null && (pkg = prj.findPackage(name)) != null) {
			names[PACKAGE] = name;
		    }
		}
	    }
	    if (pkg == null) complete = false;
	}
	
	ClassManager cls = null;
	if (names[CLASS] != null) {
	    if (pkg != null && (cls = pkg.findClass(names[CLASS])) == null) {
		String name = complete(names[CLASS], pkg.classNames());
		if (name != null && (cls = pkg.findClass(name)) != null) {
		    names[CLASS] = name;
		}
	    }
	    if (cls == null) {
		String name = complete(names[PACKAGE]+"."+names[CLASS], prj.packageNames());
		if (name != null && (pkg = prj.findPackage(name)) != null) {
		    names[PACKAGE] = name;
		    names[CLASS] = null;
		} else {
		    complete = false;
		}
	    }
	}

	MethodManager mtd = null;
	if (names[METHOD] != null) {
	    try {
		if (cls != null && (mtd = cls.findMethodByPrototype(names[METHOD])) == null) {
		    String name = complete(names[METHOD], cls.methodNames());
		    if (name != null && (mtd = cls.findMethod(name)) != null) {
			names[METHOD] = name;
		    }
		}
	    } catch (DejavaException e) {
	    }
	    if (mtd == null) complete = false;
	}	

	return complete ? (new CodePath(null, prj, pkg, cls, mtd)): null;
    }

    /**
     * complete wildcarded string
     */
    private static String complete(String str, Enumeration items) {
	boolean hasWildcard = (str.indexOf('*') != -1);
    nextItem:
	while (items.hasMoreElements()) {
	    int index = 0;
	    String item = (String)items.nextElement();
	    if (hasWildcard) {
		StringTokenizer st = new StringTokenizer(str, "*");
		while (st.hasMoreTokens()) {
		    String key = st.nextToken();
		    if (key.length() != 0 && (index = item.indexOf(key, index)) == -1) continue nextItem;
	        }
		return item;
	    } else {
		do {
		    if (item.startsWith(str, index)) return item;
		} while ((index = item.indexOf(' ', index) + 1) > 0);
	    }
	}
	return null;
    }

    public CodePath projectCodePath() {
	return new CodePath(hostManager, projectManager, null, null, null);
    }

    public CodePath packageCodePath() {
	return new CodePath(hostManager,
			    projectManager, packageManager, null, null);
    }

    public CodePath classCodePath() {
	return new CodePath(hostManager,
			    projectManager, packageManager, classManager, null);
    }

    public boolean setManager(Manager manager) {
	if (manager != null) {
	    if (manager instanceof MethodManager) {
		methodManager((MethodManager)manager);
	    } else if (manager instanceof ClassManager) {
		classManager((ClassManager)manager);
	    } else if (manager instanceof PackageManager) {
		packageManager((PackageManager)manager);
	    } else if (manager instanceof ProjectManager) {
		projectManager((ProjectManager)manager);
	    } else {
		return false;
	    }
	} else {
	    if (methodManager != null) {
		methodManager((MethodManager)null);
	    } else if (classManager != null) {
		classManager((ClassManager)null);
	    } else if (packageManager != null) {
		packageManager((PackageManager)null);
	    } else if (projectManager != null) {
		projectManager((ProjectManager)null);
	    } else {
		return false;
	    }
	}
	return true;
    }

    public Manager getManager() {
	if (methodManager     != null) return methodManager;
	if (classManager      != null) return classManager;
	if (sourceFileManager != null) return sourceFileManager;
	if (packageManager    != null) return packageManager;
	if (projectManager    != null) return projectManager;
	return null;
    }

    public ProjectManager projectManager() {
	return projectManager;
    }

    public PackageManager packageManager() {
	return packageManager;
    }

    public SourceFileManager sourceFileManager() {
	return sourceFileManager;
    }

    public ClassManager classManager() {
	return classManager;
    }

    public MethodManager methodManager() {
	return methodManager;
    }

    public ProjectManager projectManager(String name) {
	return projectManager(SystemManager.findProject(name));
    }

    public ProjectManager projectManager(ProjectManager manager) {
	projectManager = manager;
	packageManager(packageManager != null ? packageManager.name() : null);
	return projectManager;
    }

    public PackageManager packageManager(String name) {
	return packageManager(projectManager != null ?
			      projectManager.findPackage(name) :
			      null);
    }

    public PackageManager packageManager(PackageManager manager) {
	packageManager = manager;
	classManager(classManager != null ? classManager.name() : null);
	return packageManager;
    }

    public ClassManager classManager(String name) {
	return classManager(packageManager != null ?
			    packageManager.findClass(name) :
			    null);
    }

    public ClassManager classManager(ClassManager manager) {
	classManager = manager;
	methodManager(methodManager != null ? methodManager.name() : null);
	if (classManager != null) {
	    sourceFileManager = (SourceFileManager)classManager.parent();
	} else {
	    sourceFileManager = null;
	}
	return classManager;
    }

    public MethodManager methodManager(String name) {
	MethodManager mm;
	if (name != null && classManager != null) {
	    try {
		mm = classManager.findMethodByPrototype(name);
		if (mm == null) {
		    mm = classManager.findMethod(name);
		}
	    } catch (DejavaException e) {
		mm = null;
	    }
	} else {
	    mm = null;
	}
	return methodManager(mm);
    }

    public MethodManager methodManager(MethodManager manager) {
	methodManager = manager;
	return methodManager;
    }

    public String projectName() {
	return (projectManager != null) ? projectManager.name() : null;
    }

    public String packageName() {
	return (packageManager != null) ? packageManager.name() : null;
    }

    public String className() {
	return (classManager != null) ? classManager.name() : null;
    }

    public String methodPrototype() {
	return (methodManager != null) ? methodManager.spec().prototype() : null;
    }

    /**
     * get the full class name for this path
     */
    public String fullClassName() {
	return ((packageManager != null) ? packageManager.name() + "." : "") +
	       ((classManager != null) ? classManager.name() : "");
    }

    /**
     * get the string replesentation of this CodePath
     */
    public String printString() {
	return (hostManager    != null ? "//" + hostManager + "/"    : "")
	     + (projectManager != null ? projectManager.name() + "/" : "")
	     + (packageManager != null ? packageManager.name() + "." : "")
	     + (classManager   != null ? classManager.name()   + "#" : "")
	     + (methodManager  != null ? methodManager.name()        : "")
	     + ((classManager  == null
	      && methodManager == null
	      && sourceFileManager != null)
			? "@" + sourceFileManager.name() : "")
	    ;
    }

    public String printShortString() {
	if (methodManager != null) {
	    return methodPrototype();
	} else if (classManager != null) {
	    return (packageManager != null ? packageManager.name() + "." : "")+
		   (classManager   != null ? classManager.name() : "");
	} else {
	    return (hostManager    != null ? "//" + hostManager + "/"    : "")+
		   (projectManager != null ? projectManager.name() + "/" : "")+
		   (packageManager != null ? packageManager.name() : "");
	}
    }

    /**
     * get the string replesentation for CodePath from names
     */
    public static String printString(String names[]) {
	return  (names[HOST]    != null ? "//" + names[HOST] + "/" : "") +
		(names[PROJECT] != null ? names[PROJECT] + "/" : "") +
		(names[PACKAGE] != null ? names[PACKAGE] + "." : "") +
		(names[CLASS]   != null ? names[CLASS] + "#" : "") +
		(names[METHOD]  != null ? names[METHOD] : "");
    }

    /**
     * returns if specified CodePath is mathes with me
     */
    public boolean matches(CodePath with, boolean exact) {
	if (with == null) {
	    return false;
	}
	if (with.projectManager != null) {
	    if (projectManager == null ||
		projectManager != with.projectManager) {
		return false;
	    }
	} else {
	    if (exact && projectManager != null) {
		return false;
	    }
	}

	if (with.packageManager != null) {
	    if (packageManager == null ||
		packageManager != with.packageManager) {
		return false;
	    }
	} else {
	    if (exact && packageManager != null) {
		return false;
	    }
	}

	if (with.sourceFileManager != null) {
	    if (sourceFileManager == null ||
		sourceFileManager != with.sourceFileManager) {
		return false;
	    }
	} else {
	    if (exact && sourceFileManager != null) {
		return false;
	    }
	}

	if (with.classManager != null) {
	    if (classManager == null ||
		classManager != with.classManager) {
		return false;
	    }
	} else {
	    if (exact && classManager != null) {
		return false;
	    }
	}

	if (with.methodManager != null) {
	    if (methodManager == null ||
		!methodManager.name().equals(with.methodManager.name())) {
		return false;
	    }
	} else {
	    if (exact && methodManager != null) {
		return false;
	    }
	}
	return true;
    }

    /**
     * returns if specified CodePath is matches with me
     */
    public boolean matches(CodePath with) {
	return matches(with, true);
    }

    public boolean hasInfluenceOn(CodePath cp) {
 	return cp != null
	  &&  projectManager != null
	  && (projectManager == cp.projectManager() || packageManager == null)
	  && (packageManager == cp.packageManager() || sourceFileManager == null)
	  && (sourceFileManager == cp.sourceFileManager || classManager == null);
    }

    /**
     * returns a String that represents the value of this Object
     */
    public String toString() {
        return super.getClass() +
	       "[hostManager=" +
	       (hostManager       == null ? "NULL" : hostManager) +
	       ",projectManager=" +
	       (projectManager    == null ? "NULL" : projectManager.name()) +
	       ",packageManager=" +
	       (packageManager    == null ? "NULL" : packageManager.name()) +
	       ",sourceManager=" +
	       (sourceFileManager == null ? "NULL" : sourceFileManager.name()) +
	       ",classManager=" +
	       (classManager      == null ? "NULL" : classManager.name()) +
	       ",methodManager=" +
	       (methodManager     == null ? "NULL" : methodManager.name()) +
	       ",path=" + printString() +
	       "]";
    }

    /**
     * test main routine
     */
    public static void main(String argv[]) {
	System.out.println((new CodePath(argv[0])).toString());
    }
}
