/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import java.io.*;
import java.util.*;
import dejava.lang.*;
import dejava.util.Changes;
import dejava.util.ChangeList;
import dejava.gui.*;

/**
 * abstract class for dejava management classes
 */
public abstract class Manager {
    /**
     * name of this manager
     * project name, file name, class name, and so on
     */
    protected String name;

    /**
     * the parent manager
     */
    protected Manager parent;

    /**
     * my younger brothers (will be listed below)
     */
    protected Manager brother;

    /**
     * my children(linked list)
     */
    protected Manager child;

    /**
     * construct from File and name
     *
     * this is protected, so only subclasses create me
     */
    protected Manager(Manager parent, String name) {
	this.parent = parent;
	this.name = name;
	brother = null;
	child = null;
    }

    /**
     * get name
     */
    public final String name(){
	return name;
    }

    /**
     * rename
     */
    public Manager rename(String newName) throws DejavaException {
	renamed(name + " -> " + newName);
	name = newName;
	return this;
    }

    /**
     * get CodePath managed by me
     */
    abstract public CodePath getCodePath();

    /**
     * whether it is editable or not
     */
    abstract public boolean isEditable();

    /**
     * set new code/comment for this manager
     */
    abstract public Manager setCode(String newCode) throws DejavaException;

    /**
     * get source code or comment
     */
    public String getCode() {
	try {
	    return getCodeInternal();
	} catch (DejavaException e) {
	    SystemManager.exception("Could not get code/comment: " +
			getCodePath().printString(), e);
	    return "/* COULD NOT GET CODE/COMMENT: " +
			getCodePath().printString() + "*/";
	} catch (java.io.IOException e) {
	    SystemManager.exception("Could not get code/comment: " +
			getCodePath().printString(), e);
	    return "/* COULD NOT GET CODE/COMMENT: " +
			getCodePath().printString() + "*/";
	}
    }

    /**
     * get source code or comment
     */
    abstract protected String getCodeInternal()
	throws java.io.IOException, DejavaException ;

    /**
     * get parent of this manager
     */
    public Manager parent() {
	return parent;
    }

    /**
     * get brother of this manager
     */
    protected Manager brother() {
	return brother;
    }

    /**
     * get Enumeration for me and brothers
     */
    Enumeration enumeration() {
	return new ManagerEnumerator(this);
    }

    /**
     * get Enumeration for brothers
     */
    Enumeration brothers() {
	return new ManagerEnumerator(brother);
    }

    /**
     * get Enumeration for children
     */
    Enumeration children() {
	return new ManagerEnumerator(child);
    }

    /**
     * get Enumeration for children
     */
    Enumeration childNames() {
	return new ManagerEnumerator(child, true);
    }

    /**
     * get number of brothers
     */
    int numBrothers() {
	int count = 0;
	for (Enumeration e = brothers(); e.hasMoreElements(); e.nextElement())
	   count++;
	return count;
    }

    /**
     * get number of children
     */
    int numChildren() {
	int count = 0;
	for (Enumeration e = children(); e.hasMoreElements(); e.nextElement())
	   count++;
	return count;
    }

    /**
     * find out named Manager
     */
    Manager find(String name) {
	for (Enumeration e = enumeration(); e.hasMoreElements(); ) {
	    Manager manager = (Manager)e.nextElement();
	    if (manager.name().equals(name)) {
		return manager;
	    }
	}
	return null;
    }

    /**
     * find out brother
     */
    protected Manager findChild(String name) {
	if (child != null) {
	    return child.find(name);
	} else {
	    return null;
	}
    }

    /**
     * add brother
     */
    public void add(Manager newBrother) {
	if (brother == null) {
	   brother = newBrother;
	} else {
	   brother.add(newBrother);
	}
    }

    /**
     * add child
     */
    public void addChild(Manager newChild) {
	if (child == null) {
	   child = newChild;
	} else {
	   child.add(newChild);
	}
    }

    /**
     * remove me
     */
    public Manager remove() {
	return parent().removeChild(this);
    }

    /**
     * remove me
     */
    protected Manager removeChild(Manager child) {
	Manager prev = null;
	Enumeration c = children();
	while (c.hasMoreElements()) {
	    Manager tmp = (Manager)c.nextElement();
	    if (tmp == child) break;
	    prev = tmp;
	}
	if (prev == null) {
	    this.child = child.brother();
	} else {
	    prev.brother = child.brother();
	}
	child.brother = null;
	if (child.removing() != null) {
	    if (prev == null) {
		child.brother = this.child;
		this.child = child;
	    } else {
		child.brother = prev.brother();
		prev.brother = child;
	    }
	    return child;
	}
	child.dispose();
	return null;
    }

    /**
     * adopt children into specified parent's family
     */
    public void adoptChildren(Manager adopter) {
	for (Enumeration c = children(); c.hasMoreElements(); ) {
	   ((Manager)c.nextElement()).parent = adopter;
	}
	adopter.addChild(child);
    }

    /**
     * sync all updated files managed by me
     */
    public void sync() {
	for (Enumeration e = children(); e.hasMoreElements(); ) {
	    Manager man = (Manager)e.nextElement();
	    man.sync();
	}
    }

    /**
     * set the new change for this Manager
     */
    public Manager changed(String newCode) {
	return setChanges(newCode, Changes.ACTION_CHANGED);
    }

    /**
     * set the new change for this Manager
     */
    public Manager created(String newCode) {
	return setChanges(newCode, Changes.ACTION_CREATED);
    }

    /**
     * I'm going to be removed
     */
    Manager removing() {
	try {
	    CodePath cp = getCodePath();
	    ChangeList list = cp.projectManager().changes();
	    list.addChanges(cp, Changes.ACTION_REMOVED, getCode());
	    changed();
	    return null;
	} catch (java.io.IOException e) {
	    SystemManager.information("Could not set changes. Change information may be lost.");
	    return this;
	}
    }

    /**
     * set the new change for this Manager
     */
    public Manager renamed(String newCode) {
	return setChanges(newCode, Changes.ACTION_RENAMED);
    }

    /**
     * set the new change for this Manager
     */
    public Manager setChanges(String newCode, String action) {
	try {
	    CodePath cp = getCodePath();
	    ChangeList list = cp.projectManager().changes();
	    Changes chg = list.addChanges(cp, action, newCode);
	    changed(chg);
	    return this;
	} catch (java.io.IOException e) {
	    SystemManager.information("Could not set changes. Change information may be lost.");
	    return null;
	}
    }

    /**
     * mark me as changed
     */
    public void changed() {
    }

    /**
     * mark me as changed
     */
    public void changed(Changes chg) {
	changed();
    }

    /**
     * returns if something's changed
     */
    public boolean isChanged() {
	return false;
    }

    /**
     * return backup file
     */
    protected File backupFile(File original) {
	return new DejavaFile(original.getPath() + ".org");
    }

    /**
     * return new file
     */
    protected File newFile(File original) {
	return new DejavaFile(original.getPath() + ".new");
    }

    /**
     * dispose all of my children
     */
    protected void disposeChildren() {
	if (child != null) {
	     child.dispose();
	     child = null;
	}
    }

    /**
     * dispose me
     */
    protected void dispose() {
	disposeChildren();
	if (brother != null) {
	    brother.dispose();
	    brother = null;
	}
	name = null;
    }
}

/**
 * the enumurator class for linked list of Manager
 */
class ManagerEnumerator implements Enumeration {
    /**
     * current element
     */
    protected Manager manager;

    /**
     * whether this enumeration returns Manager's name or Manager itself
     */
    private boolean name;

    ManagerEnumerator(Manager manager) {
	this(manager, false);
    }

    ManagerEnumerator(Manager manager, boolean name) {
	this.manager = manager;
	this.name = name;
    }

    /*
     * java.util.Enumeration stuff
     */
    public boolean hasMoreElements() {
	return manager != null;
    }

    /*
     * java.util.Enumeration stuff
     */
    public Object nextElement(){
	Manager ret = manager;
	if (ret != null) {
	    manager = ret.brother();
	}
	if (name) {
	    return ret.name();
	} else {
	    return ret;
	}
    }
}

/**
 * the enumurator class for method prototypes
 */
class MethodPrototypeEnumerator extends ManagerEnumerator {
	AccessFlags hideFlags = null;

    MethodPrototypeEnumerator(Manager manager, AccessFlags hide) {
	super(manager, false);
	hideFlags = hide;
    }

    public boolean hasMoreElements() {
	if (hideFlags != null) {
	    while (manager != null) {
		MethodSpec spec = ((MethodManager)manager).spec();
		if (spec == null) break;
		if (!spec.inhibitedBy(hideFlags)) break;
		nextElement();
	    }
	}
	return manager != null;
    }

    /*
     * java.util.Enumeration stuff
     */
    public Object nextElement(){
	return ((MethodManager)super.nextElement()).spec().prototype();
    }
}
