/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import dejava.lang.*;
import dejava.util.*;
import java.io.*;
import java.util.*;

/**
 * method management class
 * has no children
 */
public class MethodManager extends Manager {
    /**
     * specification of this method
     */
    private MethodSpec spec;

    /**
     * range in source file or changes file of my source code
     */
    private SourceCodeRange range;

    /**
     * changes for this method
     */
    private Changes changes;

    /**
     * read a method description via CodeReader
     * @return MethodManager for the description
     */
    synchronized
    static MethodManager readFrom(ClassManager parent, CodeReader reader, int nest) {
	try {
	    MethodSpec spec = MethodSpec.fromString(reader.description(), parent.name());
	    if (spec != null) {
		MethodManager mman = new MethodManager(parent, spec);
		int startPoint = reader.startPoint();
		int startLine  = reader.startLine();
		if (nest > 1) reader.nextDescription(false);
		int endPoint = reader.endPoint();
		int endLine  = reader.endLine();
		mman.range = new SourceCodeRange(startPoint, endPoint, startLine, endLine);
		return mman;
	    } else {
		if (nest > 1) reader.nextDescription(false);
		return null;
	    }
	} catch (dejava.lang.DescriptionException e) {
	    SystemManager.exception(e);
	    return null;
	}
    }

    MethodManager(Manager parent, MethodSpec spec) {
	super(parent, spec.identityName());
	this.spec = spec;
	changes = null;
    }

    public MethodSpec spec() {
	if (spec == null) try {
	    ((ClassManager)parent()).readSource();
	} catch (DejavaException e) {
	    // not reached
	}
	return spec;
    }

    public Enumeration messages() throws DescriptionException {
	if (spec == null) try {
	    ((ClassManager)parent()).readSource();
	} catch (DejavaException e) {
	    // not reached
	}
	return (new IdentifiersInCode(getCode())).identifiers();
    }

    /**
     * dejava.sys.Manager stuff
     */
    public CodePath getCodePath() {
	if (parent != null) {
	    CodePath cp = parent.getCodePath();
	    cp.methodManager(this);
	    return cp;
	} else {
	    return new CodePath(null, null, null, null, this);
	}
    }

    /**
     * get the source file manager
     */
    public SourceFileManager sourceFileManager() {
	return (SourceFileManager)parent.parent;
    }

    public void sync() {
	parent().sync();
    }

    public void changed(Changes chg) {
	//chg.setPrevious(changes);
	changes = chg;
	changed();
    }

    public void changed() {
	parent.changed();
    }

    /**
     * whether it is changed or not
     */
    public boolean isChanged() {
	return (changes != null);
    }

    /**
     * whether this method is editable or not
     */
    public boolean isEditable() {
	if (parent != null) {
	    return parent.isEditable();
	} else {
	    return false;
	}
    }

    /**
     * set new code for this method
     */
    public Manager setCode(String newCode) throws DescriptionException {
	CodeReader reader = new CodeReader(newCode);
	return setCodeInternal(reader, newCode, reader.nextDescription());
    }

    /**
     * set new code for this method
     */
    protected Manager setCodeInternal(CodeReader reader, String newCode, int nest)
    throws DescriptionException {
	if (nest < 0) {	// -1:EOF 0:abstruct 1:method description
	    throw new DescriptionException("no description");
	}
	String description = reader.description();
	if (nest > 0) reader.nextDescription(false);
	if (reader.nextDescription() >= 0) {
	    throw new DescriptionException("too many description or not enough open brace");
	}
	MethodSpec newSpec = new MethodSpec(description, parent().name());
	if (newSpec.ident(spec)) {
	    if (!newSpec.sameReturnType(spec)) {
		//if (spec.isConstructor() &&
		//    newSpec.methodName().euqals(spec.methodName())) {
		//    SystemManager.information("A constructor should not be defined to return any type.");
		//    return null;
		//}
		if (!SystemManager.confirm(
			    "Method " + spec.identityName() +
			    " has been defined to return " + spec.returnType() +
			    ".\nChange it?",
			    "Change")) return null;
		spec = newSpec;
	    }
	    return changed(newCode);
	} else {
	    MethodManager target;
	    Enumeration m = parent().children();
	    while (m.hasMoreElements()) {
		target = (MethodManager)m.nextElement();
		if (target.spec().ident(newSpec)) {
		    boolean override =
			SystemManager.confirm(
				"Method " + target.spec().prototype() +
				" already exists in this class.\nOverride it?",
				"Override");
		    if (override) {
			return target.setCode(newCode);
		    } else {
			return null;
		    }
		}
	    }
	    // create new method after me
	    target = new MethodManager(parent(), newSpec);
	    target.brother = brother;
	    brother = target;
	    return target.created(newCode);
	}
    }

    /**
     * get source code for this method
     */
    protected String getCodeInternal() throws java.io.IOException, DejavaException{
	if (changes != null) {
	    return changes.getCode();
	} else {
	    sourceFileManager().readSource();
	    if (parent == null)
		throw new DejavaNotFoundException("method " + name() + " seems to have been removed");
	    File f = file();
	    String code = range.getCode(f);
	    return code;
	}
    }

    /**
     * get source code for this method
     */
    public File file() {
	return ((ClassManager)parent).file();
    }

    /**
     * write code on PrintStream
     */
    void writeOn(PrintStream ps, boolean synced) throws java.io.IOException {
	String code = getCode();
	int tail = code.length();
	while (tail > 0 && code.charAt(tail - 1) < ' ') tail--;
	ps.println("");
	ps.println(code.substring(0, tail));
	if (synced && isChanged()) changes.synced();
    }

    /**
     * return offset from beggining of this method
     */
    public int lineOffset(int line) {
	if (line <= range.endLine()) {
	    return line - (int)range.startLine();
	} else {
	    return -1;
	}
    }

    void copyFrom(MethodManager mm, boolean force) {
	if (force || changes == null) {
	    spec = mm.spec;
	    range = mm.range;
	    changes = mm.changes;
	} else {
	    range = mm.range;
	}
    }

    /**
     * rename this method
     */
    public Manager rename(String newName) throws DejavaException {
	String code = getCode();
	StringBuffer newCode = new StringBuffer();
	CodeReader reader = new CodeReader(code);
	reader.nextDescription();
	int arg = code.indexOf('(', reader.startPoint());
	int index = code.lastIndexOf(spec.methodName(), arg);
	newCode.append(code.substring(0, index));
	newCode.append(newName);
	newCode.append(code.substring(arg));
	spec.rename(newName);
	name = spec.identityName();
	return renamed(newCode.toString());
    }

    /**
     * dispose class specific data
     */
    protected void dispose() {
	//disposeDetails();
	super.dispose();
    }

    void disposeDetails() {
	if (spec != null) {
	    spec.dispose();
	    spec = null;
	}
	if (range != null) {
	    range.dispose();
	    range = null;
	}
	if (changes != null) {
	    changes.dispose();
	}
    }
}
