/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import dejava.lang.*;
import dejava.util.StringSorter;
import dejava.util.StringSourceCode;
import java.io.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;

/**
 * package management class
 * children are instances of SourceFileManager
 */
public class PackageManager extends PathedManager {
    /**
     * all packages managed by the system
     * each package must have unique name
     */
    private static Hashtable packages = new Hashtable();

    /**
     * default header file for this project
     */
    private File header = null;

    public static String nameFor(String path) {
	return path.replace(File.separatorChar, '.');
    }

    public static String topName(Manager parent) {
	return "@" + parent.name();
    }

    public static PackageManager findPackage(String name) {
	if (packages.containsKey(name)) {
	    return (PackageManager)packages.get(name);
	} else {
	    return null;
	}
    }

    public static PackageManager forName(PathedManager parent, String name, boolean readOnly) {
	if (name.equals(topName(parent))) {
	    return new TopDirectoryPackageManager(parent);
	} else {
	    return new PackageManager(parent, name,
				name.replace('.', File.separatorChar));
	}
    }

    public static PackageManager forName(PathedManager parent, String name) {
	if (name.equals(topName(parent))) {
	    return new TopDirectoryPackageManager(parent);
	} else {
	    return new PackageManager(parent, name, name.replace('.', File.separatorChar));
	}
    }

    /**
     * the constructor
     */
    PackageManager(Manager parent, String name, String path) {
	super(parent, name, path);
	packages.put(name, this);
    }

    /**
     * enumerate all packages managed by the system
     */
    public static Enumeration allPackages() {
	return packages.elements();
    }

    /**
     * get package CodePath
     */
    public CodePath getCodePath() {
	if (parent != null) {
	    CodePath cp = parent.getCodePath();
	    cp.packageManager(this);
	    return cp;
	} else {
	    return new CodePath(null, null, this, null, null);
	}
    }

    /**
     * whether it is editable or not
     */
    public boolean isEditable() {
	if (commentFile().exists()) {
	    return commentFile().canWrite();
	} else {
	    return directory().canWrite();
	}
    }

    /**
     * whether it has default header or not
     */
    public boolean hasDefaultHeader(boolean mustBeEditable) {
	if (headerFile().exists()) {
	    if (headerFile().canRead()) {
		return mustBeEditable ? headerFile().canWrite() : true;
	    }
	    SystemManager.information("Warning: The default header file for " + name +
					" exists but cannot read");
	}
	return false;
    }

    /**
     * set comment for this Package
     */
    public Manager setCode(String newComment) {
	try {
	    File commentFile = commentFile();
	    File backupFile = backupFile(commentFile);
	    File newFile = newFile(commentFile);
	    FileOutputStream fos = new FileOutputStream(newFile);
	    PrintStream ps = new PrintStream(fos);
	    ps.print(newComment);
	    ps.close();
	    commentFile.renameTo(backupFile);
	    newFile.renameTo(commentFile);
	    return this;
	} catch (java.io.IOException e) {
	    return null;
	}
    }

    /**
     * get package comment
     */
    public String getCodeInternal() throws java.io.IOException {
	File commentFile = commentFile();
	if (commentFile.exists()) {
	    RandomAccessFile raf = new RandomAccessFile(commentFile, "r");
	    byte buf[] = new byte[(int)raf.length()];
	    raf.read(buf);
	    raf.close();
	    return new String(buf, 0);
	} else {
	    return "Package comment may be described here";
	}
    }

    /**
     * full name for this package
     */
    String fullName() {
	if (parent != null) {
	    return parent.name() + "." + name();
	} else {
	    return name();
	}
    }

    /**
     * get File object for package comment
     */
    File commentFile() {
	return new DejavaFile(directory(), "README");
    }

    /**
     * get File object for project default header
     */
    File headerFile() {
	if (header == null) {
	    header = new DejavaFile(directory(), ".header");
	}
	return header;
    }

    /**
     * get database file contains class list of this package
     */
    File sourcesFile() {
	return new DejavaFile(((ProjectManager)parent).profileDirectory(), fullName());
    }

    /**
     * synchronize all source files and class list database on sourcesFile
     */
    public void sync() {
	super.sync();
	updateSourcesFile();
    }

    /**
     * read sources/classes list file
     */
    void readSourcesFile() {
	/*
	if (child != null) {
	    disposeChildren();
	}
	*/
	File srcsfile = sourcesFile();
	try {
	    String line;
	    String pr = null;
	    FileInputStream fis = new FileInputStream(srcsfile);
	    DataInputStream dis = new DataInputStream(fis);
	    line = dis.readLine();
	    while (line != null) {
		SourceFileManager sfm = new SourceFileManager(this, line);
		addChild(sfm);
	        while ((line = dis.readLine()) != null && line.charAt(0) == '\t') {
		    ClassManager cm = new ClassManager(sfm, line.substring(1));
		    sfm.addChild(cm);
		}
	    }
	    dis.close();
	} catch(java.io.IOException e) {
	    SystemManager.exception("Could not read file: " + srcsfile.getPath(), e);
	}
    }

    /**
     * update sources/classes list file
     */
    void updateSourcesFile() {
	SourceFileManager.clearLastSearchedCache();
	File srcsFile = sourcesFile();
	File backupFile = backupFile(srcsFile);
	File newFile = newFile(srcsFile);
	try {
	    FileOutputStream fos = new FileOutputStream(newFile);
	    PrintStream ps = new PrintStream(fos);
	    for (Enumeration c = children(); c.hasMoreElements(); ) {
		SourceFileManager sfm = (SourceFileManager)c.nextElement();
		sfm.writeListOn(ps);
	    }
	    ps.close();
	    srcsFile.renameTo(backupFile);
	    newFile.renameTo(srcsFile);
	} catch(java.io.IOException e) {
	    SystemManager.exception("Could not write file: " + srcsFile.getPath(), e);
	}
    }

    /**
     * create a package. it makes directory if it doesn't exist
     */
    boolean create() {
	File file = file();
	boolean exist = file.exists();
	if (!exist) {
	    file.mkdirs();
	}
	return rebuild(exist);
    }

    /**
     * rebuild database for package. force Transcript raised
     */
    public boolean rebuild() {
	return rebuild(true);
    }

    /**
     * rebuild database for package
     */
    public boolean rebuild(boolean raise) {
	File dir = directory();
	if (raise) SystemManager.raiseTranscript();
	return rebuild(dir, StringSorter.sort(dir.list()));
    }

    /**
     * rebuild database for package
     */
    public boolean rebuild(File dir, String files[]) {
	disposeChildren();
	for (int i = 0; i < files.length; i++) {
	    if (files[i].endsWith(".java")||files[i].endsWith(".jav")||files[i].endsWith(".JAV")) {
		try {
		    SourceFileManager sfm = new SourceFileManager(this, files[i]);
		    sfm.readSource();
		    addChild(sfm);
		} catch (DejavaException e) {
		    SystemManager.exception("Could not read source file: ", e);
		}
	    }
	}
	updateSourcesFile();
	return child != null;
    }

    /**
     * rename a package
     */
    public Manager rename(String newName) {
	Manager ret = rename(newName, true);
	if (ret != null) {
	    ((ProjectManager)parent).updatePackagesFile();
	}
	return ret;
    }

    /**
     * rename a package
     */
    public Manager rename(String newName, boolean replacePackageStatements) {
	String newPath = newName.replace('.', File.separatorChar);
	File newFile = new DejavaFile(((PathedManager)parent).directory(), newPath);
	String oldName = name;
	File oldFile = file();
	File sourcesFile = sourcesFile();
	try {
	    newFile.mkdirs();
	    String files[] = oldFile.list();
	    boolean canRemove = true;
	    for (int i = 0; i < files.length; i++) {
		File f = new DejavaFile(oldFile, files[i]);
		if (f.isDirectory()) {
		    if (parent.findChild(oldName + "." + files[i]) != null) {
			// sub package is found
			canRemove = false;
		    } else if (files[i].equals(".") || files[i].equals("..")) {
			continue;
		    }
		} else if (files[i].endsWith(".java") && replacePackageStatements) {
		    SourceFileManager sfm = findSource(files[i]);
		    sfm.moveTo(oldName, newName);
		}
		f.renameTo(new File(newFile, files[i]));
	    }
	    if (canRemove) {
		/*
		 old package directory should be removed. but
		 java.io.File has no method for remove file... sigh
		 oldFile.remove();
		 */
	    }
	    packages.remove(oldName);
	    packages.put(newName, this);
	    super.rename(newName);
	    File newSourcesFile = sourcesFile();
	    sourcesFile.renameTo(newSourcesFile);
	    setPath(newPath);
	    SystemManager.messageln("Package " + oldName + " is renamed to " + newName + ".");
	    return this;
	} catch (Exception e) {
	    SystemManager.exception("Could not rename package " + oldName + " to " + newName, e);
	    return null;
	}
    }

    public Enumeration sources() {
	return children();
    }

    public Enumeration sourceNames() {
	return childNames();
    }

    public int numSources() {
	return numChildren();
    }

    public Enumeration classes() {
	Vector classes = new Vector();
	for (Enumeration s = sources(); s.hasMoreElements(); ) {
	    SourceFileManager sfm = (SourceFileManager)s.nextElement();
	    for (Enumeration c = sfm.classes(); c.hasMoreElements(); ) {
		classes.addElement(c.nextElement());
	    }
	}
	return classes.elements();
	// return new ManagerEnumerator(child, 2);
    }

    public Enumeration classNames() {
	Vector classes = new Vector();
	for (Enumeration s = sources(); s.hasMoreElements(); ) {
	    SourceFileManager sfm = (SourceFileManager)s.nextElement();
	    for (Enumeration c = sfm.classes(); c.hasMoreElements(); ) {
		Manager m = (Manager)c.nextElement();
		classes.addElement(m.name());
	    }
	}
	return classes.elements();
	// return new ManagerEnumerator(child, 2);
    }

    public int numClasses() {
	int count = 0;
	for (Enumeration s = sources(); s.hasMoreElements(); ) {
	    SourceFileManager sfm = (SourceFileManager)s.nextElement();
	    count += sfm.numClasses();
	}
	return count;
    }

    public ClassManager findClass(String name) {
	for (Enumeration s = sources(); s.hasMoreElements(); ) {
	    SourceFileManager sfm = (SourceFileManager)s.nextElement();
	    ClassManager cm = sfm.findClass(name);
	    if (cm != null) {
		return cm;
	    }
	}
	return null;
    }

    public SourceFileManager findSourceForClass(String name) {
	for (Enumeration s = sources(); s.hasMoreElements(); ) {
	    SourceFileManager sfm = (SourceFileManager)s.nextElement();
	    if (sfm.findClass(name) != null) {
		return sfm;
	    }
	}
	return null;
    }

    public SourceFileManager findSource(String name) {
	return (SourceFileManager)findChild(name);
    }

    /**
     * create SourceFileManager and add it to the list
     * SFMs are sorted in dictionary order
     */
    public SourceFileManager createSourceFileManager(String fname) {
	SourceFileManager sfm = new SourceFileManager(this, fname);
	SourceFileManager prev = null;
	SourceFileManager next = null;
	Enumeration s = sources();
	while (s.hasMoreElements()) {
	    next = (SourceFileManager)s.nextElement();
	    if (next.name().compareTo(fname) > 0) break;
	    prev = next;
	    next = null;
	}
	if (prev == null) {
	    sfm.brother = child;
	    child = sfm;
	} else {
	    sfm.brother = next;
	    prev.brother = sfm;
	}
	sfm.setDefaultHeader();
	return sfm;
    }

    /**
     * create a new class from ClassSpec
     * this method is only called from Browser
     */
    public ClassManager createClass(ClassSpec spec) {
	try {
	    return createClass(spec, spec.description() + " {\n}\n");
	} catch (DescriptionException e) {
	    SystemManager.exception("ClassSpec produced illegal description", e);
	    return null;
	}
    }

    /**
     * create new class from ClassSpec and code
     * file name to add the class is requested to user
     */
    ClassManager createClass(ClassSpec spec, String newCode)
    throws DescriptionException {
	if (findClass(spec.className()) != null) {
	    SystemManager.beep();
	    SystemManager.information("Class " + spec.className() + " already exists.");
	    return null;
	}
	String fname = spec.className() + ".java";
	if (!spec.isPublic()) {
	    // public class description should go into className.java
	    fname = SystemManager.requestFileName(
			"What is the filename for the class " + spec.className() + "?",
			absolutePath(), fname);
	    // null indicates cancel create operation
	    if (fname == null) return null;
	}
	return createClass(spec, newCode, fname);
    }

    /**
     * internal class creation method
     * class name confliction should be checked before calling this method
     */
    private ClassManager createClass(ClassSpec spec, String newCode, String fname)
    throws DescriptionException {
	SourceFileManager sfm = findSource(fname);
	if (sfm == null) {
	    sfm = createSourceFileManager(fname);
	}
	if (!sfm.isEditable()) {
	    return null;
	}
	ClassManager cm = new ClassManager(sfm, spec);
	if (cm.resolveDependancy()) {
	    sfm.addChild(cm);
	    updateSourcesFile();
	    cm.created(newCode);
	    return cm;
	} else {
	    return null;
	}
    }

    /**
     * returns package statment for this package
     */
    protected String packageStatement() {
	return "package " + name() + ";\n";
    }

    /**
     * remove a source file from this package
     * update database
     */
    protected Manager removeChild(Manager child) {
	if (super.removeChild(child) != null) return child;
	updateSourcesFile();
	return null;
    }

    /**
     * dispose me
     */
    protected void dispose() {
	packages.remove(name);
	super.dispose();
    }
}
