/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import dejava.util.ChangeList;
import dejava.util.StringSorter;
import dejava.gui.Transcript;
import dejava.gui.DejavaFrame;
import java.io.*;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * project management class
 * children are instances of PackageManager
 */
public final class ProjectManager extends PathedManager {
    /**
     * the directory in which database for this project goes
     */
    private File profileDirectory;

    /**
     * change history for this project
     */
    private ChangeList changes;

    /**
     * default header file for this project
     */
    private File header = null;

    public Manager remove() {
	if (equals(SystemManager.getProjects())) {
	    SystemManager.setProjects((ProjectManager)brother);
	} else {
	    Enumeration elders = SystemManager.allProjects();
	    while (elders.hasMoreElements()) {
		ProjectManager prev = (ProjectManager)elders.nextElement();
		if (equals(prev.brother)) {
		    prev.brother = brother;
		    break;
		}
	    }
	}
	disposeChildren();
	// updatePackagesFile();
	SystemManager.updateProjects();
	return null;
    }

    ProjectManager(String name, String path) throws java.io.IOException {
	super(null, name, path);
	profileDirectory = new File(path + File.separator + SystemManager.PROFDIRNAME);
	if (!profileDirectory.exists()) {
	    profileDirectory = SystemManager.profileDirectory();
	    changes = SystemManager.changes();
	} else {
	    changes = new ChangeList(new File(profileDirectory, ChangeList.CHANGES));
	}
    }

    /**
     * get project CodePath
     */
    public CodePath getCodePath() {
	return new CodePath(null, this, null, null, null);
    }

    /**
     * whether it is editable or not
     */
    public boolean isEditable() {
	if (commentFile().exists()) {
	    return commentFile().canWrite();
	} else {
	    return directory().canWrite();
	}
    }

    /**
     * whether it has default header or not
     */
    public boolean hasDefaultHeader(boolean mustBeEditable) {
	if (headerFile().exists()) {
	    if (headerFile().canRead()) {
		return mustBeEditable ? headerFile().canWrite() : true;
	    }
	    SystemManager.information("Warning: The default header file for " + name +
					" exists but cannot read");
	}
	return false;
    }

    /**
     * set comment for this Project
     */
    public Manager setCode(String newComment) {
	try {
	    File commentFile = commentFile();
	    File backupFile = backupFile(commentFile);
	    File newFile = newFile(commentFile);
	    FileOutputStream fos = new FileOutputStream(newFile);
	    PrintStream ps = new PrintStream(fos);
	    ps.print(newComment);
	    ps.close();
	    commentFile.renameTo(backupFile);
	    newFile.renameTo(commentFile);
	    return this;
	} catch (java.io.IOException e) {
	    return null;
	}
    }

    /**
     * get project comment
     */
    protected String getCodeInternal() throws java.io.IOException {
	File commentFile = commentFile();
	if (commentFile.exists()) {
	    RandomAccessFile raf = new RandomAccessFile(commentFile, "r");
	    byte buf[] = new byte[(int)raf.length()];
	    raf.read(buf);
	    raf.close();
	    return new String(buf, 0);
	} else {
	    return "Project comment may be described here";
	}
    }

    /**
     * get File object for project comment
     */
    File commentFile() {
	return new DejavaFile(directory(), "README");
    }

    /**
     * get File object for project default header
     */
    File headerFile() {
	if (header == null) {
	    header = new DejavaFile(directory(), ".header");
	}
	return header;
    }

    /**
     * get ChangeList for this project
     */
    public ChangeList changes() {
	return changes;
    }

    /**
     * rebuild database for project
     */
    public boolean rebuild() {
	SystemManager.raiseTranscript();
	disposeChildren();
	try {
	    Vector pkgs = listPackages(directory());
	    File packagesFile = packagesFile();
	    File backup = backupFile(packagesFile);
	    packagesFile.renameTo(backup);
	    FileOutputStream fos = new FileOutputStream(packagesFile);
	    if (pkgs.size() > 0) {
		PrintStream ps = new PrintStream(fos);
		for (Enumeration e = pkgs.elements(); e.hasMoreElements(); ) {
		    PackageManager p = (PackageManager)e.nextElement();
		    addChild(p);
		    ps.println(p.name());
		}
		ps.close();
		return true;
	    } else {
		return false;
	    }
	} catch (java.io.IOException e1) {
	    SystemManager.exception("Could not write packages list on "
					+ packagesFile().getPath() + ".", e1);
	    return false;
	}
    }

    /**
     * list up packages for project
     * it is called recursive
     * also lists classes for packages
     * @return package list
     */
    public Vector listPackages(File dir) throws java.io.IOException {
	PackageManager pkgMan = null;
	Vector sub = new Vector();
	String files[] = StringSorter.sort(dir.list());
	if (files == null) {
	    // no files and directories exist
	    return sub;
	}
	for (int i = 0; i < files.length; i++) {
	    String fn = files[i];
	    File f = new File(dir, fn);
	    if (f.isDirectory()) {
		Vector v = listPackages(f);
		for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
		    sub.addElement(e.nextElement());
		}
	    } else if (fn.endsWith(".java")||fn.endsWith(".jav")||fn.endsWith(".JAV")) {
		if (pkgMan == null) {
		    if (dir == directory()) {
			// is top directory
			pkgMan = new TopDirectoryPackageManager(this);
		    } else {
			String path = dir.getPath().substring(path().length() + 1);
			pkgMan = new PackageManager(this, PackageManager.nameFor(path), path);
		    }
		}
	    }
	}
	if (pkgMan != null) {
	    sub.insertElementAt(pkgMan, 0);
	    pkgMan.rebuild(dir, files);
	}
	return sub;
    }

    /**
     * add package entry
     */
    public PackageManager createPackageAfter(String pkgname, PackageManager pm) {
	if (PackageManager.findPackage(pkgname) != null) {
	    SystemManager.information("Package already exists: " + pkgname, "Abort");
	    return null;
	} else if (!directory().canWrite()) {
	    SystemManager.information("Project directory isn't writable:\n" +
					directory().getPath().toString(), "Abort");
	    return null;
	}
	PackageManager pman = PackageManager.forName(this, pkgname, false);
	if (pman != null) {
	    if (pm == null) {
		addChild(pman);
	    } else {
		pman.brother = pm.brother;
		pm.brother = pman;
	    }
	    pman.create();
    /*
	    RandomAccessFile raf = new RandomAccessFile(packagesFile(), "rw");
	    if (packagesFile().exists()) raf.seek(raf.length());
	    raf.writeBytes(pkgname + SystemManager.nl);
	    raf.close();
     */
	    updatePackagesFile();
	    return pman;
	}
	return null;
    }

    public File profileDirectory() {
	return profileDirectory;
    }

    public int numPackages() {
	return numChildren();
    }

    public Enumeration packages() {
	return children();
    }

    public Enumeration packageNames() {
	return childNames();
    }

    public PackageManager findPackage(String name) {
	return (PackageManager)findChild(name);
    }

    /**
     * get database file contains package list of this Project
     */
    File packagesFile() {
	return new DejavaFile(profileDirectory, name);
    }

    /**
     * read the file contains package list in this project
     */
    void readPackagesFile() {
	if (child != null) {
	    disposeChildren();
	}
	File pkgsfile = packagesFile();
	try {
	    if (!pkgsfile.exists()) return;
	    String line;
	    String pr = null;
	    FileInputStream fis = new FileInputStream(pkgsfile);
	    DataInputStream dis = new DataInputStream(fis);
	    while ((line = dis.readLine()) != null) {
		PackageManager pman = PackageManager.forName(this, line);
		addChild(pman);
		pman.readSourcesFile();
	    }
	    dis.close();
	    fis.close();
	} catch(java.io.IOException e) {
	    SystemManager.exception("Could not read package list file: " + pkgsfile.getPath(), e);
	}
    }

    /**
     * update package list database file
     */
    void updatePackagesFile() {
	File pkgsFile = packagesFile();
	try {
	    String line;
	    String pr = null;
	    File backupFile = backupFile(pkgsFile);
	    File newFile = newFile(pkgsFile);
	    FileOutputStream fos = new FileOutputStream(newFile);
	    PrintStream ps = new PrintStream(fos);
	    Enumeration p = packages();
	    while (p.hasMoreElements()) {
		PackageManager pman = (PackageManager)p.nextElement();
		ps.println(pman.name());
	    }
	    ps.close();
	    pkgsFile.renameTo(backupFile);
	    newFile.renameTo(pkgsFile);
	} catch(java.io.IOException e) {
	    SystemManager.exception("Could not update package list file: " + pkgsFile.getPath(), e);
	}
    }

    /**
     * remove a package from this project
     */
    protected Manager removeChild(Manager child) {
	if (super.removeChild(child) != null) return child;
	updatePackagesFile();
	return null;
    }

    /**
     * rename this project
     */
    public Manager rename() throws DejavaException {
	String newName =
	    SystemManager.request("new name for " + name, name);
	if (newName == null || newName.equals(name)) return null;
	if (SystemManager.findProject(newName) != null) {
	    SystemManager.information(
		"Specified name is already used for another project");
	    return null;
	}
	File dir = profileDirectory();
	String files[] = dir.list();
	for (int i = 0; i < files.length; i++) {
	    String newFname = null;
	    if (files[i].equals(name)) {
		newFname = newName;
	    } else if (files[i].startsWith(name + ".")) {
		newFname = newName + files[i].substring(name.length());
	    } else continue;
	    File from = new DejavaFile(dir, files[i]);
	    File to = new DejavaFile(dir, newFname);
	    from.renameTo(to);
	}
	Transcript.getTranscriptMenuBar().renameProject(name, newName);
	SystemManager.updateProjects();
	return super.rename(newName);
    }
}
