/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import dejava.lang.*;
import dejava.util.*;
import java.io.*;
import java.util.*;

import dejava.sys.PathedManager;

/**
 * source file management class
 * children are instances of ClassManager
 */
public final class SourceFileManager extends PathedManager {
    private SourceCodeRange header = null;
    private SourceCodeRange trailer = null;
    private static final StringSourceCode emptyHeader = new StringSourceCode("");

    // last modified time
    // -1 indicates not read yet
    private long lastModified = -1;

    // whether this file has been changed or not
    private boolean changed;

    // list of changed sources
    private static Hashtable changedSources = new Hashtable();

    // last searched file (for findSourceFileManager)
    private static String lastSearchedFilename = null;
    private static SourceFileManager lastFoundSourceFileManager = null;

    public static void syncAllChangedSources() {
	Enumeration sources = changedSources.elements();
	while (sources.hasMoreElements()) {
	    SourceFileManager sfm = (SourceFileManager)sources.nextElement();
	    sfm.sync();
	}
    }

    public static Enumeration changedSources() {
	return changedSources.elements();
    }

    public static SourceFileManager findSourceFileManagerForFile(String fname) {
	if (lastSearchedFilename != null &&
	    lastSearchedFilename.equals(fname)) {
	    return lastFoundSourceFileManager;
	}
	int index = fname.lastIndexOf(File.separatorChar);
	String dir = null;
	if (index >= 0) {
	    dir = fname.substring(0, index);
	    fname = fname.substring(index + 1);
	}
	Enumeration p = PackageManager.allPackages();
	while (p.hasMoreElements()) {
	    PackageManager pman = (PackageManager)p.nextElement();
	    if (index >= 0 && !pman.absolutePath().endsWith(dir)) {
		continue;
	    }
	    SourceFileManager sfm = pman.findSource(fname);
	    if (sfm != null) {
		lastSearchedFilename = fname;
		lastFoundSourceFileManager = sfm;
		return sfm;
	    }
	}
	lastSearchedFilename = fname;
	lastFoundSourceFileManager = null;
	return null;
    }

    public static void clearLastSearchedCache() {
	lastSearchedFilename = null;
	lastFoundSourceFileManager = null;
    }

    SourceFileManager(Manager parent, String path) {
	super(parent, path, path);
	changed = false;
    }

    public CodePath getCodePath() {
	if (parent != null) {
	    return parent.getCodePath();
	} else {
	    return CodePath.nullCodePath;
	}
    }

    public String getClassPath() {
	if (headerPackageName() != null) {
	    return getCodePath().projectManager().absolutePath();
	} else {
	    return file.getParent();
	}
    }

     /**
     * whether this file is editable or not
     */
    public boolean isEditable() {
	if (file().exists()) {
	    return file().canWrite();
	} else {
	    return directory().canWrite();
	}
    }

    /**
     * whether this file is updated after last compilation
     */
    public boolean isUpdatedAfterLastCompilation() {
	if (isChanged() || !file().exists()) return true;
	File directory = directory();
	long lastModified = file().lastModified();
	Enumeration c = classes();
	while (c.hasMoreElements()) {
	    ClassManager cm = (ClassManager)c.nextElement();
	    if (cm.isUpdatedAfterLastCompilation(lastModified, directory))
		    return true;
	}
	return false;
    }

    /**
     * set whole contents of me
     */
    public Manager setCode(String newCode) throws DejavaException {
	try {
	    File sourceFile = file();
	    File backupFile = backupFile(sourceFile);
	    File newFile = newFile(sourceFile);
	    SystemManager.message("Updating " + sourceFile.getPath() + " ... ");
	    PrintStream ps =
		new PrintStream(new BufferedOutputStream(new FileOutputStream(newFile)));
	    ps.print(newCode);
	    ps.close();
	    sourceFile.renameTo(backupFile);
	    newFile.renameTo(sourceFile);
	    readSource(true, false);
	    SystemManager.messageln("done.");
	} catch (java.io.IOException e) {
	    throw new DejavaException("Could not wrote file: " + e.getMessage());
	}
	return this;
    }

    /**
     * get whole contants of this file
     */
    public String getCodeInternal() throws DejavaException {
	try {
	    sync();
	    FileInputStream fis = new FileInputStream(file());
	    byte buf[] = new byte[fis.available()];
	    fis.read(buf);
	    return new String(buf, 0);
	} catch (java.io.IOException e) {
	   return "";
	}
    }

    /**
     * something's been changed in this file
     */
    public void changed() {
	changed = true;
	changedSources.put(this, this);
    }

    /**
     * whether this file is changed or not
     */
    public boolean isChanged() {
	return changed;
    }

    /**
     * sync a file
     */
    public void sync() {
	if (isChanged()) {
	    try {
		readSource();
		SystemManager.message("Updating " + absolutePath() + " ... ");
		writeSource(file(), true);
		readSource(true, false);
		SystemManager.messageln("done.");
		changedSources.remove(this);
	    } catch (DejavaException e) {
		SystemManager.exception("Could not sync source file: " + absolutePath(), e);
	    }
	}
    }

    void moveTo(String oldPkg, String newPkg) {
	String code = header().getCode(file());
	int head = code.indexOf(oldPkg, code.indexOf("package"));
	int tail = head + oldPkg.length();
	String newCode = code.substring(0, head) + newPkg + code.substring(tail);
	try {
	    ((ClassManager)child).header().setCode(newCode);
	} catch (DescriptionException e) {
	    // not reached
	    SystemManager.exception("dejava internal exception", e);
	}
	file = null;
    }

    SourceCodeRange header() {
	if (header == null) {
	    try {
		readSource();
	    } catch (DejavaException e) {
		SystemManager.exception("Could not get header", e);
		return null;
	    }
	}
	return header;
    }

    String headerCode() {
	if (header() != null) {
	    return header().getCode(file());
	} else {
	    return null;
	}
    }

    void header(SourceCodeRange newHeader) {
	header = newHeader;
    }

    /*
     * set default header for this file.
     * import if Project/Package default header file exists
     */
    void setDefaultHeader() {
	PackageManager pm = (PackageManager)parent;
	boolean isProjectDefault = true;
	File headerFile = null;
	if (pm.hasDefaultHeader(false)) {
	    isProjectDefault = false;
	    headerFile = pm.headerFile();
	} else if (((ProjectManager)pm.parent()).hasDefaultHeader(false)) {
	    headerFile = ((ProjectManager)pm.parent()).headerFile();
	}
	if (headerFile != null) {
	    StringBuffer hc = new StringBuffer();
	    try {
		DataInputStream dis =
		    new DataInputStream(new FileInputStream(headerFile));
		String buffer;
		while ((buffer = dis.readLine()) != null) {
		    hc.append(buffer).append('\n');
		}
		dis.close();
	    } catch (java.io.IOException e) {
		// file existence and readability are checked in hasDefaultHeader(),
		// so never falls into this block
		SystemManager.exception("INTERNAL ERROR: SourceFileManager#setDefaultHeader()", e);
	    }
	    if (isProjectDefault) {
		// should add package statement
		hc.append(pm.packageStatement());
	    }
	    header(new StringSourceCode(hc.toString()));
	} else {
	    header(new StringSourceCode(pm.packageStatement()));
	}
    }

    String headerPackageName() {
	String code = headerCode();
	CodeReader reader = new CodeReader(code, 1, true);
	try {
	    while (reader.nextDescription() == 0) {
		String desc = reader.description();
		if (desc.startsWith("package ")) {
		    return desc.substring(8).trim();
		}
	    }
	} catch (DescriptionException e) {
	    SystemManager.exception("Header has something bad: " + code, e);
	}
	return null;
    }

    Vector imported() {
	Vector imported = new Vector();
	String code = headerCode();
	CodeReader reader = new CodeReader(code, 1, true);
	try {
	    while (reader.nextDescription() == 0) {
		String desc = reader.description();
		if (desc.startsWith("import ")) {
		    String target = desc.substring(7).trim();
		    int lastIndex = target.lastIndexOf('.');
		    PackageManager pman =
			PackageManager.findPackage(target.substring(0, lastIndex));
		    if (pman == null) continue;
		    if (!target.endsWith("*")) {
			ClassManager cm = pman.findClass(target.substring(lastIndex + 1));
			if (cm == null) continue;
			imported.addElement(cm);
		    } else {
			imported.addElement(pman);
		    }
		}
	    }
	} catch (DescriptionException e) {
	    SystemManager.exception("Header has something bad: " + code, e);
	}
	return imported;
    }

    void readSource() throws DejavaException {
	if (lastModified == -1) {
	    // not read yet before
	    if (header != null) {
		// just created and not have been *SYNC* ed
		// so no need to read file
		return;
	    }	
	    SystemManager.message("Reading " + absolutePath() + " ... ");
	    readSource(true, false);
	    SystemManager.messageln("done.");
	 } else if (lastModified < file().lastModified()) {
	    SystemManager.messageln("File may be changed. Checking conflicts ...");
	    readSource(false, true);
	    SystemManager.messageln("done.");
	}
    }

    synchronized
    void readSource(boolean force, boolean verbose)
    throws DejavaException {
	try {
	    File sourceFile = file();
	    lastModified = file().lastModified();
	    FileInputStream file = new FileInputStream(sourceFile);
	    Vector order = new Vector();
	    Hashtable hash = new Hashtable();

	    CodeReader reader = new CodeReader(file);

	    int nest = reader.nextDescription(false);
	    if (reader.startPoint() <= 0) {
		header = emptyHeader;
	    } else {
	        header = new SourceCodeRange(0, reader.startPoint(),
					    1, reader.startLine() - 1);
	    }
	    while (nest >= 0) {
		ClassManager cman = ClassManager.readFrom(this, reader);
		ClassManager orgCman = findClass(cman.name());
		if (orgCman != null) {
		    orgCman.copyFrom(cman, force, verbose);
		    order.addElement(orgCman);
		    if (!force && verbose)
			SystemManager.messageln("  ! " + orgCman.name());
		} else {
		    cman.parent = this;
		    order.addElement(cman);
		    if (verbose)
			SystemManager.messageln("  + " + cman.name());
		}
	        nest = reader.nextDescription();
	    }
	    if (reader.startPoint() < reader.endPoint()) {
		trailer = new SourceCodeRange(
				    reader.startPoint(), reader.endPoint(),
				    reader.startLine(),  reader.endLine());
	    }
	    file.close();

	    Enumeration c = children();
	    while (c.hasMoreElements()) {
		ClassManager cm = (ClassManager)c.nextElement();
		if (!hash.containsKey(cm.name())) {
		    if (cm.code == null) {
			cm.parent = null;
			if (verbose)
			    SystemManager.messageln("  - " + cm.name());
		    }
		}
	    }

	    c = order.elements();
	    if (!c.hasMoreElements()) {
		return;
	    }
	    Manager prev = (Manager)c.nextElement();
	    child = prev;
	    while (c.hasMoreElements()) {
		prev.brother = (Manager)c.nextElement();
	        prev = prev.brother;
	    }
	    prev.brother = null;
	    //NOTYET
	    /*
	    CodePath cp = SourceFileManager.codePath(fileName);
	    String fname = fileName.substring(fileName.lastIndexOf(File.separatorChar) + 1);
	    if (packageName == null) {
		SystemManager.messageln("\nWarning: File '" + fname + "' is in package '" + cp.printString() + "',");
		SystemManager.messageln("         but no package statement presents.");
		packageName = cp.pkgName;
	    } else if (!packageName.equals(cp.pkgName)) {
		SystemManager.messageln("\nWarning: File '" + fname + "' is in package '" + cp.printString() + "',");
		SystemManager.messageln("         but package statement tells it's in " + packageName + ".");
		packageName = cp.pkgName;
	    }
	    */
	} catch (dejava.lang.DescriptionException e) {
	    SystemManager.messageln("\n  " + e.getMessage() + "\ndone.");
	    SystemManager.exception("Description error in: " + absolutePath(), e);
	    throw new DejavaSourceFileException();
	} catch (java.io.IOException e) {
	    SystemManager.messageln("\n  could not read source file\ndone.");
	    SystemManager.errorln("Could not read source: " + absolutePath());
	    throw new DejavaSourceFileException();
	}
    }

    /**
     * write source code on file
     */
    public void writeSource(File file) {
	writeSource(file, false);
    }

    /**
     * write source code on file
     */
    public void writeSource(File sourceFile, boolean clearChanges) {
	try {
	    File backupFile = backupFile(sourceFile);
	    File newFile = newFile(sourceFile);
	    FileOutputStream fos = new FileOutputStream(newFile);
	    PrintStream ps = new PrintStream(fos);
	    String headerCode = header.getCode(sourceFile).trim();
	    if (headerCode.length()> 0) {
	        ps.println(headerCode);
	        ps.println("");
	    }
	    for (Enumeration c = classes(); c.hasMoreElements(); ) {
		ClassManager cm = (ClassManager)c.nextElement();
		cm.writeOn(ps, clearChanges);
	    }
	    if (trailer != null) {
		ps.println("");
		ps.print(trailer.getCode(sourceFile));
	    }
	    ps.close();
	    sourceFile.renameTo(backupFile);
	    newFile.renameTo(sourceFile);
	    changed = false;
	} catch (java.io.IOException e) {
	    SystemManager.exception("Could not write source", e);
	}
    }

    void rebuild(boolean verbose) {
	Manager originalChildren = child;
	try {
	    readSource();
	} catch (DejavaException e) {
	    SystemManager.exception("Could not rebuild: " + absolutePath(), e);
	    return;
	}
	boolean removed = false;
	
	Enumeration original = originalChildren.enumeration();
	while (original.hasMoreElements()) {
	    Manager child = (Manager)original.nextElement();
	    if (findChild(child.name()) == null) {
		removed = true;
		if (verbose)
		    SystemManager.messageln("Class " + child.name() + " has been removed");
	    }
	}

	if (removed || child.numBrothers() + 1 != numChildren()) {
	    // need to sync database file
	    ((PackageManager)parent).updateSourcesFile();
	}
	originalChildren.dispose();
	Enumeration classes = children();
	while (classes.hasMoreElements()) {
	    ClassManager cm = (ClassManager)classes.nextElement();
	    cm.disposeChildren();
        }
    }

    /**
     * write class list onto PrintStream
     */
    void writeListOn(PrintStream ps) throws java.io.IOException {
	ps.println(path());
	for (Enumeration c = classes(); c.hasMoreElements(); ) {
	    ClassManager cm = (ClassManager)c.nextElement();
	    ps.println("\t" + cm.name());
	}
    }

    public Enumeration classes() {
	return children();
    }

    public int numClasses() {
	return numChildren();
    }

    public ClassManager findClass(String name) {
	return (ClassManager)findChild(name);
    }

    /**
     * return offset from beggining of this file
     */
    public int lineOffset(int line) {
	return line;
    }

    /**
     * return offset from beggining of this file(header)
     */
    public int headerLineOffset(int line) {
	if (line <= header.endLine()) {
	    return line - (int)header.startLine();
	} else {
	    return -1;
	}
    }

    /**
     * remove a class
     */
    protected Manager removeChild(Manager child) {
	if (super.removeChild(child) != null) return child;
	PackageManager pm = (PackageManager)parent();
	if (this.child == null) {
	    // this file has no class
	    pm.removeChild(this);
	} else {
	    pm.updateSourcesFile();
	}
	return null;
    }

    /**
     * I'm going to be removed
     */
    Manager removing() {
	File sourceFile = file();
	sourceFile.renameTo(backupFile(sourceFile));
	if (changedSources.containsKey(this)) {
	    changedSources.remove(this);
	}
	file = null;
	return null;
    }

    /**
     * rename the filename
     */
    public Manager rename(String newName) throws DejavaException {
	File newFile = new DejavaFile(directory(), newName); 
	if (newFile.exists()) {
	    throw new DejavaException("file already exists: " + newFile.getPath());
	}
	file().renameTo(newFile);
	name = newName;
	path = newName;
	file = newFile;
	((PackageManager)parent).updateSourcesFile();
	return this;
    }

    public void dispose() {
	//disposeDetails();
	super.dispose();
    }

    /**
     * dispose information for this source file
     */
    void disposeDetails() {
	if (header != null) {
	    header.dispose();
	    header = null;
	}
	if (trailer != null) {
	    trailer.dispose();
	    trailer = null;
	}
	// assume this source file has not been read yet
	lastModified = -1;
	disposeClassDetails();
	if (brother != null) {
	    ((SourceFileManager)brother).disposeDetails();
	}
    }

    private void disposeClassDetails() {
	if (child != null) {
	    ((ClassManager)child).disposeDetails();
	}
    }
}
