/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.sys;

import dejava.util.*;
import dejava.lang.ClassSpec;
import dejava.gui.Transcript;
import dejava.gui.Browser;
import dejava.gui.DejavaDialog;
import dejava.gui.DejavaFileDialog;
import java.io.*;
import java.util.*;
import java.awt.Frame;

/**
 * teikade system management class
 * it manages all projects defined in this teikade system
 */
public final class SystemManager {
    /**
     * welcome message
     */
    static final String welcome = DejavaConstants.welcome;

    /**
     * some constant names for files/directories
     */
    static final String PROFDIRNAME = ".dejava";
    static final String BEFORE_14   = ",projects";
    static final String PROJECTS    = ".projects";

    public static boolean onSolaris = false;
    public static boolean onWin95 = false;
    public static boolean onWinNT = false;
    public static boolean onWin32 = false;
    public static boolean onJDK1_0 = false;

    public static String DEJAVABIN;
    public static String CLASSPATH;
    public static String PATHSEPARATOR;

    static {
	String os = System.getProperty("os.name");
	if (os.equals("Solaris")) {
	    onSolaris = true;
	} else if (os.equals("Windows 95")) {
	    onWin95 = true;
	    onWin32 = true;
	} else if (os.equals("Windows NT")) {
	    onWinNT = true;
	    onWin32 = true;
	}
	if ("1.0".equals(System.getProperty("java.version"))) {
	    onJDK1_0 = true;
	}
	CLASSPATH = System.getProperty("java.class.path");
	PATHSEPARATOR = System.getProperty("path.separator");
	StringTokenizer pathes = new StringTokenizer(CLASSPATH, PATHSEPARATOR);
	String dejava = File.separator + "teikade" + File.separator;
	while (pathes.hasMoreTokens()) {
	    String path = pathes.nextToken();
	    if (path.indexOf(dejava) >= 0) {
		path = path + File.separator + ".." +
			      File.separator + "bin" +
			      File.separator + os.replace(' ', '_') +
			      File.separator;
		if (new File(path).exists()) {
		    DEJAVABIN = path;
		    break;
		}
	    }
	}
    }

    /**
     * File object for profile directory
     */
    private static File profdir;

    /**
     * system change list object
     */
    private static ChangeList changes;

    /**
     * system transcript window object: the default Frame for Dialogs
     */
    private static Transcript transcript;

    /**
     * linked list for projects managed by this SystemManager
     */
    private static ProjectManager projects;

    private static int debugLevel;

    /**
     * main routine for teikade
     */
    public static void main(String argv[]) {
	String dlProperty = System.getProperty("dejava.debug");
	debugLevel = dlProperty == null ? 0 : Integer.parseInt(dlProperty);

	transcript = Transcript.getTranscript();

	messageln("--- The system is now coming up. Wait a moment please ...---\n");
	try {
	    String profile = System.getProperty("dejava.profdir");
	    if (profile == null) {
		profile = System.getProperty("user.home") + File.separator + PROFDIRNAME;
	    }

	    profdir = new File(profile);
	    try {
		if (!profdir.exists()) {
		    if (profdir.mkdir()) {
			messageln("Created profile directory: " + profile);
		    } else {
			throw new java.io.IOException();
		    }
		}
	    } catch (java.io.IOException e) {
		System.err.println("Could not open profile directory. Giving up.");
		System.err.println("Profile directory was: " + profile);
		e.printStackTrace();
		System.exit(1);
	    }

	    readProjects(true);
	    File properties = new File(profdir, ".properties");
	    if (properties.exists()) {
		FileInputStream fis = new FileInputStream(properties);
		DejavaProperties.load(fis);
		fis.close();
	    }
	} catch (java.lang.Exception e) {
	    e.printStackTrace();
	    System.err.println("Exception is caught during start up. Exiting.");
	    System.exit(-1);
	}

	messageln("\n" + welcome);
	for (int i = 0; i < argv.length; i++) {
	    ProjectManager pm = findProject(argv[i]);
	    if (pm != null) {
		Browser.openOn(pm.getCodePath());
	    } else {
		beep();
		messageln("Specified project '" + argv[i] + "' is not found in projects. Check out project name.");
	    }
	}
    }

    /**
     * beep
     */
    public static void beep() {
	System.err.print("\007");
	System.err.flush();
    }

    public static void raiseTranscript() {
	transcript.show();
    }

    /**
     * append message to the transcript window
     */
    public static void message(String msg) {
	transcript.message(msg);
    }

    /**
     * append message to the transcript window
     */
    public static void message(InputStream is) {
	transcript.message(is);
    }

    /**
     * append message followed by a newline char
     */
    public static void messageln(String msg) {
	transcript.message(msg + "\n");
    }

    /**
     * append error message to the transcript window
     */
    public static void error(String msg) {
	beep();
	transcript.error(msg);
    }

    public static void errorln(String msg) {
	error(msg + "\n");
    }

    public static void exception(String msg, Exception e) {
	System.out.println("teikade caught exception: " + msg);
	e.printStackTrace();
    }

    public static void exception(Exception e) {
	errorln(e.getMessage());
    }

    public static void debug(int level, Object obj) {
	if (debugLevel >= level) {
	    debug(obj.toString());
	}
    }

    public static void debug(Object obj) {
	messageln(obj.toString());
    }

    public static int debugLevel() {
	return debugLevel;
    }

    /**
     * information
     */
    public static void information(String message) {
	DejavaDialog.information(getDefaultFrame(), message);
    }

    /**
     * information
     */
    public static void information(String message, String btnLabel) {
	DejavaDialog.information(getDefaultFrame(), message, btnLabel);
    }

    /**
     * confirm Ok or Cancel
     */
    public static boolean confirm(String message) {
        return confirm(message, DejavaDialog.OK, DejavaDialog.CANCEL);
    }

    /**
     * confirm <label> or Cancel
     */
    public static boolean confirm(String message, String label) {
        return confirm(message, label, DejavaDialog.CANCEL);
    }

    /**
     * confirm Ok or Cancel
     */
    public static boolean confirm(String message, String label1, String label2) {
        return DejavaDialog.confirm(getDefaultFrame(), message, label1, label2);
    }

    /**
     * request user input for String
     */
    public static String request(String message) {
        return DejavaDialog.request(getDefaultFrame(), message, "");
    }

    /**
     * request user input for String
     */
    public static String request(String message, String defaultString) {
        return DejavaDialog.request(getDefaultFrame(), message, defaultString);
    }

    /**
     * request file
     */
    public static String requestFile(String message, String dir, String file) {
        return DejavaFileDialog.request(getDefaultFrame(), message, dir, file);
    }

    public static String requestFileName(String message,
					 String directory, String file) {
        return DejavaFileDialog.requestName(getDefaultFrame(), message,
							       directory, file);
    }

    private static Frame getDefaultFrame() {
        return transcript.getDefaultFrame();
    }

    /**
     * get the Project Manager for given name
     */
    public static ProjectManager findProject(String name) {
	if (projects != null) {
	    return (ProjectManager)projects.find(name);
	} else {
	    return null;
	}
    }

    /**
     * get the File object for profile directory
     */
    public static File profileDirectory() {
	return profdir;
    }

   /**
     * get the File object for _projects file
     */
    public static File projectsFile() {
	return new DejavaFile(profileDirectory(), PROJECTS);
    }

   /**
     * get the File object for ,projects file
     */
    public static File before14projectsFile() {
	return new DejavaFile(profileDirectory(), BEFORE_14);
    }

   /**
     * convert ,project -> _project
     */
    public static void convertBefore14(File oldF, File newF) {
	oldF.renameTo(newF);
	// nothing to be done on contents
    }

   /**
     * get the File object for .changes file
     */
    public static File changesFile() {
	return new DejavaFile(profileDirectory(), ChangeList.CHANGES);
    }

   /**
     * get the File object for .changes file
     */
    public static File before14changesFile() {
	return new DejavaFile(profileDirectory(), ChangeList.BEFORE_14);
    }

   /**
     * get system ChangeList object
     */
    public static ChangeList changes() {
	if (changes == null) {
	    File chgf = changesFile();
	    File before14 = before14changesFile();
	    if (!chgf.exists() && before14.exists()) try {
		before14.renameTo(chgf);
	    } catch (Exception e) {
		SystemManager.information("Failed renaming ,changes -> .changes");
	    }
	    changes = new ChangeList(chgf);
	}
	return changes;
    }

    /**
     * read ,project file
     */
    static synchronized void readProjects(boolean verbose) {
	if (projects != null) {
	    disposeProjects();
	}
	File projsfile = projectsFile();
	if (!projsfile.exists()) {
	    File oldFile = before14projectsFile();
	    if (oldFile.exists()) {
		if (SystemManager.confirm("Projects file now renamed.\n" +
					  "Should I convert old one?",
					  "Please", "No, thanx")) {
		    convertBefore14(oldFile, projsfile);
		} else if (SystemManager.confirm("Exit teikade now?\n", "Yes", "No")) {
		    SystemManager.information("Okay, see you next time. bye.");
		    System.exit(0);
		}
	    }
	}
	try {
	    if (verbose) messageln("Reading projects file: " + projsfile.getPath());
	    String line;
	    String pr = null;
	    FileInputStream fis = new FileInputStream(projsfile);
	    DataInputStream dis = new DataInputStream(fis);
	    while ((line = dis.readLine()) != null) {
		int spc = line.indexOf(' ');
		if (spc < 0) {
		    spc = line.length();
		    line = line + " " + line;
		}
		pr = line.substring(0, spc);
		String path = line.substring(spc + 1);
		if (addProject(pr, path, false, false) != null) {
		    if (verbose) messageln("\t" + pr + " found on " + path);
		}
	    }
	    dis.close();
	    fis.close();
	    if (verbose) {
		int numProjects = numProjects();
		if (numProjects != 0) {
		    messageln("Found " + numProjects +
			" project" + (numProjects == 1 ? "." : "s."));
		} else {
		    messageln("No project found.");
		}
	    }
	} catch(java.io.FileNotFoundException e) {
	    messageln("No projects file exists.");
	} catch(java.io.IOException e) {
	    exception("Could not read projects file.", e);
	}
    }

    /**
     * update .project file
     */
    static synchronized void updateProjects() {
	File projsfile = projectsFile();
	try {
	    message("Updating projects file ...");
	    FileOutputStream fos = new FileOutputStream(projsfile);
	    PrintStream ps = new PrintStream(fos);
	    Enumeration en = allProjects();
	    while (en.hasMoreElements()) {
		ProjectManager pm = (ProjectManager)en.nextElement();
		ps.println(pm.name() + " " + pm.path());
	    }
	    ps.close();
	    fos.close();
	    messageln(" done.");
	} catch(java.io.IOException e) {
	    exception("Could not write projects file.", e);
	}
    }

    /**
     * internal project creation routine
     * does not create project directory and profile directory
     */
    private static ProjectManager addProject(String projname, String path,
					boolean rebuild, boolean writeOnFile) {
	if (findProject(projname) != null) {
	    // never fall in this block
	    errorln("PrintStream file consistency error: Project already exists: " + projname);
	    return null;
	}
	try {
	    ProjectManager pman = new ProjectManager(projname, path);
	    if (pman != null) {
		if (projects == null) {
		    projects = pman;
		} else {
		    projects.add(pman);
		}
		Transcript.addProjectToMenu(projname);
		if (rebuild) {
		    pman.rebuild();
		} else {
		    pman.readPackagesFile();
		}
		if (writeOnFile) {
		   /*
		    RandomAccessFile raf = new RandomAccessFile(projectsFile(), "rw");
		    raf.seek(raf.length());
		    raf.writeBytes(projname + " " + path + SystemManager.lf);
		    raf.close();
		    */
		    updateProjects();
		}
		return pman;
	    } else {
		information("Could not set up project " + projname + " on " + path);
		return null;
	    }
	} catch (java.io.IOException e) {
	    exception("SystemManager#addProject(String,String,boolean,boolean)", e);
	    return null;
	}
    }

    /**
     * add project entry
     */
    public static ProjectManager createProject(String projname, String path, boolean rebuild) {
	if (findProject(projname) != null) {
	    information("Project already exists: " + projname);
	    return null;
	}
	File projectDir = new File(path);
	if (!projectDir.exists()) {
	    boolean create = confirm("Directory " + path + " not found. Create it?", "Create");
	    if (!create) return null;
	    projectDir.mkdirs();
	}
	File profDir = new File(projectDir, PROFDIRNAME);
	if (!profDir.exists()) {
	    boolean create = confirm("Information about project should go under this directory?\n" +
				     "Or under your home directory?",
				     "Project directory", "Home directory" );
	    if (create) profDir.mkdir();
	}
	return addProject(projname, path, rebuild, true);
    }

    /**
     * set projects entry
     */
    public static void setProjects(ProjectManager pman) {
	projects = pman;
    }

    /**
     * get projects entry
     */
    public static ProjectManager getProjects() {
	return projects;
    }

    /**
     * enumerate all ProjectManagers in the system
     */
    public static Enumeration allProjects() {
	return new ManagerEnumerator(projects);
    }

    /**
     * enumerate all ProjectManager's names in the system
     */
    public static Enumeration allProjectNames() {
	return new ManagerEnumerator(projects, true);
    }

    /**
     * returns number of projects in the System
     */
    public static int numProjects() {
	if (projects != null) {
	    return projects.numBrothers() + 1;
	} else {
	    return 0;
	}
    }

    /**
     * dispose all ProjectManagers
     */
    private static void disposeProjects() {
	projects.dispose();
	projects = null;
    }

    /**
     * sync all projects
     */
    public static void sync() {
	for (Enumeration p = allProjects(); p.hasMoreElements(); ) {
	    ((ProjectManager)p.nextElement()).sync();
	}
    }
}
