/*
 * Copyright (c) 1995 PFU Limited.
 *	author Osamu Satoh
 */
package dejava.util;

import dejava.sys.*;
import java.io.*;
import java.util.*;

public class ChangeList {
    public static final String BEFORE_14 = ",changes";
    public static final String CHANGES   = ".changes";

    Changes list;
    File file;
    int last;
    int count;

    /**
     * the constructor
     */
    public ChangeList(String fn) {
	this(new File(fn));
    }

    /**
     * the constructor
     */
    public ChangeList(File f) {
	file = f;
	list = null;
	last = -1;
	count = 0;
    }

    public File file() {
	return file;
    }

    public int count() {
	return count;
    }

    /**
     * get code of specified changes
     */
    public String getCode(Changes chg) {
	byte buf[] = new byte[chg.length()];
	try {
	    RandomAccessFile raf = new RandomAccessFile(file, "r");
	    raf.seek(chg.startPoint);
	    raf.read(buf);
	    raf.close();
	    return new String(buf, 0);
	} catch (java.io.IOException e) {
	    SystemManager.exception("Could not get code", e);
	    return "";
	}
    }

    /**
     * enumerate all changes in sequencial order
     */
    public Enumeration elements() {
	return new ChangeListEnumeration(list);
    }

    /**
     * create new changes and add it to the list
     *
     *    last->+--------------------------+
     *		|reverse link(last - last')|
     *		+--------------------------+
     *		| changes length(end - pos)|
     *     pos->+--------------------------+
     *		|   cp.printString() + \n  |
     *		+--------------------------+
     *		|       action + \n        |
     *		+--------------------------+
     *		|      date/time + \n      |
     *   start->+--------------------------+
     *		|                          |
     *		|        code body         |
     *		|                          |
     *     end->+--------------------------+
     *		| reverse link (end - last)|
     *		+--------------------------+
     */
    public synchronized
    Changes addChanges(CodePath cp, String action, String code)
    throws java.io.IOException {
	RandomAccessFile raf = null;
	Changes newChg = null;
	try {
	    if (last == -1) {
		last = findLastReverseLink();
	    }
	    raf = new RandomAccessFile(file, "rw");
	    if (raf.length() == 0) {
		raf.writeBytes("0\n");
	    } else {
		raf.seek(raf.length());
	    }
	    Date date = new Date();
	    StringBuffer dateBuffer = new StringBuffer();
	    String tmp = "0" + date.getYear();
	    dateBuffer.append(tmp.substring(tmp.length() - 2)).append('/');
	    tmp = "0" + (date.getMonth() + 1);
	    dateBuffer.append(tmp.substring(tmp.length() - 2)).append('/');
	    tmp = "0" + date.getDate();
	    dateBuffer.append(tmp.substring(tmp.length() - 2)).append(' ');
	    tmp = "0" + date.getHours();
	    dateBuffer.append(tmp.substring(tmp.length() - 2)).append(':');
	    tmp = "0" + date.getMinutes();
	    dateBuffer.append(tmp.substring(tmp.length() - 2));
	    String dateString = dateBuffer.toString();
	    String header = cp.printString() + "\n" + action + "\n" + dateString + "\n";
	    String body = (code.endsWith("\n")) ? code : (code + "\n");
	    raf.writeBytes(new Integer(header.length() + body.length()).toString() + "\n");
	    raf.writeBytes(header);
	    int start = (int)raf.length();
	    raf.writeBytes(body);
	    int end = (int)raf.length();
	    raf.writeBytes(new Integer(end - last).toString() + "\n");
	    last = end;
	    newChg = new Changes(cp, action, dateString, start, end);
	    newChg.next = list;
	    list = newChg;
	} finally {
	    if (raf != null) raf.close();
	}
	count++;
	return newChg;
    }

    /**
     * create new changes and add it to the list
     */
    public Changes addChanges(CodePath cp, String code)
    throws java.io.IOException{
	return this.addChanges(cp, Changes.ACTION_CHANGED, code);
    }

    /**
     * enumerate all changes in sequencial order
     * read from the file
     */
    public Enumeration elementsFromFile(int maxcount, boolean reverse)
    throws java.io.IOException {
	if (reverse) {
		reverseReadFromFile(maxcount); 
	} else {
		readFromFile(maxcount);
	}
	return new ChangeListEnumeration(list);
    }

    /**
     * read the file and create change list
     */
    private void readFromFile(int maxcount) throws java.io.IOException {
	list = null;
	Changes lastChange = null;
	RandomAccessFile raf = new RandomAccessFile(file, "r");
	raf.seek(0);

	// discard first reverse link
	String line = raf.readLine();
	if (line == null) {
	    return;
	}
	while ((maxcount > 0) && ((line = raf.readLine()) != null)) {	// next change's total bytes
	    // get length of this change
	    int len = Integer.parseInt(line);

	    // read header information
	    int pos = (int)raf.getFilePointer();
	    String cp = raf.readLine();
	    String action = raf.readLine();
	    String date = raf.readLine();

	    // read body of changes
	    int start = (int)raf.getFilePointer();
	    byte buf[] = new byte[len - (start - pos)];
	    raf.read(buf);
	    int end = (int)raf.getFilePointer();

	    // create new Changes instance and add it to the list
	    Changes change = new Changes(cp, action, date, start, end);
	    if (lastChange == null) {
		list = change;
	    } else {
		lastChange.next = change;
	    }
	    lastChange = change;
	    change.next = null;
	    change.changes(this);

	    // discard reverse link
	    line = raf.readLine();

	    maxcount--;
	}
	raf.close();
    }

    /**
     * read the file and create change list reverse version
     */
    private void reverseReadFromFile(int maxcount) throws java.io.IOException {
	list = null;
	Changes lastChange = null;
	int end;
	if (last == -1) {
	    if ((end = findLastReverseLink()) == -1) return;
	} else {
	    end = last;
	}
	RandomAccessFile raf = new RandomAccessFile(file, "r");
	raf.seek(end);
	// get the reverse link
	int revlink = Integer.parseInt(raf.readLine());
	while (maxcount > 0 && revlink != 0) {
	    end = end - revlink;
	    raf.seek(end);
	    revlink = Integer.parseInt(raf.readLine());
	    int len = Integer.parseInt(raf.readLine());
	    int pos = (int)raf.getFilePointer();
	    String cp = raf.readLine();
	    String action = raf.readLine();
	    String date = raf.readLine();
	    int start = (int)raf.getFilePointer();
	    byte buf[] = new byte[len - (start - pos)];
	    raf.read(buf);
	    Changes change = new Changes(cp, action, date, start, pos + len);
	    if (list == null) {
		list = change;
	    } else {
		lastChange.next = change;
	    }
	    lastChange = change;
	    maxcount--;
	}
	raf.close();
    }

    /**
     * find out last reaverse link starting position
     */
    private int findLastReverseLink() throws java.io.IOException {
	RandomAccessFile raf;
	if (!file.exists()) {
	    raf = new RandomAccessFile(file, "rw");
	    raf.close();
	    return 0;
	}
	raf = new RandomAccessFile(file, "r");
	int end = (int)raf.length() - 1;
	while (end > 0) {
	    raf.seek(end - 1);
	    if (raf.read() == '\n') break;
	    --end;
	}
	raf.close();
	return end;
    }
}

class ChangeListEnumeration implements Enumeration {
    Changes current;

    /**
     * the constructor
     */
    ChangeListEnumeration(Changes c) {
	current = c;
    }

    /**
     * java.io.Enumeration stuffs
     */
    public boolean hasMoreElements() {
	return (current != null);
    }

    /**
     * java.io.Enumeration stuff
     */
    public Object nextElement() {
	Object ret = current;
	current = current.next;
	return ret;
    }
}
