/*
 * Copyright (c) 1995 PFU Limited.
 *      author Osamu Satoh
 */

import java.io.*;
import java.util.*;
import java.awt.*;
import dejava.gui.DejavaDialog;

public class Installer {
    private static final Frame frame = new Frame();
    private static final TextArea messages = new TextArea(20, 60);
    private static PrintStream log;
    private static int messageLength = 0;
    private static final String ulj_packages = "/usr/local/java/packages";
    private static final String osname = System.getProperty("os.name").replace(' ', '_');

    protected static String config = "config.list";
    protected static String copy = "copy.list";
    protected static String convert = "convert.list";
    protected static String finAll = "finmsg.all";
    protected static String finCnf = "finmsg.cnf";
    protected static String foption = "-f";	// option for specify property file
    protected final Properties props = new Properties(System.getProperties());
    protected int exitCode = 0;

    private static void message(String msg) {
	log.print(msg);
	messages.insertText(msg, messageLength);
	messageLength += msg.length();
    }

    private static void information(String msg) {
	DejavaDialog.information(frame, msg);
    }

    private static boolean confirm(String msg) {
	return DejavaDialog.confirm(frame, msg, "Ok", "Cancel");
    }

    private static boolean confirm(String msg, String okLabel, String cancelLabel) {
	return DejavaDialog.confirm(frame, msg, okLabel, cancelLabel);
    }

    private static String request(String msg, String init) {
	return DejavaDialog.request(frame, msg, init);
    }

    /**
     * Get String property
     */
    public static String getString(Properties props, String key, boolean trim) {
	try {
	    if (trim) {
	        return props.getProperty(key).trim();
	    } else {
	        return props.getProperty(key);
	    }
	} catch (NullPointerException e) {
	    return null;
	}
    }

    /*
     * Get parsed string property
     */
    public static String getParsedString(Properties props, String key, boolean trim) {
	String name = getString(props, key, true);
	if (name != null) {
	    StringBuffer buf = new StringBuffer();
	    int to = 0;
	    int from = 0;
	    while (from < name.length() &&
	        (from = name.indexOf('%', to) + 1) > 0) {
		if (from > to + 1) {
		    buf.append(name.substring(to, from - 1));
	        }
		to = name.indexOf('%', from);
		if (to < from) {
		    log.println("Warning: property syntax error: " + name);
		    to = from;
		    break;
		}
		String propKey = name.substring(from, to);
		String value = getParsedString(props, propKey, trim);
		if (value != null) {
		    buf.append(value);
		} else {
		    // we should do something, but what?
		}
		to = to + 1;
	    }
	    buf.append(name.substring(to));
	    char fs = File.separatorChar;
	    if (trim)
	        return buf.toString().replace('/', fs).trim();
	    else
	        return buf.toString().replace('/', fs);
	} else {
	    return name;
	}
    }

    public static void main(String args[]) {
	File logFile = new File("install.log");
	try {
	    log = new PrintStream(
			  new BufferedOutputStream(
			  new FileOutputStream(logFile)));
	    log.println("**** installation log ****");
	} catch (IOException e) {
	    System.out.println("Could not set up log file. Log messages are redirected to System.out.");
	    log = System.out;
	    log.println("**** installation log ****");
	}
	messages.setEditable(false);
	frame.add("Center", messages);
	frame.setFont(new Font("Courier", Font.PLAIN, 14));
	frame.setTitle("Package installer");
	frame.pack();
	frame.show();
	Installer installer = new Installer();
	try {
	    if (installer.parseArguments(args) &&
		installer.welcome() &&
		installer.preparations() &&
		installer.doInstall()) {
		information(installer.prop(finAll));
		log.println("\n" + installer.prop(finAll));
	    } else {
		message("\nInstallation aborted.\n");
	    }
	} catch (Exception e) {
	    information("Installation failed:\n" + e.getMessage());
	    log.println("\nInstallation failed:\n" + e.getMessage());
	    //e.printStackTrace();
	}
	log.close();
	System.exit(installer.exitCode);
    }

    protected Installer() {
	props.put("usage",
		"usage: java Installer [" + foption + " propertyfile]\n" +
		"\t" + foption + ": property filename(default: " + config + ")\n" +
		"  properties must contain at least following information:\n" +
		"     name: name of the package\n" +
		"  version: version of the package\n" +
		"     date: date/time when the package created on/at\n");
	props.put(finCnf, "\n%name%-%version% was succesfully configured.\n");
	props.put(finAll, "\n%name%-%version% was succesfully installed.\n");
	props.put("files.copy", copy);
	props.put("files.convert", convert);
	props.put("source.dir", System.getProperty("user.dir"));
	// props.put("install.dir", ulj_packages + "/%name%");
    }

    public String prop(String key) {
	return getParsedString(props, key, true);
    }

    protected void usage() {
	System.out.println(prop("usage"));
    }

    private boolean parseArguments(String args[])
    throws Exception {
	File configFile = new File("config." + osname);
	if (!configFile.exists()) {
	    configFile = new File(config);
	}
	if (args.length != 0) {
	    if (args[0].startsWith(foption)) {
		if (args[0].length() == 2) {
		    configFile = new File(args[1]);
		} else {
		    configFile = new File(args[0].substring(2));
		}
	    }
	}
	if (configFile.canRead()) {
	    message("Reading configuration file: " + configFile + " ...");
	    props.load(new FileInputStream(configFile));
	    message(" done.\n\n");
	} else if (args.length == 0) {
	    System.out.println(prop("usage"));
	    return false;
	} else {
	    information("Couldn't load configuration file: " + configFile);
	    return false;
	}

	String p = prop("require.os.name");
	if (p != null && p.indexOf(System.getProperty("os.name")) == -1) {
	     information("Operating system must be one of:\n" + p);
	     return false;
	}
	p = prop("require.os.arch");
	if (p != null && p.indexOf(System.getProperty("os.arch")) == -1) {
	     information("System architecture must be one of:\n" + p);
	     return false;
	}
	p = prop("require.java.version");
	if (p != null && p.indexOf(System.getProperty("java.version")) == -1) {
	     information("Java version must be one of:\n" + p);
	     return false;
	}
	p = prop("require.java.class.version");
	if (p != null && p.indexOf(System.getProperty("java.class.version")) == -1) {
	     information("Java class version must be one of:\n" + p);
	     return false;
	}
	return true;
    }

    protected boolean welcome() {
	message("Installing:\t" + prop("name") + "\n" +
		"   version:\t" + prop("version") + "\n" +
		"      date:\t" + prop("date") + "\n");
	return true;
    }

    public boolean preparations() throws Exception {
	message("configuration started:\n");
	String dir;
	dir = request("sources are located in ...", prop("source.dir"));
	if (dir == null) {
	     information("Installation aborted.");
	     return false;
	}
	props.put("source.dir", dir);
	message("  source.dir:\t" + dir + "\n");

	if (prop("install.dir") != null) {
	    dir = request("should be installed under ...", prop("install.dir"));
	    if (dir == null) {
	         information("Installation aborted.");
	         return false;
	    }
	    props.put("install.dir", dir);
	    message("  install.dir:\t" + dir + "\n");
	}
	if (askProperties() && 
	    confirm(prop(finCnf), "proceed", "abort")) {
	    message("configuration finished.\n\n");
	    message("installation started:\n");
	    return true;
	} else {
	    message("configuration aborted.\n");
	    return false;
	}
    }

    private boolean askProperties() {
	Enumeration keys = props.propertyNames();
	while (keys.hasMoreElements()) {
	    String key = (String)keys.nextElement();
	    if (key.startsWith("req.")) {
		String defaultValue = prop(key);
		key = key.substring(4);
		String msg = prop("msg." + key);
		if (msg == null || msg.length() == 0) {
		    information("Message string not found for property:" + key);
		    exitCode = -1;
		    return false;
		}
		String value = request(msg, defaultValue);
		if (value == null) return false;
		props.put(key, value);
		message("  " + key + ":\t" + value + "\n");
	    } else if (key.startsWith("sel.")) {
		String msg = prop(key);
		key = key.substring(4);
		String yes = prop("yes." + key);
		if (yes == null || yes.length() == 0) {
		    information("Selection item not found for property:" + key);
		    exitCode = -1;
		    return false;
		}
		int yesIndex = yes.indexOf(',');

		String no = prop("no." + key);
		if (no == null || no.length() == 0) {
		    information("Selection item not found for property:" + key);
		    exitCode = -1;
		    return false;
		}
		int noIndex = no.indexOf(',');

		boolean yn = confirm(msg,
					yes.substring(0, yesIndex),
					no.substring(0, noIndex));
		String value;
		if (yn) {
		    value = yes.substring(yesIndex + 1);
		} else {
		    value = no.substring(noIndex + 1);
		}
		props.put(key, value);
		message("  " + key + ":\t" + value + "\n");
	    } else if (key.startsWith("cnf.")) {
		String msg = prop(key);
		if (!confirm(msg)) return false;
	    }
	}
	return true;
    }

    /**
     * Override this method to support additional features
     */
    protected boolean doInstall()
    throws Exception {
	String prop;
	if ((prop = prop("files.copy")) != null) {	// files to be copied
	    Properties pf = propertiesFromFile(prop);
	    Enumeration files = pf.propertyNames();
	    while (files.hasMoreElements()) {
		String fn = (String)files.nextElement();
		String target = getString(pf, fn, true);
		props.put("_tmp_", fn);
		fn = getParsedString(props, "_tmp_", true);
		props.put("_tmp_", target);
		target = getParsedString(props, "_tmp_", true);
		installFile(fn, target, false);
	    }
	}
	if ((prop = prop("files.convert")) != null) {	// files to be converted
	    Properties pf = propertiesFromFile(prop);
	    Enumeration files = pf.propertyNames();
	    while (files.hasMoreElements()) {
		String fn = (String)files.nextElement();
		String target = getString(pf, fn, true);
		props.put("_tmp_", fn);
		fn = getParsedString(props, "_tmp_", true);
		props.put("_tmp_", target);
		target = getParsedString(props, "_tmp_", true);
		installFile(fn, target, true);
	    }
	}
	return true;
    }

    protected Properties propertiesFromFile(String fname)
    throws Exception {
	Properties p =  new Properties();
	p.load(new FileInputStream(new File(prop("source.dir"), fname)));
	return p;
    }

    /**
     * Copy source files to install directory
     */
    protected void installFile(String fname, String target, boolean convert)
    throws Exception {
	if (fname.equals(".")) {
	    fname = "";
	}
	File inf  = new File(prop("source.dir") + File.separator + fname);
	if (target.length() == 0) {
	    target = fname;
	}
	if (target.charAt(0) != File.separatorChar &&
	    (osname.equals("Solaris") || target.charAt(1) != ':')) {
	        target = prop("install.dir") + File.separator + target;
	}
	File outf = new File(target);
	if (!inf.canRead()) {
	    exitCode = -1;
	    throw new Exception("Could not read file: " + inf);
	} else if (outf.exists() && !outf.canWrite()) {
	    exitCode = -1;
	    throw new Exception("Could not write file: " + outf);
	}
	installFile(inf, outf, convert);
    }

    /**
     * Copy source files to install directory
     */
    protected void installFile(File inf, File outf, boolean convert)
    throws Exception {
	if (inf.isDirectory()) {
	    String files[] = inf.list();
	    if (!outf.exists() && !outf.mkdirs()) {
	        exitCode = -1;
	        throw new Exception("Could not make directory: " + outf);
	    }
	    for (int i = 0; i < files.length; i++) {
	 	File src = new File(inf, files[i]); 
	 	File dest = new File(outf, files[i]); 
		installFile(src, dest, convert);
	    }
	} else {
	    File parent = new File(outf.getParent());
	    if (!parent.exists() && !parent.mkdirs()) {
	        exitCode = -1;
	        throw new Exception("Could not make directory: " + parent);
	    }
	    if (convert) {
		message("converting " + inf.getName() + " ...");
		DataInputStream dis = new DataInputStream(
					    new BufferedInputStream(
					    new FileInputStream(inf)));
		PrintStream ps = new PrintStream(
					    new BufferedOutputStream(
					    new FileOutputStream(outf)));
		String line;
		try {
		    while ((line = dis.readLine()) != null) {
			props.put("_tmp_", line);
			ps.println(getParsedString(props, "_tmp_", false));
		    }
		} catch (IOException e) {
		    exitCode = -1;
		    throw new IOException("Could not write file: "
					    + outf);
		}
	        dis.close();
		ps.close();
		message(" done.\n");
	    } else {
		message("copying " + inf.getName() + " ...");
		BufferedInputStream bis = new BufferedInputStream(
					    new FileInputStream(inf));
		BufferedOutputStream bos = new BufferedOutputStream(
					    new FileOutputStream(outf));
		byte buf[] = new byte[1024 * 8];
		int len;
		while ((len = bis.read(buf) ) > 0) try {
		    bos.write(buf, 0, len);
		} catch (IOException e) {
		    exitCode = -1;
		    throw new IOException("Could not write file: " + outf);
		}
		bis.close();
		bos.close();
		message(" done.\n");
	    }
        }
    }
}
