/* EXTERNAL(format).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   95-03-29  created (extracted from formats.i)
*/

#include <assert.h>
#include <string.h>
#include "external.h"
#include "mylib.h"
#include "header.h"

bool ExternalEntry::aWriteArrows = FALSE;

void ExternalEntry::convertSpecialFields(void)
{
    // converts "special" fields to (extra) comments:
    //
    //   - part
    //   - missing panels
    //
    // also clears "original pages" field if not needed

    switch(field(ePart).delimitedString()[0])
    {
    case ' ':
    case '\0':
        break;
    case 'x':
        field(eOriginalPages).clear();
        break; // NO mentioning of original pages
    case 'f':
        comment().append(" [final part]");
        field(eOriginalPages).clear();
        break;
    default:
        comment().append(" [part:");
        comment().append(field(ePart).delimitedString());
        comment().append("]");
        field(eOriginalPages).clear();
    }
    field(ePart).clear();

    switch (field(eMissingPanels).delimitedString()[0])
    {
    case 'r':
        comment().append(" [one row missing]");
        break;
    case '1':
    case 'e':
        comment().append(" [1 panel missing]");
        break;
    case ' ':
    case '\0':
        break;
    default:
        comment().append(" [");
        comment().append(field(eMissingPanels).delimitedString());
        comment().append(" panels missing]");
    }
    field(eMissingPanels).clear();
}


void ExternalEntry::fillExplanations(ExternalEntry & p2)
{
    // note that the lengths of some fields are restricted, which is
    // not checked here!
    sStoryCode.pointTo("Story");
    p2.sStoryCode.pointTo("code");
    sEntryCode.pointTo("Issue");
    sPages.set("Pg");
    sPageLayout = "L";

    switch (aFormat)
    {
    case eYMorZTFormat:
    case eNewspaperFormat:
        field(eDate    ).set("Date");
        field(eEndDate ).set("End-date");
        break;
    case eCountryFormat:
    case eWesternStoryFormat:
    case eDBIFormat:
    case eWesternDBIFormat:
        break; // don't print dates, they will only be comments
    case eBarksFormat:
        field(eDate).set("Date");
        break;
    default:
        assert(FALSE);
    }

    if (aFormat == eBarksFormat)
    {
        // a TRICK to make sure the fields are not printed!
        field   (eWriter ).set("CB");
        p2.field(eWriter ).set("CB");
        field   (eArtist ).set("CB");
        p2.field(eArtist ).set("CB");
    }
    else
    {
        field   (ePlotter).set("Plo");
        p2.field(ePlotter).set("ttr");
        field   (eWriter ).set("Wri");
        p2.field(eWriter ).set("ter");
        field   (eArtist ).set("Art");
        p2.field(eArtist ).set("ist");
        field   (eInker  ).set("Ink");
        p2.field(eInker  ).set("er");
    }
    field   (eHero   ).set("He-");
    p2.field(eHero   ).set("ro");

    sTitlePrefix = "Tag";
    aTitle.pointTo("Title or description");
}


void ExternalEntry::scanShlFields(MyDelimitedInputString & pString)
{
    // reads all fields indicated by [...].
    // They should be ShortList fields or Comments, because
    // ReprintFields should not occur in the external format

    PointerIntoInputString lCommentPointer;
    PointerIntoInputString lPrefix;

    while (pString.getPrefixedComment(lPrefix, PREFIXlength, lCommentPointer))
    {
        ShortListFieldIndex s(lPrefix);
        if (s.valid())
        {
            field(s).insert(lCommentPointer);
        }
        else
        {
            StoryCommentFieldIndex scf(lPrefix);
            if (scf.valid())
            {
                storyComment().append(lCommentPointer);
            }
            else
            {
                // 'repair' prefix+contents
                comment().append(" [");
                if (lPrefix[0] != '\0')
                {
                    static char lErrorText[80];  // ** no nice code
                    sprintf(lErrorText, "Prefix (%s) unknown", lPrefix.zeroString());
                    aErrorText = lErrorText;
                    comment().append(lPrefix);
                    comment().append(":");
                }
                comment().append(lCommentPointer);
                comment().append("]");
            }
        }
    }
}


void ExternalEntry::scanCommon( MyFixedPosInputString & pString,
                                int pPlotterPosition,
                                bool pCodesOnly)
{
    // reads title and all fields indicated by [...].
    // Title always starts at position pTitlePosition, or after
    // a '#' character. So in the 'right' part of a split input string

    if (pString.empty())
    {
        return;
    }

    pString.split(cEarlyTitleChar, pPlotterPosition + 5 * PERSONlength);

    if (pCodesOnly)
    {
        return; // the split WAS necessary, the rest isn't
    }

    PointerIntoInputString lPrefix;
    (void) pString.getPrefix(lPrefix, TITLEPREFIXlength, '[');
    sTitlePrefix = lPrefix; // calling operator=
    pString.getDelimitedPointer(aTitle, '[');
    scanShlFields(pString);

    // scan creators and title hero, if not already found
    // as a comment (!)
    // They are always on the same place

    if (field(ePlotter).empty())
    {
        PointerIntoInputString lPerson;
        pString.scanFixedPosInBuffer(pPlotterPosition, PERSONlength, lPerson);
        field(ePlotter).insert(lPerson);
    }
    if (field(eWriter).empty())
    {
        PointerIntoInputString lPerson;
        pString.scanFixedPosInBuffer(pPlotterPosition + PERSONlength,
                   PERSONlength, lPerson);
        field(eWriter).insert(lPerson);
    }
    if (field(eArtist).empty())
    {
        PointerIntoInputString lPerson;
        pString.scanFixedPosInBuffer(pPlotterPosition + 2 * PERSONlength,
                   PERSONlength, lPerson);
        field(eArtist).insert(lPerson);
    }
    if (field(eInker).empty())
    {
        PointerIntoInputString lPerson;
        pString.scanFixedPosInBuffer(pPlotterPosition + 3 * PERSONlength,
                   PERSONlength, lPerson);
        field(eInker).insert(lPerson);
    }
    if (field(eHero).empty())
    {
        PointerIntoInputString lPerson;
        pString.scanFixedPosInBuffer(pPlotterPosition + 4 * PERSONlength,
                   PERSONlength, lPerson);
        field(eHero).insert(lPerson);
    }
}


void ExternalEntry::putPersonText(MyExternalOutputFile & pFile, ShortListFieldEnum pIndex)
{
    // write a Person in its fixed column, or:
    //   - spaces if no person available
    //   - a reference if there is more than one person
    //   - a reference if the name of the person is too long
    // the 'reference' is either "" or a -> if we want to: just change
    // the constant cArrow
    //
    // the Person is cleared if it is written (so we won't print it again as
    // a comment)

    const char * cArrow = "";
    if (aWriteArrows)
    {
        cArrow = "->";
    }

    const char * lPersonPtr = "";
    bool lPersonPrinted = TRUE;

    switch (field(pIndex).numberOfElements())
    {
    case 0:
        lPersonPrinted = FALSE;
        break;
    case 1:
        lPersonPtr = field(pIndex).element(0);
        if (strlen(lPersonPtr) == PERSONlength)
        {
            switch (lPersonPtr[PERSONlength - 1])
            {
            case '?':
            case ')':
            case '+':
                break;
            default:
                lPersonPtr = cArrow;
                lPersonPrinted = FALSE;
            }
        }
        else if (strlen(lPersonPtr) > PERSONlength)
        {
            lPersonPtr = cArrow;
            lPersonPrinted = FALSE;
        }
        break;
    default:
        lPersonPtr = cArrow;
        lPersonPrinted = FALSE;
    }
    pFile.printF("%-*s", PERSONlength, lPersonPtr);
    if (lPersonPrinted)
    {
        field(pIndex).clear();
    }
}


void ExternalEntry::putCommon(MyExternalOutputFile & pFile)
{
    // writes Hero, Title, ShortList fields, and comment
    // this routine should be called as the LAST put-routine: other
    // routines can empty the ShortList fields that should not be written.

    // unlike scanCommon, the Creators are not written here
    //   (because we sometimes don't want the Plotter)

    bool lSpace = TRUE;

    putPersonText(pFile, eHero);
    if (aTitle.empty())
    {
        lSpace = FALSE;
        // no space before the first comment
    }
    else
    {
        if (sTitlePrefix.theString()[0] != '\0')
        {
            pFile.putS(sTitlePrefix.theString());
            pFile.putS(":");
        }
        pFile.putS(aTitle);
    }

    ShortListFieldIndex s;
    while (s.next())
    {
        if (! field(s).empty())
        {
            if (lSpace)
            {
                pFile.putS(" ");
            }
            lSpace = TRUE;
            pFile.printF("[%s:%s]", s.prefix(),
                           field(s).delimitedString());
        }
    }

    if (! comment().empty())
    {
        if (lSpace)
        {
            pFile.putS(" ");
            lSpace = TRUE;
        }
        assert(comment().element()[0] == ' ');
        assert(comment().element()[1] == '[');
        pFile.putS(comment().element() + 1);
        // comment always starts with a space, and already has the []...
    }

    if (! storyComment().empty())
    {
        if (lSpace)
        {
            pFile.putS(" ");
        }
        StoryCommentFieldIndex scf;
        pFile.printF("[%s:%s]",
                     StoryCommentFieldIndex::thePrefix(),
                     storyComment().element());
    }
}


void ExternalEntry::scanStoryCountry(
    MyFixedPosInputString & pString, bool pCodesOnly)
{
    scanCommon (pString, 19, pCodesOnly);
    pString.scanFixedPosInBuffer (0, STORYCODElength, sStoryCode);

    if (pCodesOnly)
    {
        return;
    }

    if (field(ePart).empty())
    {
        field(ePart).setFixedPosString (pString, 12, 1);
    }
    if (field(eMissingPanels).empty())
    {
        field(eMissingPanels).setFixedPosString (pString, 13, 1);
    }
    pString.scanFixedPos (14, sPages);
    pString.scanFixedPos (17, sPageLayout);
}


void ExternalEntry::putStoryCountry(MyExternalOutputFile & pFile)
{
    pFile.putSAligned(sStoryCode, STORYCODElength);

    const char * lPart = field(ePart).element(0);
    if (strlen(lPart) <= 1)
    {
        pFile.printF("%-1s", lPart);
        field(ePart).clear();
    }
    pFile.printF("%-1s%s%c%-1s",
        field(eMissingPanels).delimitedString(),
        sPages.theString(),
        sPageLayout.theChar(),
        aUnsolved);

    field(eMissingPanels).clear();

    putPersonText (pFile, ePlotter);
    putPersonText (pFile, eWriter);
    putPersonText (pFile, eArtist);
    putPersonText (pFile, eInker);
    putCommon     (pFile);
}


void ExternalEntry::scanStoryWestern(
    MyFixedPosInputString & pString, bool pCodesOnly)
{
    scanCommon (pString, 28, pCodesOnly);
    pString.scanFixedPosInBuffer(0, STORYCODElength, sStoryCode);

    if (pCodesOnly)
    {
        return;
    }

    if (field(ePart).empty())
    {
        field(ePart).setFixedPosString (pString, 12, 1);
    }
    if (field(eMissingPanels).empty())
    {
        field(eMissingPanels).setFixedPosString (pString, 13, 1);
    }
    pString.scanFixedPos (14, sPages);
    pString.scanFixedPos (17, sPageLayout);
    if (field(eDate).empty())
    {
        field(eDate).setFixedPosString (pString, 19, DATElength);
    }
}


void ExternalEntry::putStoryWestern(MyExternalOutputFile & pFile)
{
    pFile.putSAligned(sStoryCode, STORYCODElength);
    const char * lPart = field(ePart).element(0);
    if (strlen(lPart) <= 1)
    {
        pFile.printF("%-1s", lPart);
        field(ePart).clear();
    }
    pFile.printF("%1s%s%c%1s",
        field(eMissingPanels).delimitedString(),
        sPages.theString(),
        sPageLayout.theChar(),
        aUnsolved);
    field(eMissingPanels).clear();

    pFile.printF("%-*s ", STORYCODElength, field(eDate).delimitedString());
    field(eDate).clear();

    putPersonText (pFile, eWriter);
    putPersonText (pFile, eArtist);
    putPersonText (pFile, eInker);
    putCommon     (pFile);
}


void ExternalEntry::scanStoryYMorZT(
    MyFixedPosInputString & pString, bool pCodesOnly)
{
    scanCommon(pString, 28, pCodesOnly);

    pString.scanFixedPosInBuffer(0, 6, sStoryCode);
    // less room reserved for storycode!

    if (pCodesOnly)
    {
        return;
    }

    if (field(eDate).empty())
    {
        field(eDate).setFixedPosString (pString, 9, DATElength);
    }
    if (field(eEndDate).empty())
    {
        field(eEndDate).setFixedPosString (pString, 19, DATElength);
    }
}


void ExternalEntry::putStoryYMorZT(MyExternalOutputFile & pFile)
{
    pFile.putSAligned(sStoryCode, 6);
    pFile.printF(" %1s %-*s  %-*s ",
        aUnsolved,
        DATElength, field(eDate).delimitedString(),
        DATElength, field(eEndDate).delimitedString());
    field(eDate).clear();
    field(eEndDate).clear();

    putPersonText (pFile, ePlotter);
    putPersonText (pFile, eWriter);
    putPersonText (pFile, eArtist);
    putPersonText (pFile, eInker);
    putCommon     (pFile);
}


void ExternalEntry::scanStoryNewspaper(
    MyFixedPosInputString & pString, bool pCodesOnly)
{
    scanCommon(pString, 21, pCodesOnly);

    pString.scanFixedPosInBuffer(0, 11, sStoryCode);
    // field(eDate).setFixedPosString (pString, 3, DATElength);
    // date is part of story code, we don't put it in separately!

    if (pCodesOnly)
    {
        return;
    }

    if (field(eEndDate).empty())
    {
        field(eEndDate).setFixedPosString (pString, 12, DATElength);
    }
}


void ExternalEntry::putStoryNewspaper(MyExternalOutputFile & pFile)
{
    pFile.putSAligned(sStoryCode, 11);
    pFile.printF("%1s%-*s ",
        aUnsolved,
        DATElength, field(eEndDate).delimitedString());

    field(eDate).clear(); // is part of story code
    field(eEndDate).clear();

    putPersonText (pFile, ePlotter);
    putPersonText (pFile, eWriter);
    putPersonText (pFile, eArtist);
    putPersonText (pFile, eInker);
    putCommon     (pFile);
}


const struct
{
    const char * sStart;
    int sColumn;
} cReprintTable[] = // ** for now only Dutch Barks issues (!)
{ // this table is scanned from top to bottom. That's why an entry like
  // "71" should come *before* "7".
  { "5", 0 },
  { "6", 0 },
  { "X", 0 }, // 'X' for paintings in DD Extra (CB!)
  { "70", 0 },
  { "71", 0 },
  { "72", 0 },
  { "AV", 0 },
  { "PO", 0 },
  { "7", 1 },
  { "8", 1 },
  { "9", 1 },
  { "BV", 2 },
  { "OD", 2 },
  { "KV", 3 },
  { 0, 0 }
};


void ExternalEntry::writeToTheRightColumn(
    const char * pIssue,
    ENTRYCODEtype pColumn[],
    Comment & pComment)
{
    int lDest = 4;

    for (int i = 0; cReprintTable[i].sStart != 0; i++)
    {
        if (strncmp( cReprintTable[i].sStart, pIssue,
                     strlen(cReprintTable[i].sStart)) == 0)
        {
            lDest = cReprintTable[i].sColumn;
            break; // EXIT FOR
        }
    }

    if (pColumn[lDest][0] == '\0')
    {
        strcpy(pColumn[lDest], pIssue);
    }
    else if (pColumn[4][0] == '\0')
    {
        strcpy(pColumn[4], pIssue);
    }
    else
    {
        // room already occupied
        pComment.append(" [also in:");
        pComment.append(pIssue);
        pComment.append("]");
        MyLib::log("No room for %s", pIssue);
    }
}


void ExternalEntry::writeReprintsInColumns(
    MyExternalOutputFile & pOutputString,
    const char * pEntryCode,
    ShortList & pReprints,
    Comment & pComment)
{
    const int cMaxColumns = 5;
    ENTRYCODEtype lColumn[cMaxColumns];
    int i;

    for (i = 0; i < cMaxColumns; i++)
    {
        lColumn[i][0] = '\0';
    }

    writeToTheRightColumn(pEntryCode, lColumn, pComment);
    int count = pReprints.numberOfElements();
    for (i = 0; i < count; i++)
    {
        writeToTheRightColumn(pReprints.element(i), lColumn, pComment);
    }

    for (i = 0; i < cMaxColumns; i++)
    {
        pOutputString.printF("%-*s", ENTRYCODElength - 3, lColumn[i]);
        // "- 3" because ENTRYCODElength is much wider than we use in
        // the Dutch index....
    }

}


void ExternalEntry::putStoryBarks(
    MyExternalOutputFile & pFile,
    bool pWriteReprintsInColumns)
{
    // only meant for CB output: date column, no creator columns
    // (reprints in columns).
    const char * cSelector = "CB";

    pFile.putSAligned(sStoryCode, STORYCODElength);
    pFile.printF(" %s%c ",
        sPages.theString(),
        sPageLayout.theChar());
    pFile.printF("%-*s    ", DATElength, field(eDate).delimitedString());

    if (pWriteReprintsInColumns)
    {
        ENTRYCODEtype lBuffer;
        writeReprintsInColumns( pFile,
                                cleanedEntryCode(lBuffer),
                                reprint(eNetherlandsCountry), // !!
                                comment());
    }

    pFile.printF("%-*s ", PERSONlength - 1,
        field(eHero).delimitedString());
        // if that's too wide: who cares.

    if (! title().empty())
    {
        if (sTitlePrefix.theString()[0] != '\0')
        {
            pFile.printF("%s:", sTitlePrefix.theString());
        }
        pFile.putS(title());
    }

    if (! comment().empty())
    {
        pFile.putS(comment().element());
        // comment always starts with a space, and already has the []...
    }

    if (! storyComment().empty())
    {
        pFile.printF(" [%s:%s]",
                     StoryCommentFieldIndex::thePrefix(),
                     storyComment().element());
    }

    ShortListFieldIndex s;
    while (s.next())
    {
        switch (s.number())
        {
        case eWriter: // later, if not THE creator
        case eArtist: // later, if not THE creator
        case eHero:   // already printed above
        case eDate:   // already printed above
            break;
        default:
            if (field(s).numberOfElements() > 0)
            {
                pFile.printF(" [%s:%s]", s.prefix(),
                               field(s).delimitedString());
            }
        }
    }

    switch (field(eWriter).numberOfElements())
    {
    case 0:
        pFile.putS(" [writ:?]");
        break;
    case 1:
        {
            const char * lWriter = field(eWriter).element(0);
            if (strcmp(lWriter, cSelector) != 0)
            {
                pFile.printF(" [writ:%s]", lWriter);
            }
        }
        break;
    default:
        pFile.printF(" [writ:%s]",
                      field(eWriter).delimitedString());
    }

    switch (field(eArtist).numberOfElements())
    {
    case 0:
        pFile.putS(" [art:?]");
        break;
    case 1:
        {
            const char * lArtist = field(eArtist).element(0);
            if (strcmp(lArtist, cSelector) != 0)
            {
                pFile.printF(" [art:%s]", lArtist);
            }
        }
        break;
    default:
        pFile.printF(" [art:%s]",
                      field(eArtist).delimitedString());
    }
}


void ExternalEntry::scanEntryWestern(
    MyFixedPosInputString & pString, bool pCodesOnly)
{
    scanCommon(pString, 28, pCodesOnly);

    pString.scanFixedPosInBuffer (2, ENTRYCODElength, sEntryCode);
    pString.scanFixedPosInBuffer (19, STORYCODElength, sStoryCode);

    if (pCodesOnly)
    {
        return;
    }

    if (field(ePart).empty())
    {
        field(ePart).setFixedPosString (pString, 12, 1);
    }
    if (field(eMissingPanels).empty())
    {
        field(eMissingPanels).setFixedPosString (pString, 13, 1);
    }
    pString.scanFixedPos (14, sPages);
    pString.scanFixedPos (17, sPageLayout);

    // Western stories never have a Plotter entry! So overrule it
    // The story code uses the space of the plotter
    field(ePlotter).clear();
}


void ExternalEntry::scanEntryFakeWestern(MyFixedPosInputString & pString)
{
    pString.scanFixedPosInBuffer (0, STORYCODElength, sStoryCode);
    pString.scanFixedPosInBuffer (2, ENTRYCODElength, sEntryCode);
}


void ExternalEntry::putEntryWestern(MyExternalOutputFile & pFile)
{
    bool lOriginalStory = sEntryCode.plusWIs(sStoryCode);
    if (lOriginalStory)
    {
        pFile.printF("W ");
    }
    else // reprint
    {
        pFile.printF("r ");
    }

    pFile.putSAligned(sEntryCode, ENTRYCODElength);
    const char * lPart = field(ePart).element(0);
    if (strlen(lPart) <= 1)
    {
        pFile.printF("%-1s", lPart);
        field(ePart).clear();
    }
    pFile.printF("%1s%s%c%1s",
        field(eMissingPanels).delimitedString(),
        sPages.theString(),
        sPageLayout.theChar(),
        aUnsolved);
    field(eMissingPanels).clear();

    if (lOriginalStory)
    {
        // no need to print story code twice, print date in stead
        pFile.printF("%-*s ", STORYCODElength, field(eDate).delimitedString());
        field(eDate).clear();
    }
    else
    {
        pFile.putSAligned(sStoryCode, STORYCODElength);
        pFile.putS(" ");
    }

    putPersonText (pFile, eWriter);
    putPersonText (pFile, eArtist);
    putPersonText (pFile, eInker);
    putCommon     (pFile);
}


void ExternalEntry::scanEntryCountry(
    MyFixedPosInputString & pString, bool pCodesOnly)
{
    scanCommon(pString, 28, pCodesOnly);

    pString.scanFixedPosInBuffer(0, 9, sEntryCode);
    // EntryCode is less than in Western files!

    pString.scanFixedPosInBuffer(9, STORYCODElength, sStoryCode);

    if (pCodesOnly)
    {
        return;
    }

    if (field(ePart).empty())
    {
        field(ePart).setFixedPosString (pString, 21, 1);
    }
    if (field(eMissingPanels).empty())
    {
        field(eMissingPanels).setFixedPosString (pString, 22, 1);
    }
    pString.scanFixedPos (23, sPages);
    pString.scanFixedPos (26, sPageLayout);
}


void ExternalEntry::putEntryCountry(MyExternalOutputFile & pFile)
{
    pFile.putSAligned(sEntryCode, 9);
    pFile.putSAligned(sStoryCode, STORYCODElength);
    const char * lPart = field(ePart).element(0);
    if (strlen(lPart) <= 1)
    {
        pFile.printF("%-1s", lPart);
        field(ePart).clear();
    }
    pFile.printF("%1s%s%c%1s",
        field(eMissingPanels).delimitedString(),
        sPages.theString(),
        sPageLayout.theChar(),
        aUnsolved);
    field(eMissingPanels).clear();

    putPersonText (pFile, ePlotter);
    putPersonText (pFile, eWriter);
    putPersonText (pFile, eArtist);
    putPersonText (pFile, eInker);
    putCommon     (pFile);
}


void ExternalEntry::scanExternal(MyFixedPosInputString & pString,
                                 bool pCodesOnly)
{
    switch (aFormat)
    {
    // story files:
    case eCountryFormat:
        scanStoryCountry(pString, pCodesOnly);
        break;
    case eWesternStoryFormat:
        assert (pString.start() != 'r');
        scanStoryWestern(pString, pCodesOnly);
        break;
    case eYMorZTFormat:
        scanStoryYMorZT(pString, pCodesOnly);
        break;
    case eNewspaperFormat:
        scanStoryNewspaper(pString, pCodesOnly);
        break;

    // entry files:
    case eDBIFormat:
        scanEntryCountry(pString, pCodesOnly);
        break;
    case eWesternDBIFormat:
        if (pString.start() == 'r') // a reprint line
        {
            scanEntryWestern(pString, pCodesOnly);
        }
        else
        {
            // create a "reprint" line from the story line
            scanEntryFakeWestern(pString);
        }
        break;

    default:
        // eStoryBarks only for output
        assert(FALSE);
    }
}


void ExternalEntry::putReprintsAsComments(MyExternalOutputFile & pFile,
                                          const CountrySet & pCountrySet)
{
    CountryIndex c;
    bool lAnyReprint = FALSE;
    while (c.next())
    {
        if (pCountrySet.includes(c))
        {
            if (!reprint(c).empty())
            {
                pFile.printF(" [%s:%s]", c.reprintPrefix(),
                             reprint(c).delimitedString());
                lAnyReprint = TRUE;
            }
        }
    }
    if (!lAnyReprint)
    {
//        pFile.printF(" [@@no reprints]"); // **** handy for testing
    }
}


void ExternalEntry::putReprintsOnSeparateLines(
    MyFormattedOutputFile & pFile,
    const CountrySet & pCountrySet)
{
    CountryIndex c;
    while (c.next())
    {
        if (pCountrySet.includes(c))
        {
            if (!reprint(c).empty())
            {
                pFile.forceNewLine();
                pFile.printF("%-3s: %s", c.abbreviation(),
                             reprint(c).delimitedString());
            }
        }
    }
}


void ExternalEntry::putDescriptionLayout(MyExternalOutputFile & pFile)
{
    pFile.printF("#");
    pFile.putS(sStoryCode);
    pFile.printLine();
    pFile.printF("<!--");
    pFile.printF(" %s%c ",
        sPages.theString(),
        sPageLayout.theChar());

    // we're not interested in the following fields:
    field(ePart).clear();
    field(eMissingPanels).clear();
    field(ePlotter).clear();
    field(eWriter).clear();
    field(eArtist).clear();
    field(eInker).clear();
    field(eDate).clear();
    field(eEndDate).clear();

    putCommon(pFile);
    pFile.printF(">");

    // headers and footers are still standard-output; they will have to be **
    // deleted...
}


void ExternalEntry::putExternal(MyExternalOutputFile & pFile)
{
    switch (aFormat)
    {
    // story files:
    case eCountryFormat:
        putStoryCountry(pFile);
        break;
    case eWesternStoryFormat:
        putStoryWestern(pFile); // only needed in the UTILITY program...
        break;
    case eYMorZTFormat:
        putStoryYMorZT(pFile);
        break;
    case eNewspaperFormat:
        putStoryNewspaper(pFile);
        break;
    case eBarksFormat:
        putStoryBarks(pFile, FALSE);
        break;

    // entry files:
    case eDBIFormat:
        putEntryCountry(pFile);
        break;
    case eWesternDBIFormat:
        putEntryWestern(pFile);
        break;

    default:
        assert(FALSE);
    }
}


int ExternalEntry::indentation(Format pFormat)
{
    switch (pFormat)
    {
    case eCountryFormat:      return 40;
    case eWesternStoryFormat: return 0;  // not in use!
    case eYMorZTFormat:       return 49;
    case eNewspaperFormat:    return 42;
    case eBarksFormat:        return 40;

    case eDBIFormat:          return 49;
    case eWesternDBIFormat:   return 49;
    default:
        assert(FALSE);
    }
    return 0;
}


void ExternalEntry::setArrow(bool p)
{
    aWriteArrows = p;
}