/* GATHER.CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   Object to gather all data of each story.
   Input is a bunch of files in intermediate format, each sorted by
   story code;
   Output is:
   - one big entry file in intermediate format, ready to be
     re-sorted by entry code, and post-formatted for the
     end output;
   - one big story file in intermediate format, ready to be post-formatted.

   95-10-13  created (some functions from DIZNILIS)
   97-04-28  takeBestTitle: new parameter pAllowedCountries
*/

#include "gather.h"

void GatherStoryWithEntries::completeEntries(const CountrySet & pAllowedTitleCountries)
{
    // complete story information mostly comes from the story
    // entry. Two things can come from the issue entries:
    // - reprint information
    // - the right title

    if (theStoryEntry() != 0)
    {
        storyInformationInIssueEntries();
        titlesInAllEntries(pAllowedTitleCountries);
        reprintsAndXRefsInAllEntries();
    }
}


void GatherStoryWithEntries::doit(const CountrySet & pAllowedTitleCountries)
{
    MyInternalOutputFile lStoryOutputFile (FileName("stories", "ins"));
    CountryFileArray lEntryOutputFileArray (cStep3Extension);

    STORYCODEtype lLastStoryCode;
    strcpy(lLastStoryCode, "UNEQ");

    GatherStoryWithEntries * lStoryWithEntries = new GatherStoryWithEntries;
    assert(lStoryWithEntries != 0);

    StoryFileIndex lIndex;
    while (lIndex.next())
    {
        MyInputFile lInputFile (FileName(lIndex.prefix(), cStep2Extension));
        while (TRUE) // break in loop
        {
            // read the next entry
            MyDelimitedInputString * lInputString =
                new MyDelimitedInputString;
            assert(lInputString != 0);
            bool lEOF = (lInputString->read(lInputFile) == eEndOfFile);

            // some SORT programs produce an empty line at the
            // end of the file, that's why also test on empty line
            if (lEOF || lInputString->empty())
            {
                delete lInputString;
                break; // end-of-file   EXIT INNER WHILE LOOP
            }
            else
            {
                InternalEntry * lEntry = new InternalEntry;
                assert(lEntry != 0);
                lEntry->scan(*lInputString);

                if (lEntry->sStoryCode != lLastStoryCode)
                {
                    lStoryWithEntries->completeEntries(pAllowedTitleCountries);
                    lStoryWithEntries->write(
                        lStoryOutputFile,
                        lEntryOutputFileArray);

                    delete lStoryWithEntries;
                    lStoryWithEntries = new GatherStoryWithEntries;
                    assert(lStoryWithEntries != 0);
                    lEntry->sStoryCode.copyToChars(lLastStoryCode, STORYCODElength);
                }
                lStoryWithEntries->add(lEntry, lInputString);
                // lStoryWithEntries will delete the lEntry and its
                // lInputString eventually
            }
        }
    } // while lIndex.next()

    // the very last one
    lStoryWithEntries->completeEntries(pAllowedTitleCountries);
    lStoryWithEntries->write(lStoryOutputFile, lEntryOutputFileArray);
    delete lStoryWithEntries;

    addUnsolvedToOutputFileArray(lEntryOutputFileArray);
}


void GatherStoryWithEntries::addUnsolvedToOutputFileArray(CountryFileArray & pFileArray)
{
    MyInputFile lInputFile (FileName(cUnsolvedFileName, cStep1Extension));

    MyDelimitedInputString lInputString;

    while (lInputString.read(lInputFile) != eEndOfFile)
    {
        InternalEntry lEntry;
        lEntry.scan(lInputString, TRUE);
        // TRUE = only to find out the country!

        MyInternalOutputFile & l = pFileArray.element(lEntry.country());
        l.putInputString(lInputString);
        l.putNewLine();
    }
}


void GatherStoryWithEntries::storyInformationInIssueEntries(void)
{
    for (int i = 0; i < aNumber; i++)
    {
        if (kindOfEntry(i) == StoryWithEntriesRecord::eReprintEntry)
        {
            completeEntryWithStoryData(entry(i), entry(aTheStoryIndex));
        }
    }
}


void GatherStoryWithEntries::titlesInAllEntries(const CountrySet & pAllowedTitleCountries)
{
    int i, j;

    for (j = 0; j < aNumber; j++)
    {
        // check for '.': Titles in the Dutch files
        // can be '..', we want to keep these titles,  ****
        // but never use them in other files

        if (  (kindOfEntry(j) != StoryWithEntriesRecord::eXRefEntry)
           && (titleOrigin(j) == StoryWithEntriesRecord::eTitleHere)
           && (entry(j).title()[0] != '.')
           )
        {
            for (i = 0; i < aNumber; i++)
            {
                if ((i != j) && (kindOfEntry(i) != StoryWithEntriesRecord::eXRefEntry))
                {
                    takeBestTitle(entry(i), entry(j), titleOrigin(i),
                                  pAllowedTitleCountries);
                }
            }
        }
    }
}


void GatherStoryWithEntries::reprintsAndXRefsInAllEntries(void)
{
    int i, j;
    ENTRYCODEtype jEntryCodeBuffer;

    for (j = 0; j < aNumber; j++)
    {
        Entry & jEntry = entry(j);

        switch (kindOfEntry(j))
        {
        default:
            assert(FALSE);
        case StoryWithEntriesRecord::eStoryEntry:
            break; // nothing

        case StoryWithEntriesRecord::eReprintEntry:
            if (jEntry.goodPart())
            {
                (void) jEntry.cleanedEntryCode(jEntryCodeBuffer);
                for (i = 0; i < aNumber; i++)
                {
                    if ((i != j) && (kindOfEntry(i) != StoryWithEntriesRecord::eXRefEntry))
                    {
                        Entry & iEntry = entry(i);
                        if (iEntry.goodPart())
                        {
                            // reprints: add reference
                            iEntry.reprint(jEntry.country()).insert(
                                jEntryCodeBuffer);
                        }
                    }
                }
            }
            break;
        case StoryWithEntriesRecord::eXRefEntry:
            for (i = 0; i < aNumber; i++)
            {
                if ((i != j) && (kindOfEntry(i) != StoryWithEntriesRecord::eXRefEntry))
                {
                    Entry & iEntry = entry(i);
                    if (iEntry.goodPart())
                    {
                        // add cross reference
                        iEntry.field(eXRefd).concat(jEntry.field(eXRefd));
                    }
                }
            }
            kindOfEntry(j) = StoryWithEntriesRecord::eNoEntry;
            // this way the 'write' will not complain about xref-entries
        } // end switch
    }
}


void GatherStoryWithEntries::takeBestTitle(
    Entry & pTarget,
    Entry & pAlternative,
    StoryWithEntriesRecord::TitleOrigin & pTargetOrigin,
    const CountrySet & pAllowedTitleCountries)
{
    StoryWithEntriesRecord::TitleOrigin lNewOrigin;
    CountryIndex lAlternativeCountry(pAlternative.country());

    if (pTarget.country().number() == eStoryCountry)
    {
        if (pAllowedTitleCountries.includes(lAlternativeCountry))
        {
            lNewOrigin = StoryWithEntriesRecord::eForeignTitle;
        }
        else
        {
            return;
        }
    }
    else if (lAlternativeCountry == eStoryCountry)
    {
        lNewOrigin = StoryWithEntriesRecord::eOriginalTitle;
    }
    else
    {
        switch (pTarget.country().number())
        {
        case eWesternCountry:
            if (pAllowedTitleCountries.includes(lAlternativeCountry))
            {
                lNewOrigin = StoryWithEntriesRecord::eForeignTitle;
            }
            else
            {
                return;
            }
            break;
        case eGladstoneCountry:
            return;
            // English titles from the Story entries are more
            // important than English titles from a different
            // (English) Issue entry.
        default:
            if (pTarget.country() == lAlternativeCountry)
            {
                lNewOrigin = StoryWithEntriesRecord::eTitleElsewhere;
            }
            else
            {
                // eForeignTitle; we don't like that in an issues file...
                return;
            }
        } // switch
    } // else

    if (pTargetOrigin < lNewOrigin)
    {
        // lNewOrigin is more important: copy the title and
        // adjust its prefix, if necessary

        if (  (lNewOrigin == StoryWithEntriesRecord::eTitleElsewhere)
           || (lNewOrigin == StoryWithEntriesRecord::eForeignTitle)
           )
        {
            pTarget.sTitlePrefix =
                lAlternativeCountry.reprintTitlePrefix();
        }
        else
        {
            pTarget.sTitlePrefix = pAlternative.sTitlePrefix.theString();
        }

        pTarget.copyTitle(pAlternative);
        pTargetOrigin = lNewOrigin;
    }
}


void GatherStoryWithEntries::completeEntryWithStoryData(
    Entry & pTarget,
    Entry & pStory)
{
    // fill in various fields, except the title!
    // The title will be filled in later

    assert(pTarget.sStoryCode == pStory.sStoryCode);

    bool lCopyExactPages = TRUE;

    if (pStory.sPages.empty())
    {
        // no need to do anything
        // empty pages field occurs in newspaper strips

        // if (pTarget.sPages.empty()): something wrong...
    }
    else if (pTarget.sPages.empty())
    {
        pTarget.sPages.set(pStory.sPages);
        pTarget.sPageLayout = pStory.sPageLayout;
        // only copy page layout if pages has been copied as well
    }
    else
    {
        if (! (pStory.sPages.equalTo(pTarget.sPages)))
        {
            if (pStory.field(eExactPages).empty())
            {
                pTarget.field(eOriginalPages).set(pStory.sPages.theString());
            }
            else
            {
                pTarget.field(eOriginalPages).set(
                    pStory.field(eExactPages).delimitedString());
            }
            lCopyExactPages = FALSE;
            // 'exactpages' should not be copied from story to target, since
            // it makes no sense if 'pages' is different.
        }
    }

    ShortListFieldIndex s;
    while (s.next())
    {
        if (s.number() == ePart)
        {
            // copy, except when [part:1]. The reprint might concern
            // the entire story
            if (pTarget.field(s).empty())
            {
                const char * lPart = pStory.field(s).element(0);
                if (strcmp(lPart, "1") != 0)
                {
                    pTarget.field(s).concat(pStory.field(s));
                }
            }
        }
        else if ((s.number() != eExactPages) || lCopyExactPages)
        {
            if (pTarget.field(s).empty())
            {
                if (! pStory.field(s).empty())
                {
                    pTarget.field(s).concat(pStory.field(s));
                }
            }
            else
            {
                // target overrules story-data.

                switch (s.number())
                {
                case ePlotter:
                case eWriter:
                case eArtist:
                case eInker:
                case eHero:
                case eAppearing:
                case eSer:
                    pTarget.comment().append(" [org.");
                    pTarget.comment().append(s.prefix());
                    pTarget.comment().append(":");
                    pTarget.comment().append(pStory.field(s).delimitedString());
                    pTarget.comment().append("]");
                    break;
                default:
                    break;
                }
            }
        }
    }

    pTarget.comment().append(pStory.comment());

    // not copied: reprint information
}
