/* HEADER.CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   95-04-10  created
   96-08-16  checked
*/

#include <assert.h>
#include <string.h>
#include "header.h"
#include "mylib.h"

Header::Header(void)
:
    aCountry(eNoCountry)
{
    aLevel[0] = '\0';
// ***    aHeaders = new ShortList[eLastHeaderField];
//    assert (aHeaders != 0);
}


Header::Header(const CountryIndex & pCountry)
:
    aCountry(pCountry)
{
    aLevel[0] = '\0';
// ***    aHeaders = new ShortList[eLastHeaderField];
//    assert (aHeaders != 0);
}


Header::~Header()
{
// ***    delete [] aHeaders;
}


void Header::scanInternal(MyDelimitedInputString & pString)
{
    pString.getDelimitedString (aLevel, 1);
    pString.getDelimitedPointer (aStoryCode);

    PointerIntoInputString lCountryPointer;
    pString.getDelimitedPointer (lCountryPointer);
    aCountry = CountryIndex(lCountryPointer);

    pString.getDelimitedPointer (aTitle);

    PointerIntoInputString lPrefix;
    while (pString.getPrefix(lPrefix, PREFIXlength))
    {
        if (pString.atEnd())
        {
            // prefix without contents, does occur in headers
        }
        else
        {
            PointerIntoInputString lPointer;
            pString.getDelimitedPointer(lPointer);

            HeaderFieldIndex h(lPrefix);
            if (h.valid())
            {
                field(h).insert(lPointer);
            }
            else
            {
                aComment.append(lPointer);
            }
        }
    }
}


void Header::putInternal(MyInternalOutputFile & pFile, bool pSkipWesternPrefix)
{
    pFile.putDelimitedString(aLevel);
    if (  pSkipWesternPrefix
       && aStoryCode.startsWithWSpace()
       )
    {
        aStoryCode.shiftRight();
        pFile.putDelimitedString(aStoryCode);
        aStoryCode.shiftLeft();
    }
    else
    {
        pFile.putDelimitedString(aStoryCode);
    }
    pFile.putDelimitedString(aCountry.abbreviation());
    pFile.putDelimitedString(aTitle);

    HeaderFieldIndex h;
    while (h.next())
    {
        if (!field(h).empty())
        {
            pFile.putPrefix(h.prefix());
            pFile.putDelimitedString(field(h).delimitedString());
        }
    }
    if (! aComment.empty())
    {
        pFile.putPrefix("comment"); // anything except..; see scan()
        pFile.putDelimitedString(aComment.element());
    }
    pFile.putNewLine();
}


void Header::scanExternal(MyFixedPosInputString & pString)
{
    // layout: StoryCode h level Title [prefix:field] [prefix:field]
    // 	       'h' on position 'cHeaderIndicatorPosition'
    //	       level right after the 'h'

    // if level = 0:
    // Title == Filename (without extension)

    pString.scanFixedPosInBuffer(0, cHeaderIndicatorPosition - 1, aStoryCode);

    if (aStoryCode[0] == '0')
    {
        aStoryCode.pointTo("");
        // in external format, the first position cannot be a space
        // (for that's a comment line). So we need a trick to indicate
        // an empty story code in the header.
    }

    pString.scanFixedPos (cHeaderIndicatorPosition + 1, 1, aLevel);

    // everything behind 'level' position is delimited
    pString.split('#', cHeaderIndicatorPosition + 2);
    // from now on, we can use the methods of MyDelimitedInputString

    pString.skipSpaces();
    pString.getDelimitedPointer(aTitle, '[');

    // the rest
    PointerIntoInputString lPrefix;
    PointerIntoInputString lCommentPointer;
    while (pString.getPrefixedComment(lPrefix, PREFIXlength, lCommentPointer))
    {
        if (lCommentPointer[0] != '\0')
        {
            HeaderFieldIndex h(lPrefix);
            if (h.valid())
            {
                field(h).insert(lCommentPointer);
            }
            else
            {
                aComment.append(lCommentPointer);
            }
	}
    }
    if (!pString.atEnd())
    {
	MyLib::log("LEFTOVER in header of %s", aStoryCode.zeroString());
    }
}


void Header::putExternal(MyExternalOutputFile & pFile)
{
    // some nice layout, not necessarily like the input

    if (aLevel[0] == '0') // file header
    {
        return;
    }

    if (aStoryCode[0] == '0')   // story or entry code is fake
    {
        if (!aTitle.empty())
        {
            pFile.putS(aTitle);
        }
    }
    else
    {
        pFile.putS(aStoryCode);

        if (! aTitle.empty() && ! aStoryCode.empty())
        {
            pFile.putS(" = ");
            pFile.putS(aTitle);
        }
    }

    HeaderFieldIndex h;
    while (h.next())
    {
	if (!field(h).empty())
        {
	    pFile.printF(" [%s:%s]",
                          h.prefix(),
		          field(h).delimitedString());
	}
    }
    if (! aComment.empty())
    {
        pFile.printF(" [%s]", aComment.element());
    }
    pFile.printLine();
}


void Header::putRawExternal(MyExternalOutputFile & pFile)
{
    // exactly like the input

    if (aLevel[0] == '0') // file header
    {
        return;
    }

    pFile.putSAligned(aStoryCode, cHeaderIndicatorPosition);
    pFile.printF("h%s ", aLevel);
    pFile.putS(aTitle);

    HeaderFieldIndex h;
    while (h.next())
    {
	if (!field(h).empty())
        {
	    pFile.printF(" [%s:%s]",
                          h.prefix(),
		          field(h).delimitedString());
	}
    }
    if (! aComment.empty())
    {
        pFile.printF(" [%s]", aComment.element());
    }
    pFile.printLine();
}
