/* MYOUTPUT.H, (c) Harry Fluks 1994

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   Objects: MyOutputString, MyOutputFile,
            MyInternalOutputFile, MyExternalOutputFile,
            MyBufferedOutputFile,
            MyUnformattedOutputFile, MyFormattedOutputFile,
            MyWebPageOutputFile,
            MyFluksWebPageOutputFile, MyBarksWebPageOutputFile

   Various output files, with various layouts and properties.
   All outputfiles are in the subdirectory './data'.

   Inheritance tree:

   MyOutputFile--+->Internal
                 |
                 +->External--+->Buffered---+--->Unformatted
                              |             |
                              |             +--->Formatted
                              |
                              +->WebPage----+--->FluksWebPage
                                            |
                                            +--->BarksWebPage

   96-07-04  created
   96-08-21  lots of inheritance added
   96-08-22  checked
   97-06-23  writeExtraText() now possible for all ExternalOutputFiles
   97-06-24  Fluks/BarksWebPage added
*/

#ifndef MYOUTPUT_H
#define MYOUTPUT_H

#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "mylib.h"
#include "myinput.h"
#include "pointeri.h"

class MyOutputString
{
public: // but only to be used by the other classes in this header file...
    MyOutputString(int pBufferSize = MAXLINEsize);
    ~MyOutputString();

    bool empty(void) { check(); return aBuffer[0] == '\0'; }
    void clear(void);

    char * buffer(void) { check(); return aBuffer; }
    char * end(void) { goToEnd(); return aEnd; }

    void compress(void);
    void trimTrail(void);
    void fillSpacesToColumn(int pToColumn);

    void check(void) const;
private:
    void goToEnd();

    char * aBuffer;
    int aMaxLength;
    char * aEnd;  // optimisation for consecutive 'printf' calls
};


class MyOutputFile
{
public:
    MyOutputFile(const FileName &);
    virtual ~MyOutputFile();

protected:
    FILE * aFile;
};


class MyInternalOutputFile: public MyOutputFile
{
public:
    MyInternalOutputFile(const FileName &);
    virtual ~MyInternalOutputFile();

    void putPrefix                     (const char * pValue);
    void putDelimitedString            (const char * pValue);
    void putDelimitedString            (const PointerIntoInputString &);
    void putDelimitedStringFixedLength (const char * pValue, int pLength);
    void putDelimitedStringFixedLength (const PointerIntoInputString &, int pLength);
    void putDelimitedStringWithSpace   (const PointerIntoInputString &);
    void putDelimitedInteger           (const int);
    void putInputString                (const MyInputString &);

    void putNewLine(void);
};


class MyExternalOutputFile: public MyOutputFile
{
public:
    MyExternalOutputFile(const FileName &);
    virtual ~MyExternalOutputFile();

    virtual void printF      (const char * pFormat, ...) = 0;
    virtual void putS        (const char * pString) = 0;
    virtual void putS        (const PointerIntoInputString &) = 0;
    virtual void putSAligned (const PointerIntoInputString &, int pWidth) = 0;

    void putInputString      (const MyInputString &);
    // not virtual

    virtual void printLine(void) = 0;

    virtual void headerPrefix  (int pLevel);
    virtual void headerPostfix (int pLevel);

    static void setProgramId(const char *);

    void writeExtraText(const FileName &);

protected:
    static const char * gProgramId;
};


class MyBufferedOutputFile: public MyExternalOutputFile
{
public:
    MyBufferedOutputFile(const FileName &, int pBufferSize);
    virtual ~MyBufferedOutputFile();

    virtual void printF      (const char * pFormat, ...);
    virtual void putS        (const char * pString);
    virtual void putS        (const PointerIntoInputString &);
    virtual void putSAligned (const PointerIntoInputString &, int pWidth);

    // virtual void printLine(void); not redefined => this class is abstract

protected:
    MyOutputString aOutputString;
};


class MyUnformattedOutputFile: public MyBufferedOutputFile
{
public:
    MyUnformattedOutputFile(const FileName &, int pBufferSize = MAXLINEsize);
    virtual ~MyUnformattedOutputFile();

    virtual void printLine(void);

    void compress(void);
};


class MyFormattedOutputFile: public MyBufferedOutputFile
{
public:
    MyFormattedOutputFile(
        const FileName &,
        int pMaxLineWidth,
        int pIndentation,
        int pBufferSize = MAXLINEsize);
    virtual ~MyFormattedOutputFile();

    void forceNewLine(void);

    virtual void printLine(void);
    // can print multiple lines if the buffer is too wide for aMaxLineWidth

    void fillSpacesToColumn(int);
    void changeIndentation(int p) { aIndentation = p; }

private:
    void printControlled(char *);

    int aMaxLineWidth;
    int aIndentation;
};


class MyWebPageOutputFile: public MyExternalOutputFile
{
protected: // caller should use MyBarksWebPage- or MyFluksWebPageOutputFile
    MyWebPageOutputFile(const FileName &);
    virtual ~MyWebPageOutputFile();

public:
    virtual void printF      (const char * pFormat, ...);
    virtual void putS        (const char * pString);
    virtual void putS        (const PointerIntoInputString &);
    virtual void putSAligned (const PointerIntoInputString &, int pWidth);

    virtual void printLine(void);

    virtual void headerPrefix  (int pLevel);
    virtual void headerPostfix (int pLevel);
};


class MyFluksWebPageOutputFile: public MyWebPageOutputFile
{
public:
    MyFluksWebPageOutputFile(const FileName &);
    virtual ~MyFluksWebPageOutputFile();
};


class MyBarksWebPageOutputFile: public MyWebPageOutputFile
{
public:
    MyBarksWebPageOutputFile(const FileName &);
    virtual ~MyBarksWebPageOutputFile();
};

#endif
