/* MYINPUT.CC, (c) Harry Fluks 1994

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-07-04  created
   96-08-23  checked
*/

#include <ctype.h>  // toupper
#include <stdlib.h> // exit
#include "myinput.h"
#include "buffers.h"


MyInputFile::MyInputFile(const FileName & p, bool pAppend)
:
    aFile(0)
{
    const char * lName = p.fullNameInOrgDir();
    aFile = fopen(lName, (pAppend ? "r+" : "r"));

    if (aFile == 0)
    {
        lName = p.fullNameInDataDir();
        aFile = fopen(lName, (pAppend ? "r+" : "r"));

        if (aFile == 0)
        {
            MyLib::log("cannot open file %s or %s!",
                       p.fullNameInOrgDir(), lName);
            printf("cannot open file %s!\n", lName);
            exit(1);
        }
    }
}


MyInputFile::~MyInputFile()
{
    assert(aFile != 0);
    (void) fclose(aFile);
}


MyInputString::MyInputString(int pMaxLength)
:
    aBuffer(0),
    aMaxLength(pMaxLength),
    aUsedByOthers(0)
{
    aBuffer = Buffers::newBuffer(9, pMaxLength + 1);
    aBuffer[0] = '\0';
}


MyInputString::~MyInputString()
{
    Buffers::deleteBuffer(9, aBuffer);
    assert(aUsedByOthers == 0);
}


InputLineType MyInputString::read(MyInputFile & pFile)
{
    aBuffer[0] = '\0';

    if (fgets(aBuffer, aMaxLength - 1, pFile.theFile()))
    {
        Progress::addRead();

        reset(); // virtual function
        if (!MyLib::trimnewline(aBuffer))
        {
            MyLib::log("Line has no newline - possibly longer than %ld",
                       (long) aMaxLength);
            MyLib::log(aBuffer);
        }
        check();
        return theType();
    }
    else
    {
        return eEndOfFile;
    }
}


bool MyInputString::matchAnyText(const char * pMatchText) const
{
    // find substring in string
    for (char * c = aBuffer; *c != '\0'; c++)
    {
        char * e = c;
        const char * d = pMatchText;
        while (toupper(*e) == toupper(*d))
        {
            e++;
            d++;
        }
        if (*d == '\0')
        {
            return TRUE;
        }
    }
    return FALSE;
}


void MyInputString::replace(char pChar1, char pChar2)
{
    for (char * c = aBuffer; *c != 0; c++)
    {
        if (*c == pChar1)
        {
            *c = pChar2;
        }
    }
}


void MyInputString::pointIntoBuffer(bool p)
{
    if (p)
    {
        aUsedByOthers++;
    }
    else
    {
        assert(aUsedByOthers > 0);
        aUsedByOthers--;
    }
}


InputLineType MyInputString::theType(void)
{
    switch(aBuffer[0])
    {
    case '\0': return eEmptyLine;
    case ' ':  return eCommentLine;
    case 'r':  return eWesternReprintLine;
    case '#':  return eFollowUpLine;
    default:
        if (strlen(aBuffer) > cHeaderIndicatorPosition)
        {
            if (aBuffer[cHeaderIndicatorPosition] == 'h')
            {
                return eHeaderLine;
            }
        }
        return eEntryLine;
    }
}


void MyInputString::check(void) const
{
    assert(aBuffer != 0);
    assert(strlen(aBuffer) < aMaxLength);
}


MyDelimitedInputString::MyDelimitedInputString(int pMaxLength)
:
    MyInputString(pMaxLength),
    aPosition(aBuffer)
{
}


MyDelimitedInputString::~MyDelimitedInputString()
{
}


void MyDelimitedInputString::reset(void)
{
    aPosition = aBuffer;
}


void MyDelimitedInputString::skip(void)
{
    assert(aPosition != 0);
    if (*aPosition != '\0')
    {
        aPosition++;
    }
}


void MyDelimitedInputString::skipSpaces(void)
{
    assert(aPosition != 0);
    check();
    while (*aPosition == ' ')
    {
        aPosition++;
    }
}


void MyDelimitedInputString::getDelimitedString(
    char * pTarget,
    int pMaxLength,
    char pDelimiter)
{
    assert(aPosition != 0);
    assert(pTarget != 0);
    check();

    char * d = pTarget;
    int i = 0;

    while ((*aPosition != '\0') && (*aPosition != pDelimiter))
    {
        *d++ = *aPosition++;
        i++;
        if (i > pMaxLength)
        {
            MyLib::log("Field too long in (%s) - max is %d!", aBuffer, pMaxLength);
            break;
        }
    }
    *d = '\0';
    skip(); // delimiter
}


void MyDelimitedInputString::getDelimitedPointer(
    PointerIntoInputString & pTarget,
    char pDelimiter)
{
    assert(aPosition != 0);
    check();

    const char * lStart = aPosition;
    int lLength = 0;
    while ((*aPosition != '\0') && (*aPosition != pDelimiter))
    {
        aPosition++;
        lLength++;
    }
    if (*aPosition != '\0')
    {
        aPosition++;
    }
    pTarget.pointTo(this, lStart, lLength);
}


bool MyDelimitedInputString::getPrefix(
    PointerIntoInputString & pPrefix,
    int pMaxLength,
    char pDelimiter)
{
    assert(aPosition != 0);
    check();

    static const char cPrefixDelimiter = ':';

    pPrefix.pointTo("");
    if (atEnd())
    {
        return FALSE;
    }

    // find a cPrefixDelimiter
    char * t = aPosition;
    int lLength = 0;
    while (  (*t != '\0')       && (*t != cPrefixDelimiter)
          && (*t != pDelimiter) && (lLength < pMaxLength)
          )
    {
        t++;
        lLength++;
    }
    if (*t != cPrefixDelimiter)
    {
        // no prefix found within the desired length
        pPrefix.pointTo("");
    }
    else
    {
        pPrefix.pointTo(this, aPosition, lLength);
        aPosition = t;
        skip();
    }
    return TRUE;
}


bool MyDelimitedInputString::getPrefixedComment(
    PointerIntoInputString & pPrefix,
    int pPrefixLength,
    PointerIntoInputString & pComment)
{
    assert(aPosition != 0);
    // syntax: spaces[prefix:comment]

    check();

    // skip spaces and opening '['
    while (*aPosition == ' ')
    {
        aPosition++;
    }
    if (*aPosition == '[')
    {
        aPosition++;
    }

    (void) getPrefix(pPrefix, pPrefixLength, ']');

    // aPosition is now start of comment. Find out its length
    int lLength = 0;
    char * d = aPosition;
    while ((*d!= '\0') && (*d != ']'))
    {
        d++;
        lLength++;
    }
    pComment.pointTo(this, aPosition, lLength);

    aPosition += lLength;
    skip();
    return ((! pComment.empty()) || (! pPrefix.empty()));
}



MyFixedPosInputString::MyFixedPosInputString(int pMaxLength)
:
    MyDelimitedInputString(pMaxLength)
{
    aBuffer[0] = '\0';
    aPosition = 0; // so use of MyDelimitedInputString-functions will fail
}


MyFixedPosInputString::~MyFixedPosInputString()
{
}


void MyFixedPosInputString::scanFixedPos(
    int pStartPos,
    int pMaxLength,
    char * pTarget) const
{
    // also deals with the situation that pString is not long enough

    assert(pTarget != 0);
    check();

    if (pStartPos >= strlen(aBuffer))
    {
        pTarget[0] = '\0';
    }
    else
    {
        strncpy(pTarget, aBuffer + pStartPos, pMaxLength);
        pTarget[pMaxLength] = '\0';
        MyLib::trimtrail(pTarget);
    }
}


void MyFixedPosInputString::scanFixedPos(
    int pStartPos,
    Pages & pTarget) const
{
    check();

    if (pStartPos >= strlen(aBuffer))
    {
        pTarget.clear();
    }
    else
    {
        pTarget.set(aBuffer + pStartPos);
        if (pStartPos + 2 < strlen(aBuffer))
        {
            switch(aBuffer[pStartPos + 2])
            {
            default:
                MyLib::log("Unknown broken-page char (%c)",
                           aBuffer[pStartPos + 2]);
                // NO break
            case ' ':
                pTarget.setBroken(FALSE);
                break;
            case '+':
                pTarget.setBroken(TRUE);
                break;
            }
        }
    }
}


void MyFixedPosInputString::scanFixedPos(
    int pStartPos,
    PageLayout & pTarget) const
{
    check();

    if (pStartPos >= strlen(aBuffer))
    {
        pTarget = " "; // calls PageLayout::operator=
    }
    else
    {
        pTarget = aBuffer + pStartPos;
    }
}


void MyFixedPosInputString::scanFixedPosInBuffer(
    int pStartPos,
    int pMaxLength,
    PointerIntoInputString & pTarget)
{
    check();

    int lBufferLength = strlen(aBuffer);
    if (pStartPos >= lBufferLength)
    {
        pTarget.pointTo("");
    }
    else
    {
        if (pStartPos + pMaxLength > lBufferLength)
        {
            pTarget.pointTo(this, aBuffer + pStartPos, lBufferLength - pStartPos);
        }
        else
        {
            pTarget.pointTo(this, aBuffer + pStartPos, pMaxLength);
        }
    }
}


void MyFixedPosInputString::split(char pChar, int pPosition)
{
    aPosition = aBuffer;

    // if char occurs before pPosition, it's the delimiter
    // between part 1 and 2.
    // else, pPosition is the start of part 2.

    check();
    if (*aBuffer == '\0')
    {
        return; // cannot split empty string
    }

    aPosition++;
    // 0th position should not be scanned - e.g. '#' has different meaning

    while (*aPosition != '\0')
    {
        if (*aPosition == pChar)
        {
            *aPosition = '\0';
            aPosition++;
            return;
        }
        aPosition++;
    }
    // no pChar found
    if (strlen(aBuffer) > pPosition)
    {
        assert(pPosition > 1);
        aBuffer[pPosition - 1] = '\0';
        aPosition = aBuffer + pPosition;
    }
    else
    {
        aPosition = MyLib::atend(aBuffer);
    }
}
