/* PERSON.CC, (c) Harry Fluks 1997

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   97-09-27  created
*/

#include <stdlib.h> // atoi
#include "person.h"
#include "mylib.h"
#include "buffers.h"

static const char cNoAbbreviation = '0'; // to be used in the DBL file
static const int cMaxTitle = 30; // maximum #chars in a title
static const int cNamePosition = 15; // counting from 0!

Person::Person(const char * pLastName)
:
    aUsed(FALSE),
    aFullName(0),
    aFullNameLatin1(0),
    aField(0)
{
    aAbbreviation[0] = '\0';

    if (pLastName != 0) // a Person, not read from a file => "complete" Person
    {
        aFullName = Buffers::newBuffer(6, strlen(pLastName) + 1);
        strcpy(aFullName, pLastName);

// **** no need        aField = new ShortList[eLastPersonField];
//        assert (aField != 0);
    }

    RoleFieldIndex r;
    while (r.next())
    {
        aRoleCount[r.number()] = 0;
    }

    CountryIndex c;
    while (c.next())
    {
        aUsedIn[c.number()] = 0;
    }
}


Person::~Person()
{
    if (aField != 0)
    {
        delete [] aField;
    }
    if (aFullName != 0)
    {
        Buffers::deleteBuffer(6, aFullName);
    }
    if (aFullNameLatin1 != 0)
    {
        Buffers::deleteBuffer(7, aFullNameLatin1);
    }
}



void Person::scanInternal(MyDelimitedInputString & pString, bool pAllFields)
{
    char lBuffer[81];

    // full name:
    pString.getDelimitedString(lBuffer, 80);
    assert (aFullName == 0);
    aFullName = Buffers::newBuffer(6, strlen(lBuffer) + 1);
    strcpy(aFullName, lBuffer);

    // abbreviation:
    pString.getDelimitedString(aAbbreviation, PERSONABBRlength);

    // full name in Latin-1:
    pString.getDelimitedString(lBuffer, 80);
    if (strlen(lBuffer) > 0)
    {
        assert (aFullNameLatin1 == 0);
        aFullNameLatin1 = Buffers::newBuffer(7, strlen(lBuffer) + 1);
        strcpy(aFullNameLatin1, lBuffer);
    }

    if (!pAllFields)
    {
        return; // leaving some attributes "empty"
    }

    // now we are going to use the "aField" field:
    assert (aField == 0);
    aField = new ShortList[eLastPersonField];
    assert (aField != 0);

    PointerIntoInputString lPrefix;
    while (pString.getPrefix(lPrefix, PREFIXlength))
    {
        if (pString.atEnd())
        {
            // prefix without contents, does occur (?)
        }
        else
        {
            PointerIntoInputString lPointer;
            pString.getDelimitedPointer(lPointer);

            PersonFieldIndex h(lPrefix);
            if (h.valid())
            {
                field(h).insert(lPointer);
            }
            else
            {
                RoleFieldIndex r(lPrefix);
                if (r.valid())
                {
                    aRoleCount[r.number()] = atoi(lPointer.zeroString());
                }
                else
                {
                    CountryIndex c(lPrefix);
                    if (c.valid())
                    {
                        aUsedIn[c.number()] = atoi(lPointer.zeroString());
                    }
                    else
                    {
                        field(eComment).insert(lPointer);
                    }
                }
            }
        }
    }
}


void Person::putInternal(MyInternalOutputFile & pFile)
{
    if (aFullName == 0)
    {
        pFile.putDelimitedString("");
    }
    else
    {
        pFile.putDelimitedString(aFullName);
    }

    pFile.putDelimitedString(aAbbreviation);

    if (aFullNameLatin1 == 0)
    {
        pFile.putDelimitedString("");
    }
    else
    {
        pFile.putDelimitedString(aFullNameLatin1);
    }

    if (aField != 0)
    {
        PersonFieldIndex h;
        while (h.next())
        {
            if (!field(h).empty())
            {
                pFile.putPrefix(h.prefix());
                pFile.putDelimitedString(field(h).delimitedString());
            }
        }
    }
    RoleFieldIndex r;
    while (r.next())
    {
        if (aRoleCount[r.number()] != 0)
        {
            pFile.putPrefix(r.prefix());
            pFile.putDelimitedInteger(aRoleCount[r.number()]);
        }
    }
    CountryIndex c;
    while (c.next())
    {
        if (aUsedIn[c.number()] != 0)
        {
            pFile.putPrefix(c.abbreviation());
            pFile.putDelimitedInteger(aUsedIn[c.number()]);
        }
    }
    pFile.putNewLine();
}


void Person::scanExternal(MyFixedPosInputString & pString, bool pCreatorLayout)
{
    // we are SURE we will use "aField":
    assert (aField == 0);
    aField = new ShortList[eLastPersonField];
    assert (aField != 0);

    if (pCreatorLayout)
    {
/* pCreatorLayout == TRUE:
MAb  Marsal          Abella Bresco [also Marcal with cedille]

used for creators and indexers. Abbreviation can only be 3 characters.
*/
        pString.scanFixedPos(0, 5, aAbbreviation);

        char lBuffer[80];
        pString.scanFixedPos(5, 16, lBuffer);
        if (lBuffer[0] != '\0')
        {
            field(eFirstName).insert(lBuffer);
        }

        pString.split('#', 21);
        // from now on, we can use the methods of MyDelimitedInputString

        char lBuffer2[81];
        pString.skipSpaces();
        pString.getDelimitedString(lBuffer2, 80, '[');
        MyLib::trimtrail(lBuffer2);
        if (lBuffer2[0] != '\0')
        {
            field(eLastName).insert(lBuffer2);
        }

        // now we can build the full name:
        // note that lBuffer and lBuffer2 contain the data
        assert (aFullName == 0);
        const char * lSpace = " ";
        if ((lBuffer[0] == '\0') || (lBuffer2[0] == '\0'))
        {
            lSpace = "";
        }
        aFullName = Buffers::newBuffer(
            6, strlen(lBuffer) + strlen(lBuffer2) + strlen(lSpace) + 1);
        sprintf(aFullName, "%s%s%s", lBuffer, lSpace, lBuffer2);
    }
    else
    {
/* pCreatorLayout == FALSE:
DD             Donald Duck [blah blah]

used for heroes and subseries. No latin-1 fields, no "firstname" field.
*/
        pString.scanFixedPos(0, PERSONABBRlength, aAbbreviation);

        pString.split('#', 15);
        // from now on, we can use the methods of MyDelimitedInputString

        char lBuffer2[81];
        pString.skipSpaces();
        pString.getDelimitedString(lBuffer2, 80, '[');
        MyLib::trimtrail(lBuffer2);
        assert (aFullName == 0);
        aFullName = Buffers::newBuffer(6, strlen(lBuffer2) + 1);
        strcpy(aFullName, lBuffer2);
    }

    PointerIntoInputString lPrefix;
    PointerIntoInputString lCommentPointer;
    while (pString.getPrefixedComment(lPrefix, PREFIXlength, lCommentPointer))
    {
        if (lCommentPointer[0] != '\0')
        {
            PersonFieldIndex h(lPrefix);
            if (h.valid())
            {
                field(h).insert(lCommentPointer);
            }
            else
            {
                field(eComment).insert(lCommentPointer);
            }
	}
    }
    if (!pString.atEnd())
    {
	MyLib::log("LEFTOVER in person record of %s", aFullName);
    }
    // note that aRoleCount and aUsedIn are not read from the file!

    // also fill in aFullNameLatin1

    ShortList & lShl1 = field(eFirstNameLatin1);
    ShortList & lShl2 = field(eLastNameLatin1);

    if (lShl1.empty() && lShl2.empty())
    {
        // keep aFullNameLatin1 == 0
        // happens a.o. if pCreatorLayout == FALSE
    }
    else
    {
        const char * lCh1 = lShl1.delimitedString();
        if (lShl1.empty())
        {
            lCh1 = field(eFirstName).delimitedString();
        }

        const char * lCh2 = lShl2.delimitedString();
        if (lShl2.empty())
        {
            lCh2 = field(eLastName).delimitedString();
        }

        const char * lSpace = " ";
        if ((lCh1[0] == '\0') || (lCh2[0] == '\0'))
        {
            lSpace = "";
        }

        assert(aFullNameLatin1 == 0);
        aFullNameLatin1 = Buffers::newBuffer(
            7, strlen(lCh1) + strlen(lCh2) + strlen(lSpace) + 1);
        assert(aFullNameLatin1 != 0);
        sprintf(aFullNameLatin1, "%s%s%s", lCh1, lSpace, lCh2);
    }
}


void Person::fullOutput(MyFormattedOutputFile & pOutputFile)
{
    pOutputFile.printF("%-*s%s", cNamePosition, aAbbreviation, fullName(FALSE));
    if (aFullNameLatin1 != 0)
    {
        pOutputFile.printF(" (%s)", fullName(TRUE));
    }

/******** aField kan 0 zijn en zo...
    if (! field(eExampleStoryPrefix).empty())
    {
        pOutputFile.forceNewLine();
        if (field(eExampleStoryPrefix).delimitedString()[0] == '1')
        {
            pOutputFile.printF(" [1st story:");
        }
        else
        {
            pOutputFile.printF(" [example story:");
        }
        pOutputFile.printF("%s", field(eExampleStoryCode).delimitedString());

        if (! field(eExampleStoryTitle).empty())
        {
            pOutputFile.printF(", \"%s\"",
                field(eExampleStoryTitle).delimitedString());
        }
        pOutputFile.printF("]");
    }
*/

/*** we don't output comment fields etc. (!) */

    bool lFirst = TRUE;
    RoleFieldIndex r;
    while (r.next())
    {
        if (aRoleCount[r.number()] != 0)
        {
            if (lFirst)
            {
                pOutputFile.forceNewLine();
                lFirst = FALSE;
            }
            pOutputFile.printF(" [%s:%d]", r.prefix(), aRoleCount[r.number()]);
        }
    }

    lFirst = TRUE;
    CountryIndex c;
    while (c.next())
    {
        if (aUsedIn[c.number()] != 0)
        {
            if (lFirst)
            {
                pOutputFile.forceNewLine();
                lFirst = FALSE;
            }
            pOutputFile.printF(" [%s:%d]",
                               c.abbreviation(),
                               aUsedIn[c.number()]);
        }
    }
    pOutputFile.printLine();
}


const char * Person::fullName(bool pInLatin1) const
{
    if (  ! pInLatin1
       || (aFullNameLatin1 == 0)
       )
    {
        if (aFullName == 0)
        {
            return "UNINITIALISED PERSON"; // should not occur...
        }
        else
        {
            return aFullName;
        }
    }
    else
    {
        return aFullNameLatin1;
    }
}


void Person::update(
    RoleFieldEnum pRole,
    CountryIndex pCountry,
    unsigned short int pNumberOfTimes,
    const PointerIntoInputString * pStoryCode,
    const PointerIntoInputString * pStoryTitle,
    bool pStoryIsFirst)
{
    aUsed = TRUE;
    assert(pRole < eLastRoleField);
    aRoleCount[pRole] += pNumberOfTimes;

    if (pCountry.number() < eLastCountry)
    {
        aUsedIn[pCountry.number()] += pNumberOfTimes;
    }

/********* TE VEEL GEHEUGEN en aField kan nu 0 zijn
    if (  (pStoryCode != 0)
       && ! pStoryCode->empty()
       )
    {
        assert(pStoryTitle != 0);
        if (field(eExampleStoryPrefix).empty() || pStoryIsFirst)
        {
            field(eExampleStoryPrefix).clear();
            field(eExampleStoryCode).clear();
            field(eExampleStoryTitle).clear();

            if (pStoryIsFirst)
            {
                field(eExampleStoryPrefix).insert("1");
            }
            else
            {
                field(eExampleStoryPrefix).insert("x"); // *** unreadable...
            }
            field(eExampleStoryCode).insert(*pStoryCode);

            char lBuffer[cMaxTitle + 1];
            pStoryTitle->copyToChars(lBuffer, cMaxTitle);
            field(eExampleStoryTitle).insert(lBuffer);
            if (pStoryTitle->length() > cMaxTitle)
            {
                field(eExampleStoryTitle).insert("...");
            }
        }
    }
*******/
}


ShortList & Person::field(PersonFieldIndex p) const
{
    assert(aField != 0);
    return aField[p.number()];
}


ShortList & Person::field(PersonFieldEnum p) const
{
    assert(aField != 0);
    return aField[p];
}


bool Person::hasAbbreviation(void) const
{
    return (  (aAbbreviation[0] != cNoAbbreviation)
           && (aAbbreviation[0] != '\0')
           );
}


const char * Person::key(void) const
{
    if (hasAbbreviation())
    {
        return aAbbreviation;
    }
    else
    {
        return fullName(FALSE);
    }
}
