/* PREFORMA(tter).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   First program in the generating cyclus; transforms all input files
   to an internal format, ready to be sorted.

   95-03-10  created
   96-06-01  also write to unsolved-file
   97-09-01  objectified
*/

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include "preforma.h"


Preformatter::Preformatter(bool pHeadersOnly)
:
    aOutputFileArray(cStep1Extension),
    aHeadersOnly(pHeadersOnly),
    aHeroList(TRUE),
    aCreatorList(TRUE), // TRUE: complete Persons
    aSubseriesList(TRUE),
    aIndexerList(TRUE)
{
}


Preformatter::~Preformatter()
{
}


void Preformatter::doit(void)
{
    aHeroList     .readFromExternalFile(FileName("heroes", "dbl"), FALSE);
    aCreatorList  .readFromExternalFile(FileName("creators", "dbl"), TRUE);
    aSubseriesList.readFromExternalFile(FileName("subserie", "dbl"), FALSE);
    aIndexerList  .readFromExternalFile(FileName("indexers", "dbl"), TRUE);
    {
        MyInternalOutputFile lHeaderFile(FileName("headers", "ins"));
        MyInternalOutputFile lHeader1File(FileName("headers1", "ins"));

        DBSFileIndex lDBSFileIndex;
        while (lDBSFileIndex.next())
        {
            processOneStoryFile(lDBSFileIndex, lHeaderFile, lHeader1File);
        }
    }
    {
        MyInternalOutputFile lHeaderFile(FileName("headers", "ine"));
        MyInternalOutputFile lHeader1File(FileName("headers1", "ine"));
        MyInternalOutputFile lUnsolvedFile(FileName(cUnsolvedFileName, cStep1Extension));

        DBIFileIndex lDBIFileIndex;
        while (lDBIFileIndex.next())
        {
            processOneEntryFile( lDBIFileIndex,
                                 lHeaderFile, lHeader1File, lUnsolvedFile);
        }
    }
    aHeroList     .writeToInternalFile(FileName("heroes", "inl"));
    aCreatorList  .writeToInternalFile(FileName("creators", "inl"));
    aSubseriesList.writeToInternalFile(FileName("subserie", "inl"));
    aIndexerList  .writeToInternalFile(FileName("indexers", "inl"));
}


void Preformatter::writeInternalHeader(
    MyFixedPosInputString & pString,
    MyInternalOutputFile & pOutputFile,
    MyInternalOutputFile & p1OutputFile,
    const CountryIndex & pCountry,
    bool pSkipWesternPrefix)
{
    Header lHeader(pCountry);
    lHeader.scanExternal(pString);
    lHeader.putInternal(pOutputFile, pSkipWesternPrefix);
    if (lHeader.level() == '1')
    {
        lHeader.putInternal(p1OutputFile, pSkipWesternPrefix);
    }

    if (pCountry != eStoryCountry)
    {
        // add indexer to legend. Don't do this for a story-entry, because
        // Western entries would be counted double.
        int lNumberOfTimes = 1;
        if (lHeader.field(eNumberOfIssues).numberOfElements() == 1)
        {
            lNumberOfTimes = atoi(
                lHeader.field(eNumberOfIssues).element(0));
        }
        aIndexerList.addPersons(lHeader, eIndexer, lNumberOfTimes, eIndexerRole);
    }
}


void Preformatter::writeXRef(ExternalEntry & pEntry, bool pItsAnEntry)
{
    int lNumber = pEntry.field(eXRef).numberOfElements();

    for (int i = 0; i < lNumber; i++)
    {
        // make entry with story code = eXRef-field, and
        // eXRefd-field = story code.
        // And the rest empty.

        InternalEntry lEntry;

        // Make sure story code is not too long
        STORYCODEtype lStoryCodeBuffer;
        strncpy(lStoryCodeBuffer, pEntry.field(eXRef).element(i), STORYCODElength);
        lStoryCodeBuffer[STORYCODElength] = '\0';
        lEntry.sStoryCode.pointTo(lStoryCodeBuffer);

        if (pItsAnEntry)
        {
            LINEtype lBuffer;
            sprintf(lBuffer, "%s/%s",
                             pEntry.country().abbreviation(),
                             pEntry.sEntryCode.zeroString());
            lEntry.field(eXRefd).insert(lBuffer);
        }
        else
        {
            lEntry.field(eXRefd).insert(pEntry.sStoryCode.zeroString());
        }
        lEntry.setCountry(eStoryCountry); // for all certainty
        StoryFileIndex lIndex(lEntry.sStoryCode);
        if (lIndex.valid())
        {
            lEntry.put(aOutputFileArray.element(lIndex));
        }
        else
        {
            MyLib::log("Invalid xref (%s)", lEntry.sStoryCode.zeroString());
        }
    }
}


void Preformatter::processOneStoryFile(
    DBSFileIndex & pFile,
    MyInternalOutputFile & pHeaderFile,
    MyInternalOutputFile & pHeader1File)
{
    MyFixedPosInputString lInputString;
    MyInputFile lStoryInputFile(pFile.fileName());

    InputLineType lType;
    while ((lType = lInputString.read(lStoryInputFile)) != eEndOfFile)
    {
        switch (lType)
        {
        case eHeaderLine:
            {
                writeInternalHeader(lInputString, pHeaderFile, pHeader1File,
                                    CountryIndex(eStoryCountry),
                                    FALSE); // FALSE = don't skip Western prefix
            }
            break;
        case eEntryLine:
            if (!aHeadersOnly)
            {
                ExternalEntry lEntry(pFile.format());
                lEntry.scanExternal(lInputString);
                lEntry.setCountry(eStoryCountry);
                StoryFileIndex lIndex(lEntry.sStoryCode);
                assert(lIndex.valid());
                lEntry.put(aOutputFileArray.element(lIndex));

                aHeroList.addPersons(lEntry, eHero, eHeroRole);
                aHeroList.addPersons(lEntry, eAppearing, eAppRole);
                aCreatorList.addPersons(lEntry, ePlotter, ePlotterRole);
                aCreatorList.addPersons(lEntry, eWriter, eWriterRole);
                aCreatorList.addPersons(lEntry, eArtist, eArtistRole);
                aCreatorList.addPersons(lEntry, eInker, eInkerRole);
                aSubseriesList.addPersons(lEntry, eSer, eNoRole);

                writeXRef(lEntry, FALSE);
            }
            break;
        case eWesternReprintLine:
        default:
            ;
        }
    }
}


void Preformatter::processOneEntryFile(
    DBIFileIndex & pFile,
    MyInternalOutputFile & pHeaderFile,
    MyInternalOutputFile & pHeader1File,
    MyInternalOutputFile & pUnsolvedFile)
{
    MyFixedPosInputString lInputString;
    MyInputFile lEntryInputFile(pFile.fileName());

    InputLineType lType;
    CountrySet lDummy;

    while ((lType = lInputString.read(lEntryInputFile)) != eEndOfFile)
    {
        switch (lType)
        {
        case eHeaderLine:
            writeInternalHeader(lInputString, pHeaderFile, pHeader1File,
                                pFile.country(), TRUE);
            break;
        case eEntryLine:
        case eWesternReprintLine:
            if (!aHeadersOnly)
            {
                ExternalEntry lEntry(pFile.format());
                lEntry.scanExternal(lInputString);
                lEntry.setCountry(pFile.country());
                StoryFileIndex lIndex(lEntry.sStoryCode);

                if (lIndex.valid())
                {
                    // MAYBE a story can be found
                    lEntry.put(aOutputFileArray.element(lIndex));
                }
                else
                {
                    // we KNOW we'll never find a story with this entry
                    lEntry.setUnsolved();
                    lEntry.put(pUnsolvedFile);
                }
                aHeroList.addPersons(lEntry, eHero, eHeroRole);
                aHeroList.addPersons(lEntry, eAppearing, eAppRole);
                aCreatorList.addPersons(lEntry, ePlotter, ePlotterRole);
                aCreatorList.addPersons(lEntry, eWriter, eWriterRole);
                aCreatorList.addPersons(lEntry, eArtist, eArtistRole);
                aCreatorList.addPersons(lEntry, eInker, eInkerRole);
                aSubseriesList.addPersons(lEntry, eSer, eNoRole);
                writeXRef(lEntry, TRUE);
            }
            break;
        default:
            ;
        }
    }
}
