/* REPORT.CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-01-07  created
   96-08-11  checked
*/

#include <assert.h>
#include "report.h"
#include "personle.h"
#include "webentry.h"


ReportOptions::ReportOptions(
    Format pFormat,
    const CountrySet pReprintCountrySet,
    const char pLayoutChar,
    const int pWidth)
:
    aFormat(pFormat),
    aReprintCountrySet(pReprintCountrySet),
    aLayout(eSeparateReprintLines),
    aWidth(pWidth)
{
    switch (pLayoutChar)
    {
    case 'l': aLayout = eSeparateReprintLines; break;
    case 'c': aLayout = eReprintsInColumns;    break;
    case 'r': aLayout = eReprintsComments;     break;
    case 'd': aLayout = eDescriptionLayout;    break;
    case 'w': aLayout = eWebPage;              break;
    default: ;
    }
}


Report::Report( const FileName & pDefaultName,
                const ReportOptions & pOptions,
                bool pInLatin1)
:
    aPersonLegend(0),
    aFile(0),
    aDefaultName(pDefaultName),
    aFormat(pOptions.aFormat),
    aReprintCountrySet(pOptions.aReprintCountrySet),
    aLayout(pOptions.aLayout),
    aWidth(pOptions.aWidth),
    aInLatin1(pInLatin1)
{
}


Report::~Report()
{
    close();
    if (aPersonLegend != 0)
    {
        delete aPersonLegend;
        // close() didn't do this; person legend was re-used over and over
    }
}


void Report::newFile(const char * pName)
{
    close();
    open(FileName(pName, aDefaultName)); // extension is still default
}


MyExternalOutputFile & Report::theOutputFile(void)
{
    open(aDefaultName);
    return *aFile;
}


void Report::writeEntry(ExternalEntry & p)
{
    open(aDefaultName);

    assert(aPersonLegend != 0);
    aPersonLegend->add(p, aReprintCountrySet);
    // this may add a little too much in case of web pages, but we
    // decide later whether or not to use it.
    // 'add' first, because the putExternal below is destructive

    p.field(eOldCode).clear();  // never write this one in a report
    p.field(eOldCodeA).clear(); // never write this one in a report
    p.field(eOldCodeB).clear(); // never write this one in a report

    ExternalEntry::setArrow(TRUE);
    // we want to have a nice "->" in the columns, when applicable

    if (aLayout == ReportOptions::eWebPage)
    {
        WebEntry::putWeb(p, *aFile, aReprintCountrySet, aPersonLegend);
    }
    else
    {
        p.changeFormat(aFormat);
        p.convertSpecialFields();

        switch (aLayout)
        {
        case ReportOptions::eReprintsInColumns:
            // eReprintsInColumns only supported in Barks output
            p.putStoryBarks(*aFile, TRUE);
            // TRUE == write reprints in columns
            break;
        case ReportOptions::eReprintsComments:
            p.putExternal(*aFile);
            p.putReprintsAsComments(*aFile, aReprintCountrySet);
            break;
        case ReportOptions::eSeparateReprintLines:
            p.putExternal(*aFile);
            p.putReprintsOnSeparateLines(
                *((MyFormattedOutputFile *) aFile),  // ugly typecast...
                aReprintCountrySet);
            break;
        case ReportOptions::eDescriptionLayout:
            p.putDescriptionLayout(*aFile);
        }
        aFile->printLine();
    }
}


void Report::writeHeader(Header & p)
{
    open(aDefaultName);

    if (aLayout == ReportOptions::eWebPage)
    {
        ((MyWebPageOutputFile *) aFile)->headerPrefix(2);
        p.putExternal(*aFile);
        ((MyWebPageOutputFile *) aFile)->headerPostfix(2);
    }
    else if (aLayout == ReportOptions::eDescriptionLayout)
    {
        // no need for headers!
    }
    else
    {
        p.putExternal(*aFile);
    }
}


void Report::startOwnHeader(void)
{
    open(aDefaultName);
    aFile->headerPrefix(1);
}


void Report::endOwnHeader(void)
{
    open(aDefaultName);
    aFile->headerPostfix(1);
}


void Report::changeFormat(Format p)
{
    if (aFormat != p)
    {
        aFormat = p;
        // also write new explanation lines, if we are in a file

        if (aFile != 0)
        {
            aFile->printLine(); // empty separating line

            ((MyFormattedOutputFile *) aFile)->changeIndentation(
                ExternalEntry::indentation(aFormat));

            writeExplanations();
        }
    }
}


void Report::open(const FileName pFileName)
{
    if (aFile == 0)
    {
        if (aLayout == ReportOptions::eWebPage)
        {
            aFile = new MyFluksWebPageOutputFile(FileName(pFileName, "htm"));
        }
        else
        {
            aFile = new MyFormattedOutputFile(
                       pFileName,
                       aWidth,
                       ExternalEntry::indentation(aFormat));
        }
        assert(aFile != 0);

        if (aPersonLegend == 0)
        {
            aPersonLegend = new PersonLegend;
            assert (aPersonLegend != 0);
        }
        else
        {
            aPersonLegend->reInitialise();
        }
        writeExplanations();
    }
    // else: file already open
}


void Report::writeExplanations(void)
{
    if (aLayout == ReportOptions::eReprintsInColumns)
    {
        // we don't bother in this case; the columns
        // would not be written correctly because reprint columns are
        // inbetween the entry columns.
    }
    else if (  (aLayout == ReportOptions::eWebPage)
            || (aLayout == ReportOptions::eDescriptionLayout)
            )
    {
        // no explanation needed
    }
    else
    {
        assert(aFile != 0);

        ExternalEntry lStory1(aFormat);
        ExternalEntry lStory2(aFormat);

        lStory1.fillExplanations(lStory2);

        lStory1.putExternal(*aFile);
        aFile->printLine();
        lStory2.putExternal(*aFile);
        aFile->printLine();

        aFile->printLine(); // empty separating line
    }
}


void Report::close(void)
{
    if (aFile != 0)
    {
        assert(aFile != 0);
        assert(aPersonLegend != 0);

        if (aLayout == ReportOptions::eWebPage)
        {
            aPersonLegend->writeWeb(*aFile);
        }
        else if (aLayout == ReportOptions::eDescriptionLayout)
        {
            // no legend needed!
        }
        else
        {
            aPersonLegend->write(*aFile, aInLatin1);
            // but not delete - it can be re-used in a next file
        }

        delete aFile;
        aFile = 0;
    }
}
