/* SERIESLI(st).CC, (c) Harry Fluks 1996

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-04-11  created
   96-08-07  checked
*/

#include <assert.h>
#include <ctype.h> // isalpha
#include "seriesli.h"
#include "header.h"
#include "buffers.h"

class SeriesListElement: public Listable
{
public:
    SeriesListElement(const char * pSeriesCode, const char * pFullName);
    ~SeriesListElement();

    virtual int compare(const char *);
    bool used(void) const;
    void update(void);
    void reInitialise(void);

    char * fullName(void) const { return aFullName; }
    char * seriesCode(void) const { return aSeriesCode; }

private:
    char * aSeriesCode;
    char * aFullName;
    bool aUsed;
};


SeriesListElement::SeriesListElement(
    const char * pSeriesCode,
    const char * pFullName)
:
    aUsed(FALSE)
{
    assert(pSeriesCode != 0);
    aSeriesCode = Buffers::newBuffer(1, strlen(pSeriesCode) + 1);
    strcpy(aSeriesCode, pSeriesCode);

    assert(pFullName != 0);
    aFullName = Buffers::newBuffer(2, strlen(pFullName) + 1);
    assert(aFullName != 0);
    strcpy(aFullName, pFullName);
}


SeriesListElement::~SeriesListElement()
{
    Buffers::deleteBuffer(1, aSeriesCode);
    Buffers::deleteBuffer(2, aFullName);
}


int SeriesListElement::compare(const char * pKey)
{
    assert(pKey != 0);
    assert(aSeriesCode != 0);
    return SortOrder::strcmp(aSeriesCode, pKey);
}


bool SeriesListElement::used(void) const
{
    return aUsed;
}


void SeriesListElement::update(void)
{
    aUsed = TRUE;
}


void SeriesListElement::reInitialise(void)
{
    aUsed = FALSE;
}


SeriesList::SeriesList(const CountryIndex & pCountry)
:
    aList(),
    aCountry(pCountry)
{
    readFromFile();
}


SeriesList::~SeriesList()
{
    Listable * l = aList.first();
    Listable * l2;
    while (l != 0)
    {
        l2 = l;
        l = aList.drop(l);  // remove from list
        delete l2;
    }
}


const char * SeriesList::find(const char * pIssueCode) const
{
    // returns the full name of the series, or null if not found

    static char lReturnBuffer[80];

    ENTRYCODEtype lSeriesCode;
    convertIssueCodeToSeriesCode(pIssueCode, lSeriesCode);
    assert(strlen(lSeriesCode) < ENTRYCODElength);

    SeriesListElement * lSeries =
        (SeriesListElement *) aList.find(lSeriesCode);

    if (lSeries == 0)
    {
        return 0;
    }
    else
    {
        sprintf(lReturnBuffer, "%s%s%s",
                               lSeries->fullName(),
                               (lSeriesCode[0] == ' ' ? "" : " "),
                               pIssueCode + strlen(lSeriesCode));
        return lReturnBuffer;
    }
}


void SeriesList::useReprintIssues(Entry & pEntry)
{
    ShortList & lShl = pEntry.reprint(aCountry);
    int lNumber = lShl.numberOfElements();
    if (lNumber != 0)
    {
        for (int i = 0; i < lNumber; i++)
        {
            add(lShl.element(i));
        }
    }
}


void SeriesList::writeFileLegend(MyExternalOutputFile & pOutputFile)
{
    bool lFirst = TRUE;

    SeriesListElement * lCurrent = (SeriesListElement *) aList.first();

    while (lCurrent != 0)
    {
        if (lCurrent->used())
        {
            if (lFirst)
            {
                pOutputFile.headerPrefix(3);
                pOutputFile.printF("Abbreviations of reprint issues from %s (\'%s\'):",
                     aCountry.name(),
                     aCountry.abbreviation());
                pOutputFile.headerPostfix(3);
                lFirst = FALSE;
            }
            pOutputFile.printF("%-10s = ", lCurrent->seriesCode());
            pOutputFile.putS(lCurrent->fullName());
            pOutputFile.printLine();
        }
        lCurrent = (SeriesListElement *) lCurrent->next();
    }
}


void SeriesList::reInitialise(void)
{
    SeriesListElement * lCurrent = (SeriesListElement *) aList.first();

    while (lCurrent != 0)
    {
        lCurrent->reInitialise();
        lCurrent = (SeriesListElement *) lCurrent->next();
    }
}


void SeriesList::readFromFile(void)
{
    MyDelimitedInputString lInputString;

    MyInputFile lFile(FileName("headers1", "ine"));

    Listable * lElem = 0;
    while (lInputString.read(lFile) != eEndOfFile)
    {
        Header lHeader;
        lHeader.scanInternal(lInputString);

        if (  (lHeader.level() == '1') // this should be true in headers1.ine
           && (lHeader.country() == aCountry)
           )
        {
            STORYCODEtype lBuffer;
            lHeader.storyCode().copyToChars(lBuffer, STORYCODElength);
            // we need our own buffer because we find() on it

            lElem = aList.find(lBuffer);
            if (lElem == 0)
            {
                lElem = new SeriesListElement(lBuffer, lHeader.title().zeroString());
                assert(lElem != 0);
                (void) aList.insert(lElem, lBuffer);
            }
        }
    }
}


void SeriesList::add(const char * pIssueCode)
{
    assert(pIssueCode != 0);

    ENTRYCODEtype lSeriesCode;
    convertIssueCodeToSeriesCode(pIssueCode, lSeriesCode);
    assert(strlen(lSeriesCode) < ENTRYCODElength);
    SeriesListElement * lIssue = (SeriesListElement *) aList.find(lSeriesCode);

    if (lIssue == 0)
    {
        // insert new Issue
        lIssue = new SeriesListElement(lSeriesCode, "???");
        assert(lIssue != 0);

        (void) aList.insert(lIssue, lSeriesCode);
    }
    else
    {
        lIssue->update();
    }
}


void SeriesList::convertIssueCodeToSeriesCode(
    const char * pIssueCode,
    char * pTarget)
{
    assert(pIssueCode != 0);
    assert(pTarget != 0);

    // skip everything behind the first non-space/non-alphabetic char
    // (but '&' should remain in the cleaned code)

    // examples:
    // BV.12 -> BV
    // 78-03 -> <nothing>
    // X78-03 -> X
    // DDMM 18 -> DDMM
    // W WDC 123 -> W WDC

    char * lLastSpace = 0; // can point into pTarget
    char * q = pTarget;

    // test on '*' and '\'': these are characters added to the reprint issue
    for (const char * p = pIssueCode;
         *p != '\0' && *p != '*' && *p != '\'';
         p++)
    {
        if (isalpha(*p) || *p == '&')
        {
            assert(q >= pTarget);
            assert(q < pTarget + ENTRYCODElength);
            *q = *p;
            q++;
        }
        else if (*p == ' ')
        {
            assert(q >= pTarget);
            assert(q < pTarget + ENTRYCODElength);
            lLastSpace = q;
            *q = *p;
            q++;
        }
        else
        {
            break; // EXIT FOR-LOOP
        }
    }

    if (lLastSpace != 0)
    {
        assert(lLastSpace >= pTarget);
        assert(lLastSpace < pTarget + ENTRYCODElength);
        *lLastSpace = '\0';
    }
    assert(q >= pTarget);
    assert(q < pTarget + ENTRYCODElength);
    *q = '\0'; // pointing in pTarget
}
