/* SHORTLIS(t).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   95-03-24  created
   96-08-16  checked (and uses Buffers object)
*/

#include <assert.h>
#include <string.h>
#include "shortlis.h"
#include "buffers.h"


ShortList::ShortList(void)
:
    aBuffer(0),
    aBufferEnd(0),
    aCommaSeparated(FALSE)
{
}


ShortList::~ShortList()
{
    clear();
}


bool ShortList::empty(void) const
{
    return (aBuffer == 0);
}


void ShortList::clear(void)
{
    if (aBuffer != 0)
    {
        Buffers::deleteBuffer(3, aBuffer);
        aBuffer = 0;
        aBufferEnd = 0;
        aCommaSeparated = FALSE;
    }
}


void ShortList::set(const char * pElement)
{
    if (  (pElement == 0)
       || (*pElement == '\0')
       )
    {
        return;
        // don't insert empty strings
    }

    clear();

    aBuffer = Buffers::newBuffer(3, strlen(pElement) + 1);
    strcpy(aBuffer, pElement);
    aBufferEnd = MyLib::atend(aBuffer);
    aCommaSeparated = TRUE;
}


void ShortList::setFixedPosString (
    const MyFixedPosInputString & pString, int pPos, int pMaxLength)
{
    clear();

    aBuffer = Buffers::newBuffer(3, pMaxLength + 1);
    pString.scanFixedPos (pPos, pMaxLength, aBuffer);
    if (*aBuffer == '\0')
    {
        clear();
    }
    else
    {
        aBufferEnd = MyLib::atend(aBuffer);
        aCommaSeparated = TRUE;
    }
}


void ShortList::insert(const char * pElement)
{
    if (  (pElement == 0)
       || (*pElement == '\0')
       )
    {
        return;
        // don't insert empty strings
    }

    if (aBuffer == 0)
    {
        set(pElement);
    }
    else
    {
        makeCommaSeparated();
        char * lBuffer = Buffers::newBuffer(4, strlen(pElement) + strlen(aBuffer) + 2);
        assert(lBuffer != 0);

        sprintf(lBuffer, "%s,%s", aBuffer, pElement);
        Buffers::deleteBuffer(4, aBuffer);
        aBuffer = lBuffer;
        aBufferEnd = MyLib::atend(aBuffer);
    }
}


void ShortList::insert(const PointerIntoInputString & p)
{
    if (p.empty())
    {
        return; // don't insert empty strings
    }

    if (aBuffer == 0)
    {
        aBuffer = Buffers::newBuffer(3, p.length() + 1);
        p.copyToChars(aBuffer, p.length());
        aCommaSeparated = TRUE;
        aBufferEnd = MyLib::atend(aBuffer);
    }
    else
    {
        makeCommaSeparated();
        char * lBuffer = Buffers::newBuffer(5, p.length() + strlen(aBuffer) + 2);
        assert(lBuffer != 0);

        sprintf(lBuffer, "%s,", aBuffer);
        char * lEnd = MyLib::atend(lBuffer);
        p.copyToChars(lEnd, p.length());

        Buffers::deleteBuffer(5, aBuffer);
        aBuffer = lBuffer;
        aBufferEnd = MyLib::atend(lEnd);
    }
}


bool ShortList::contains(const char * pElement) const
{
    // aBuffer is like "CB,CB?,CB),CB+,CB(1st)"
    //         either comma-separated, or null-separated
    // pElement can be "cleaned" (e.g. "CB")

    assert(pElement != 0);

    if (aBuffer == 0)
    {
        return FALSE;
    }

    const char * c = aBuffer;
    const char * d;

    while (c < aBufferEnd)
    {
        for (d = pElement; *d != '\0'; d++)
        {
            if (*d != *c)
            {
                break; // not equal
            }
            c++;
        }
        // next test looks like a "cleanPerson"...
        if (  *d == '\0'
           && (  *c == '\0' || *c == ',' || *c == '?'
              || *c == ')'  || *c == '(' || *c == '+'
              )
           )
        {
            return TRUE;
        }
        while (*c != ',' && *c != '\0') // go to next separator
        {
            c++;
        }
        if (c < aBufferEnd)
        {
            c++; // skip separator
        }
    }

    return FALSE;
}


int ShortList::numberOfElements(void) const
{
    if (aBuffer == 0)
    {
        return 0;
    }

    // number of elements = number of delimiters + 1
    // delimiter can be ',' or '\0'
    int lResult = 1;
    for (const char * c = aBuffer; c < aBufferEnd; c++)
    {
        if (*c == ',' || *c == '\0')
        {
            lResult++;
        }
    }
    return lResult;
}


const char * ShortList::element(int pIndex)
{
    if (aBuffer == 0)
    {
        return "";
    }
    makeNullSeparated(); // we'll return a pointer IN the buffer!
    const char * c = aBuffer;
    for (int i = 0; i < pIndex; i++)
    {
        // go to the (pIndex-1)th delimiter
        while (*c != '\0')
        {
            c++;
        }
        if (c < aBufferEnd)
        {
            c++;
        }
    }

    return c; // can be "empty string" if pIndex > our number of elements
}


const char * ShortList::delimitedString(void)
{
    if (aBuffer == 0)
    {
        return "";
    }
    makeCommaSeparated();
    return aBuffer;
}


void ShortList::concat(ShortList & pFrom)
{
    insert(pFrom.delimitedString());
}


void ShortList::makeCommaSeparated(void)
{
    if (aCommaSeparated)
    {
        return; // already OK
    }
    assert(aBuffer != 0);
    for (char * c = aBuffer; c < aBufferEnd; c++)
    {
        if (*c == '\0')
        {
            *c = ',';
        }
    }
    aCommaSeparated = TRUE;
}


void ShortList::makeNullSeparated(void)
{
    if (!aCommaSeparated)
    {
        return; // already OK
    }
    assert(aBuffer != 0);
    for (char * c = aBuffer; c < aBufferEnd; c++)
    {
        if (*c == ',')
        {
            *c = '\0';
        }
    }
    aCommaSeparated = FALSE;
}
