/* SIMILARS(tories).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-06-30  created
   96-09-03  checked
*/

#include "similars.h"

SimilarStories::SimilarStories(const FileName & p)
:
    aLastEntryRead(p),
    aNumber(0)
{
}


SimilarStories::~SimilarStories()
{
    for (int i = 0; i < aNumber; i++)
    {
        delete aEntry[i];
    }
    for (i = 0; i < aNumber; i++)
    {
        delete aInputString[i];
    }
    // not in 1 for-loop. All Entries should be deleted before the
    // first InputString is deleted!
}


void SimilarStories::goToMatch(const Entry & pEntry)
{
    if (aNumber > 0)
    {
        if (codesCompare(*aEntry[0], pEntry) == 0)
        {
            return; // current array still matches
        }

        // delete current arrays:
        for (int i = 0; i < aNumber; i++)
        {
            delete aEntry[i];
        }
        for (i = 0; i < aNumber; i++)
        {
            delete aInputString[i];
        }
        aNumber = 0;
    }

    while (codesCompare(aLastEntryRead.entry(), pEntry) < 0)
    {
        aLastEntryRead.readNext(cDeletePrevious);
    }

    // now fill array
    while (codesCompare(aLastEntryRead.entry(), pEntry) == 0)
    {
        aEntry[aNumber] = &aLastEntryRead.entry();
        aInputString[aNumber] = &aLastEntryRead.inputString();
        aNumber++;
        aLastEntryRead.readNext(! cDeletePrevious);
        // not cDeletePrevious: we will delete the previous entry later
    }
}


Entry * SimilarStories::findBestMatch(Entry & p) const
{
    if (aNumber == 0)
    {
        return 0;
    }

    Entry * lResult = 0;
    int lNumberOfMatches = 0;

    for (int i = 0; i < aNumber; i++)  // RETURNs in loop
    {
        if (aEntry[i]->sStoryCode == p.sStoryCode) 
        {
            return aEntry[i]; // exact match, what more do we want?
        }
        if (reasonableMatch(*aEntry[i], p))
        {
            lResult = aEntry[i];
            lNumberOfMatches++;
        }
    }
    if (lNumberOfMatches > 1)
    {
        return 0;
    }

    // at this point, we have no *exact* match, at most a reasonable match

    // never make a reasonable match for a W-coded story where '-..' is
    // specified
    // e.g. W WDC 123-01
    //      0123456789
    if (  (p.sStoryCode[0] == 'W')
       && (p.sStoryCode[9] == '-')
       && (  (p.sStoryCode[10] == '?')
          || (  (p.sStoryCode[10] >= '0')
             && (p.sStoryCode[10] <= '9')
             )
          )
       )
    {
        return 0;
    }

    return lResult;
}


int SimilarStories::codesCompare(const Entry & pStory, const Entry & pEntry) const
{
    if (pEntry.sStoryCode[0] == 'W')
    {
        // exact equality up to 9th character
        // W WDC 123-01
        // 0123456789
        for (int i = 0; i < 9; i++)
        {
            if (pStory.sStoryCode[i] < pEntry.sStoryCode[i])
            {
                return -1;
            }
            else if (pStory.sStoryCode[i] > pEntry.sStoryCode[i])
            {
                return 1;
            }
        }
    }
    else
    {
        return pStory.sStoryCode.compare(pEntry.sStoryCode);
        // ** more advanced matching for I-codes?
    }
    return 0;
}


bool SimilarStories::reasonableMatch(Entry & pStory, Entry & pEntry) const
{
    // test fields: pages, page layout, plot/writ/art/ink, hero
    // if field filled in in pEntry, then it *should* be a specified in pStory
    // and at least one field should match

    if (! (pEntry.sPages.empty()))
    {
        if (! (pStory.sPages.equalTo(pEntry.sPages)))
        {
            return FALSE;
        }
    }
    if (pStory.sPageLayout != pEntry.sPageLayout)
    {
        return FALSE;
    }

    ShortListFieldIndex s;
    int lNumberOfMatchingFields = 0;
    while (s.next())
    {
        switch(s.number())
        {
        case ePlotter: case eWriter: case eArtist: case eInker:
        case eHero:
            if (  (pStory.field(s).numberOfElements() > 1)
               || (pEntry.field(s).numberOfElements() > 1)
               )
            {
                // more than 1 element is too complicated
                return FALSE;
            }
            else if (pEntry.field(s).numberOfElements() == 1)
            {
                if (pStory.field(s).numberOfElements() == 1)
                {
                    if (strcmp(pStory.field(s).element(0),
                               pEntry.field(s).element(0)) == 0)
                    {
                        lNumberOfMatchingFields++;
                    }
                    else
                    {
                        return FALSE;
                    }
                }
                else
                {
                    // no story field, so entry has more information?
                    return FALSE;
                }
            }
            break;
        default:
            ; // niente
        }
    }

    return lNumberOfMatchingFields > 0;
}
