/* BARKSWEB(pages).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   97-06-26  created
*/

#include <assert.h>
#include <ctype.h> // tolower
#include "barksweb.h"
#include "webentry.h"


class BarksWebPagesElement: public Listable
{
public:
    BarksWebPagesElement(char * pStoryCode, char * pDescription);
    ~BarksWebPagesElement();

    virtual int compare(const char *);
    bool match(MyDelimitedInputString *);
    void write(
        MyBarksWebPageOutputFile &,
        const FileName &,
        PersonLegend &,
        const CountrySet &);

    char * key(void) { return aKey; }

private:
    const char * subDirectory(void);
    const char * relativeSubDirectory(const char *);
    void codeToFileName(char *);

    char aKey[5]; // to keep things in right order
    MyDelimitedInputString * aInputString; // for storing the Entry
    char aStoryCode[STORYCODEsize];
    char * aDescription;

    static int aGlobalKey;
};


int BarksWebPagesElement::aGlobalKey = 0;


BarksWebPagesElement::BarksWebPagesElement(char * pStoryCode, char * pDescription)
:
    aInputString(0),
    aDescription(0)
{
    aGlobalKey++;
    assert(aGlobalKey < 10000);
    sprintf(aKey, "%04d", aGlobalKey);

    aDescription = new char[strlen(pDescription) + 1];
    assert(aDescription != 0);
    strcpy(aDescription, pDescription);

    assert(strlen(pStoryCode) <= STORYCODElength);
    strcpy(aStoryCode, pStoryCode);
}


BarksWebPagesElement::~BarksWebPagesElement()
{
    if (aDescription != 0)
    {
        delete aDescription;
    }
    if (aInputString != 0)
    {
        delete aInputString;
    }
}


int BarksWebPagesElement::compare(const char * p)
{
    return strcmp(aKey, p);
}


bool BarksWebPagesElement::match(MyDelimitedInputString * p)
{
    assert(p != 0);
    if (p->compare(aStoryCode) == 0)
    {
        aInputString = p;
        return TRUE;
    }
    return FALSE;
}


void BarksWebPagesElement::write(
    MyBarksWebPageOutputFile & pFile,
    const FileName & pCurrentFileName,
    PersonLegend & pPersonLegend,
    const CountrySet & pReprintCountrySet)
{
    if (aInputString == 0)
    {
        MyLib::log("Story code (%s) not in database", aStoryCode);
        pFile.printF("Story code (%s) not in database!\n", aStoryCode);
        return;
    }

    InternalEntry lEntry;
    lEntry.scan(*aInputString);

    lEntry.field(eOldCode).clear();  // never write this one ***
    lEntry.field(eOldCodeA).clear(); // never write this one ***
    lEntry.field(eOldCodeB).clear(); // never write this one ***
    lEntry.field(eXRef).clear(); // never write this one ***
    lEntry.field(eXRefd).clear(); // never write this one ***

    char lFileNameBuffer[81];
    const char * lCurrentDirectory = subDirectory();
    codeToFileName(lFileNameBuffer);
    FileName lFileName(lCurrentDirectory, lFileNameBuffer, "htm");

    // print something to pFile, the "index" file
    pFile.putS("<tr><td>");

    pFile.printF("<a href=\"%s\"> ", lFileName.fullNameNoDir());
    pFile.putS(lEntry.sStoryCode);
    pFile.putS("</a></td><td>");

    if (lEntry.sTitlePrefix.isRealTitle())
    {
        pFile.putS(" &quot;");
    }
    else
    {
        pFile.putS(" &lt;");
    }
    pFile.putS(lEntry.title());
    if (lEntry.sTitlePrefix.isRealTitle())
    {
        pFile.putS("&quot;");
    }
    else
    {
        pFile.putS("&gt;");
    }
    pFile.putS("</td></tr>");
    // that's all!

    MyBarksWebPageOutputFile lFile(lFileName);

    // some links to other pages - note that we assume we're in a subdirectory!

    lFile.printF("<p>\n");
    lFile.printF("<font size=2><b><a href=\"../welcome.htm\">HOME</a></font></b>\n");
    lFile.printF(" / <font size=2><b><a href=\"../%s\">related index</a></font></b>\n",
                 pCurrentFileName.fullNameNoDir());
    lFile.printF(" / <font size=2><b><a href=\"../mainindx.htm\">main index</a></font></b>\n");
    lFile.printF(" / <font size=2><b><a href=\"../help.htm\">help</a></font></b>\n");

    BarksWebPagesElement * lPrevious = (BarksWebPagesElement *) previous();
    if (lPrevious != 0)
    {
        char lPrevFileNameBuffer[81];
        const char * lPrevDir = lPrevious->relativeSubDirectory(lCurrentDirectory);
        lPrevious->codeToFileName(lPrevFileNameBuffer);
        lFile.printF(" / <font size=2><b><a href=\"%s%s.htm\">previous entry</a></font></b>\n",
             lPrevDir, lPrevFileNameBuffer);
    }
    BarksWebPagesElement * lNext = (BarksWebPagesElement *) next();
    if (lNext != 0)
    {
        char lNextFileNameBuffer[81];
        const char * lNextDir = lNext->relativeSubDirectory(lCurrentDirectory);
        lNext->codeToFileName(lNextFileNameBuffer);
        lFile.printF(" / <font size=2><b><a href=\"%s%s.htm\">next entry</a></font></b>\n",
             lNextDir, lNextFileNameBuffer);
    }
    lFile.printF("<p><br>\n");

    // the data itself:

    if (lEntry.sPageLayout.isStory())
    {
        lFile.printF("<b>Category:</b> comic book story");
        lFile.printLine();
        // Layout will be printed after "Pages".
    }
    else
    {
        lFile.printF("<b>Category:</b> %s", lEntry.sPageLayout.explanation());
        lFile.printLine();
    }

    lFile.printF("<b>Code:</b> ");
    lFile.putS(lEntry.sStoryCode);
    lFile.printLine();

    // first, the hero(es)
    if (! lEntry.field(eHero).empty())
    {
        lFile.printF("<b>Hero:</b> ");
        WebEntry::printShortListFullNames(lEntry, lFile, eHero, &pPersonLegend);
        lFile.printLine();
    }

    if (lEntry.title().empty())
    {
    }
    else
    {
        lFile.putS("<b>");
        lFile.putS(lEntry.sTitlePrefix.explanation());
        lFile.putS(": </b>");
        if (lEntry.sTitlePrefix.isRealTitle())
        {
            lFile.putS(" &quot;");
            lFile.putS(lEntry.title());
            lFile.putS("&quot; ");
        }
        else
        {
            lFile.putS(lEntry.title());
        }
        lFile.printLine();
    }

    if (! lEntry.sPages.empty())
    {
        lFile.printF("<b>Pages:</b> %s", lEntry.sPages.theString());
        lFile.printLine();
    }

    if (lEntry.sPageLayout.isStory())
    {
        if (lEntry.sPageLayout.isDefault())
        {
            lFile.printF("<b>Layout:</b> 4 rows per page");
        }
        else
        {
            lFile.printF("<b>Layout:</b> %s", lEntry.sPageLayout.explanation());
        }
        lFile.printLine();
    }

    if (!lEntry.comment().empty())
    {
        lFile.printF("<b>Comment:</b> %s", lEntry.comment().element());
        lFile.printLine();
    }

    ShortListFieldIndex s;
    while (s.next())
    {
        if (lEntry.field(s).empty())
        {
            switch (s.number())
            {
            case eWriter: case eArtist:
                lFile.printF("<b>%s:</b> ?", s.explanation());
                lFile.printLine();
                break;
            }
        }
        else
        {
            switch (s.number())
            {
            case ePlotter: case eWriter: case eArtist: case eInker:
                lFile.printF("<b>%s:</b> ", s.explanation());
                WebEntry::printShortListFullNames(lEntry, lFile, s, &pPersonLegend);
                lFile.printLine();
                break;
            }
        }
    }

    ShortListFieldIndex s2;
    while (s2.next())
    {
        if (! lEntry.field(s2).empty())
        {
            switch (s2.number())
            {
            case ePlotter: case eWriter: case eArtist: case eInker:
            case eHero:
                break;
            default:
                lFile.printF("<b>%s:</b> ", s2.explanation());
                WebEntry::printShortListFullNames(lEntry, lFile, s2, &pPersonLegend);
                lFile.printLine();
                break;
            }
        }
    }

    lFile.printF("<hr>\n");

    // add data from the "des-file"
    lFile.printF("%s\n", aDescription);

    CountryIndex ci;
    bool lFirst = TRUE;
    while (ci.next())
    {
        if (pReprintCountrySet.includes(ci))
        {
            if (! lEntry.reprint(ci).empty())
            {
                if (lFirst)
                {
                    lFile.printF("<hr>\n");
                    lFile.printF("<b>Publications:</b>");
                    lFile.printLine();
                    lFirst = FALSE;
                }
                lFile.printLine();
                WebEntry::printReprintFullNames(lEntry, lFile, ci, &pPersonLegend);
            }
        }
    }
}


const char * BarksWebPagesElement::subDirectory(void)
{
    const struct
    {
        const char * sCodeStart;
        int sLength;
        const char * sSubDirectory;
    } cSubDirArray[] =
    {
        {"W WDC  ", 7, "wdc0"},   // this array especially for Barks...
        {"W WDC 1", 7, "wdc1"},
        {"W WDC 2", 7, "wdc2"},
        {"W WDC 3", 7, "wdc3"},
        {"W US ", 5, "us"},
        {"W OS ", 5, "os"},
        {"W", 1, "rest"},
        {"", 0, "various"},
        {0, 0}
    };

    for (int i = 0; cSubDirArray[i].sCodeStart != 0; i++)
    {
        if (strncmp(aStoryCode, cSubDirArray[i].sCodeStart,
                                cSubDirArray[i].sLength) == 0)
        {
            return cSubDirArray[i].sSubDirectory; // EXIT FOR-LOOP
        }
    }
    return ".";
}


const char * BarksWebPagesElement::relativeSubDirectory(const char * p)
{
    const char * c = subDirectory();
    if (strcmp(c, p) == 0)
    {
        return "";
    }
    else
    {
        static char lBuffer[80];
        sprintf(lBuffer, "../%s/", c);
        return lBuffer; // pointing to static data! call only once at a time
    }
}


void BarksWebPagesElement::codeToFileName(char * pFileNameBuffer)
{
    char * c = aStoryCode;
    char * d = pFileNameBuffer;

    if (*c == 'W')
    {
        c++; // skip initial 'W' of the western codes. We need the space
             // for an 8-char file name.
    }
    else if (c[0] == 'C' && c[1] == 'B')
    {
        c++; c++; // skip 'CB': 'CB OIL 125-B' can become too long
    }

    while (*c != '\0')
    {
        switch (*c)
        {
        case '-':
        case ' ':
        case '\'':
            break;  // SKIP, also to make sure that the file name will not
                    // become too long
        case '?':
            *d = 'q'; // don't skip, because otherwise file names
                      // may be non-unique, e.g.
                      // W US   4-01 == W US  40-?1  (!)
            d++;
            break;
        case '&':
            *d = 'n';
            d++;
            break;
        default:
            *d = tolower(*c);
            d++;
        }
        c++;
    }
    *d = '\0';
    if (strlen(pFileNameBuffer) > 8)
    {
        MyLib::log("File name (%s) too long for DOS", pFileNameBuffer);
    }
}


BarksWebPages::BarksWebPages(void)
{
    aReadBuffer = new char[cBufferSize];
    assert(aReadBuffer != 0);
}


BarksWebPages::~BarksWebPages()
{
    delete aReadBuffer;
}


void BarksWebPages::read(const FileName & pFileName)
{
    MyInputFile lInputFile(pFileName);
    FILE * lFile = lInputFile.theFile();

    char lStoryCodeBuffer[STORYCODEsize];
    assert(aReadBuffer != 0);

    char c;

    // skip everything till first '#'
    while (((c = fgetc(lFile)) != EOF) && (c != '#'))
    {
        // skip char
    }

    while (c != EOF)
    {
        assert (c == '#');

        char * d = lStoryCodeBuffer;
        int i = 0;
        while (((c = fgetc(lFile)) != EOF) && (c != '\n') && i < STORYCODElength)
        {
            *d++ = c;
            i++;
        }
        *d = '\0';

        // now, read the rest
        d = aReadBuffer;
        i = 0;
        while (((c = fgetc(lFile)) != EOF) && (c != '#') && i < cBufferSize)
        {
            *d++ = c;
            i++;
        }
        *d = '\0';

        // and add an element

        BarksWebPagesElement * lNewElement =
            new BarksWebPagesElement(lStoryCodeBuffer, aReadBuffer);
        assert(lNewElement != 0);
        aList.insert(lNewElement, lNewElement->key());

        Progress::addRead(); // well, more than one line in fact, but
                             // counting them is too much work
    }
}


void BarksWebPages::addEntries(const FileName & pFileName)
{
    MyInputFile lMyInputFile(pFileName);

    MyDelimitedInputString * lCurrentInputString = new MyDelimitedInputString;
    assert(lCurrentInputString != 0);

    while (lCurrentInputString->read(lMyInputFile) != eEndOfFile)
    {
        if (match(lCurrentInputString))
        {
            lCurrentInputString = new MyDelimitedInputString;
        }
        // else: use the old one!
    }
    delete lCurrentInputString; // the last one was EOF
}


void BarksWebPages::write(
    const FileName & pFileName,
    PersonLegend & pPersonLegend,
    const CountrySet & pReprintCountrySet)
{
    MyBarksWebPageOutputFile lIndexFile(pFileName);

    lIndexFile.printF("<table>");

    Listable * l = aList.first();
    while (l != 0)
    {
        BarksWebPagesElement * lElement = (BarksWebPagesElement *) l;

        lElement->write(lIndexFile, pFileName, pPersonLegend, pReprintCountrySet);

        l = l->next();
    }
    lIndexFile.printF("</table>");
}


bool BarksWebPages::match(MyDelimitedInputString * p)
{
    Listable * l = aList.first();
    while (l != 0)
    {
        if (((BarksWebPagesElement *)l)->match(p))
        {
            return TRUE;
        }
        l = l->next();
    }
    return FALSE;
}
