/* BULKOUTP(ut).CC, (c) Harry Fluks 1994

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   STORY-OUTPUT: Merge the original DBS (stories) files with the (internal)
   files that contain all reprint information.

   94-09-23  created
   94-06-26  completed
   94-10-05  call to cleanReprintEntryCode added
   94-11-10  header lines added in creator output
   95-01-16  covers and stories in separate files; files
             have better names; output layout changed
   95-01-24  header text with program id in output files
             for 1 creator;
             annoying core dump fixed (closed 1 file too much)
   95-03-15  renamed from DIZNISTO; format of intermediate file
             has changed, and other changes
   95-03-29  output for one creator moved to another source file
   96-06-19  selection on countries possible

   ENTRY-OUTPUT: Merge the original DBI files with the (temp) files that
   contain all reprint information.

   95-03-16  created
   95-06-22  person legends added
   96-01-07  uses LastEntryRead and Report

   HEADER-OUTPUT: produce a list of already indexed issues

   95-04-25  created
   96-07-18  renamed from "done" to "headerOutput". IndexedIssues in
             separate file.

   BULKOUTPUT:

   96-08-28  created & checked (combination of STORYOUT, ENTRYOUT, HEADEROUT)
   97-10-14  legend output added
*/

#include "dbfiles.h"
#include "lastentr.h"
#include "bulkoutp.h"
#include "indexedi.h"
#include "person.h"


class StoryReport: public Report
{
public:
    StoryReport(const FileName & pDefaultName, const ReportOptions &, bool);

    void processOneFile(LastEntryRead & pLastStoryRead,
                        const FileName &);
};


StoryReport::StoryReport( const FileName & pDefaultName,
                          const ReportOptions & pOptions,
                          bool pInLatin1)
:
    Report(FileName(pDefaultName, "stories"), pOptions, pInLatin1)
{
}


void StoryReport::processOneFile(LastEntryRead & pLastStoryRead,
                                 const FileName & p)
{
    MyInputFile lInputFile(p);

    MyFixedPosInputString lInputString;

    InputLineType lType;
    while ((lType = lInputString.read(lInputFile)) != eEndOfFile)
    {
        switch (lType)
        {
        case eEntryLine:
        case eWesternReprintLine:
            // we know we aren't in a Western file, so act just as if
            // eWesternReprintLine also is a normal entry line
            {
                ExternalEntry lEntry(aFormat);
                lEntry.scanExternal(lInputString, cCodesOnly);
                if (lEntry.sStoryCode == pLastStoryRead.entry().sStoryCode)
                {
                    writeEntry(pLastStoryRead.entry());

                    pLastStoryRead.readNext(cDeletePrevious);
                }
                else
                {
                    // should not occur
                    MyLib::log("Story not in stories file! (See ~ in output)");
                    lEntry.setUnsolved("~");
                    writeEntry(lEntry); // will be incomplete
                }
            }
            break;
        case eHeaderLine:
            {
                Header lHeader;
                lHeader.scanExternal(lInputString);
                if (lHeader.level() == '0')
                {
                    // open new output file, close old one
                    TITLEtype lTitle;
                    lHeader.title().copyToChars(lTitle, TITLElength);
                    newFile(lTitle);
                }
                else
                {
                    writeHeader(lHeader);
                }
            }
            break;
        default:
            theOutputFile().putInputString(lInputString);
            theOutputFile().printLine();
        }
    }
}


class EntryReport: public Report
{
public:
    EntryReport(const FileName & pDefaultName,
                const ReportOptions &,
                bool pInLatin1);

    void processOneFile(const FileName &);
};


EntryReport::EntryReport(
    const FileName & pDefaultName,
    const ReportOptions & pOptions,
    bool pInLatin1)
:
    Report(FileName(pDefaultName, "issues"), pOptions, pInLatin1)
{
}


void EntryReport::processOneFile(const FileName & p)
{
    MyInputFile lInputFile(p);
    LastEntryRead lLastEntryRead(FileName(p, cStep4Extension));

    MyFixedPosInputString lInputString;
    InputLineType lType;
    while ((lType = lInputString.read(lInputFile)) != eEndOfFile)
    {
        switch (lType)
        {
        case eEntryLine:
        case eWesternReprintLine:
            {
                ExternalEntry lEntry(aFormat);
                lEntry.scanExternal(lInputString, cCodesOnly);

                if (lEntry.sEntryCode == lLastEntryRead.entry().sEntryCode)
                {
                    writeEntry(lLastEntryRead.entry());
                    lLastEntryRead.readNext(cDeletePrevious);
                }
                else
                {
                    static bool lLogged = FALSE;
                    if (!lLogged)
                    {
                        MyLib::log(
"Entries lost in the process! - see ~ in output file (run dizni -x inputcheck)");
                        lLogged = TRUE;
                    }
                    lEntry.setUnsolved("~");
                    // only occurs if entry files are not sorted properly
                    writeEntry(lEntry); // will be incomplete
                }
            }
            break;
        case eHeaderLine:
            {
                Header lHeader;
                lHeader.scanExternal(lInputString);
                if (lHeader.level() == '0')
                {
                    // open new output file, close old one
                    TITLEtype lTitle;
                    lHeader.title().copyToChars(lTitle, TITLElength);
                    newFile(lTitle);
                }
                else
                {
                    writeHeader(lHeader);
                }
            }
            break;
        default:
            theOutputFile().putInputString(lInputString);
            theOutputFile().printLine();
        } // end switch
    } // end while
}


void BulkOutput::processAllStoryFiles(ReportOptions & pOptions)
{
    LastEntryRead lLastStoryRead(FileName("stories", "ins"));

    DBSFileIndex lFile;
    while (lFile.next())
    {
        if (lFile.isWestern())
        {
            // skip all Western entries in the stories.ins file!
            // we happen to know that Western entries start with a 'W' or 'C'
            // ('C' = special Carl Barks entries)
            while (  (lLastStoryRead.entry().sStoryCode.startsWithWSpace())
                  || (lLastStoryRead.entry().sStoryCode[0] == 'C')
                  )
            {
                lLastStoryRead.readNext(cDeletePrevious);
            }
        }
        else
        {
            const bool cInLatin1 = FALSE; // ** no Latin1 in story files
            pOptions.aFormat = lFile.format();
            StoryReport lReport(lFile.fileName(), pOptions, cInLatin1);
            lReport.processOneFile(lLastStoryRead, lFile.fileName());
        }
    }
}


void BulkOutput::processAllEntryFiles(
    ReportOptions & pOptions,
    const CountrySet & pCountrySet)
{
    DBIFileIndex lFileIndex;
    while (lFileIndex.next())
    {
        if (pCountrySet.includes(lFileIndex.country()))
        {
            if (lFileIndex.country() == eWesternCountry)
            {
                pOptions.aReprintCountrySet.allCountries();
                pOptions.aFormat = eWesternDBIFormat;
            }
            else
            {
                pOptions.aReprintCountrySet.clear();
                pOptions.aReprintCountrySet.insert(lFileIndex.country());
                pOptions.aFormat = eDBIFormat;
            }

            EntryReport lReport(lFileIndex.fileName(),
                                pOptions,
                                lFileIndex.inLatin1());

            lReport.processOneFile(lFileIndex.fileName());
        }
    }
}


void BulkOutput::processAllHeaderFiles(
    const char * pIndexer,
    const CountrySet & pCountrySet,
    int pWidth)
{
    assert(pIndexer != 0);
    MyInputFile lInputFile (FileName("headers", "ine"));
    MyFormattedOutputFile lOutputFile
        (FileName("indexed-issues", "legend"), pWidth,
        15, // indentation
        3000); // buffer size.
    // we need considerably more buffer space than MyF.OutputFile has by
    // default. (Especially the range of "One Shots" is big.)

    IndexedIssueRange * lIndexedIssueRange = 0;
    CountryIndex lCurrentCountry(eNoCountry);

    MyDelimitedInputString lInputString;
    while (lInputString.read(lInputFile) != eEndOfFile)
    {
        Header lHeader;
        lHeader.scanInternal(lInputString);

        switch (lHeader.level())
        {
        case '0':
            // file header, of no interest
            break;
        case '1':
        case '2':
            // new level, print old data
            if (lIndexedIssueRange != 0)
            {
                if (pCountrySet.includes(lCurrentCountry))
                {
                    lIndexedIssueRange->write(lOutputFile);
                }
                delete lIndexedIssueRange;
            }
            lIndexedIssueRange = new IndexedIssueRange(lHeader);
            assert(lIndexedIssueRange != 0);

            if (lHeader.country() != lCurrentCountry)
            {
                lCurrentCountry = lHeader.country();
                if (pCountrySet.includes(lCurrentCountry))
                {
                    lOutputFile.printLine();
                    lOutputFile.printF(
                        "Completely indexed issues of %s",
                        lCurrentCountry.name());
                    if (pIndexer[0] != '\0')
                    {
                        lOutputFile.printF(", indexed by %s", pIndexer);
                    }
                    else
                    {
                        lOutputFile.printLine();
                        lOutputFile.printF(
                            "(Other issues may be unregistered, or indexed partly)");
                    }
                    lOutputFile.printLine();
                    lOutputFile.printLine();
                }
            }

            break;
        case '3':
            if (lIndexedIssueRange == 0)
            {
                MyLib::log("Level 3 header (%s) without level 1 or 2 header",
                           lHeader.storyCode().zeroString());
            }
            else
            {
                if (lHeader.field(eIndexer).contains("-"))
                {
                    lIndexedIssueRange->addToUnindexed(lHeader.storyCode());
                }
                else if (  lHeader.field(eIndexer).contains(pIndexer)
                        || (pIndexer[0] == '\0')
                        )
                {
                    lIndexedIssueRange->addNumber(lHeader.storyCode());
                }
            }
            break;
        default:
            MyLib::log("Header with level '%c' skipped", lHeader.level());
        }
    }
    // print last data
    if (lIndexedIssueRange != 0)
    {
        if (pCountrySet.includes(lCurrentCountry))
        {
            lIndexedIssueRange->write(lOutputFile);
        }
        delete lIndexedIssueRange;
    }
}


void BulkOutput::processAllLegendFiles(int pWidth)
{
    processOneLegendFile("heroes", pWidth);
    processOneLegendFile("creators", pWidth);
    processOneLegendFile("subserie", pWidth);
    processOneLegendFile("indexers", pWidth);
}


void BulkOutput::processOneLegendFile(const char * pName, int pWidth)
{
    MyInputFile lInputFile (FileName(pName, "inl"));
    MyFormattedOutputFile lOutputFile
        (FileName(pName, "legend"), pWidth,
        15); // indentation
    lOutputFile.headerPrefix(1);
    lOutputFile.printF("Legend of %s in the Disney comics database",
                       pName);
    lOutputFile.headerPostfix(1);

    MyDelimitedInputString lInputString;
    while (lInputString.read(lInputFile) != eEndOfFile)
    {
        Person lPerson;
        lPerson.scanInternal(lInputString, TRUE); // TRUE: complete person
        lPerson.fullOutput(lOutputFile);
    }
}
