/* COMMANDL(INE).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-05-30  created
   96-08-06  checked
*/

#include <assert.h>
#include <string.h>
#include <stdlib.h> // atoi
#include <stdio.h> // fprintf to stderr
#include "commandl.h"

CommandLine::CommandLine(int argc, char * argv[])
:
    aOptionStructArray(0),
    aNumberOfOptions(0),
    aAllocatedNumberOfOptions(argc - 1),
    aErrorMessage(0),
    aErrorOption(' ')
{
    if (aAllocatedNumberOfOptions > 0)
    {
        aOptionStructArray = new OptionStruct[aAllocatedNumberOfOptions];
        // maybe too many, but certainly enough
        assert(aOptionStructArray != 0);
    }

    OptionStruct * lCurrentOption = 0;

    for (int i = 1; i < argc; i++)
    {
        if (argv[i][0] == '-')
        {
            // new option
            aErrorOption = ' ';

            if (strlen(argv[i]) < 2)
            {
                aErrorMessage = "'-' without letter";
            }
            else
            {
                aErrorOption = argv[i][1];
                assert (aNumberOfOptions < aAllocatedNumberOfOptions);

                lCurrentOption = &aOptionStructArray[aNumberOfOptions];
                aNumberOfOptions++;

                lCurrentOption->sOption = argv[i][1];
                lCurrentOption->sChecked = FALSE;

                if (strlen(argv[i] + 2) > cMaxArgumentLength)
                {
                    strcpy(lCurrentOption->sArgument, "");
                    aErrorMessage = "argument TOO long";
                }
                else
                {
                    strcpy(lCurrentOption->sArgument, argv[i] + 2);
                    // might be empty string
                }
            }
        }
        else
        {
            // add to current option
            if (lCurrentOption == 0)
            {
                aErrorMessage = "argument without '-'";
            }
            else
            {
                if (strlen(lCurrentOption->sArgument) +
                    strlen(argv[i]) + 1 >= cMaxArgumentLength)
                {
                    aErrorMessage = "argument too long";
                }
                else
                {
                    if (lCurrentOption->sArgument[0] != '\0')
                    {
                        strcat(lCurrentOption->sArgument, " ");
                    }
                    strcat(lCurrentOption->sArgument, argv[i]);
                }
            }
        }
    }
}


CommandLine::~CommandLine()
{
    if (aAllocatedNumberOfOptions > 0)
    {
        delete [] aOptionStructArray;
    }
}


bool CommandLine::present(Option pOption)
{
    return (findOption(pOption) != 0);
}


const char * CommandLine::stringArgument(Option pOption, const char * pDefault)
{
    OptionStruct * lStruct = findOption(pOption);
    if (lStruct == 0)
    {
        return pDefault;
    }
    else
    {
        return lStruct->sArgument;
    }
}


int CommandLine::integerArgument(Option pOption, int pDefault)
{
    OptionStruct * lStruct = findOption(pOption);
    if (lStruct == 0)
    {
        return pDefault;
    }
    else
    {
        return atoi(lStruct->sArgument);
    }
}


void CommandLine::check( Option pOption,
                         OptionType pOptionType,
                         ArgumentType pArgumentType)
{
    if (aErrorMessage != 0)
    {
        return; // no need to check
    }

    OptionStruct * lStruct = findOption(pOption);

    aErrorOption = pOption;
    if (lStruct == 0)
    {
        if (pOptionType == eMandatory)
        {
            aErrorMessage = "missing option";
        }
    }
    else
    {
        lStruct->sChecked = TRUE;

        // check argument
        switch (pArgumentType)
        {
        case eStringArgument:
            if (lStruct->sArgument[0] == '\0')
            {
                aErrorMessage = "option needs argument";
            }
            break;
        case eIntegerArgument:
            if (lStruct->sArgument[0] == '\0')
            {
                aErrorMessage = "option needs integer argument";
            }
            else
            {
                int i = atoi(lStruct->sArgument);
                if (i == 0)
                {
                    aErrorMessage = "non-integer or zero argument";
                }
            }
            break;
        case eNoArgument:
            if (lStruct->sArgument[0] != '\0')
            {
                aErrorMessage = "argument should not be given";
            }
        }
    }
}


bool CommandLine::printError(const char * pUsage)
{
    if (aErrorMessage == 0)
    {
        // look for any unspecified options
        for (int i = 0; i < aNumberOfOptions; i++)
        {
            if (!(aOptionStructArray[i].sChecked))
            {
                aErrorOption = aOptionStructArray[i].sOption;
                aErrorMessage = "unknown option";
            }
        }
    }

    if (aErrorMessage != 0)
    {
        fprintf(stderr, "Error in option -%c:\n", aErrorOption);
        fprintf(stderr, "%s\n\n", aErrorMessage);
        fprintf(stderr, "%s\n", pUsage);
        return TRUE;
    }
    return FALSE;
}


OptionStruct * CommandLine::findOption(Option pOption)
{
    for (int i = 0; i < aNumberOfOptions; i++)
    {
        if (aOptionStructArray[i].sOption == pOption)
        {
            return &aOptionStructArray[i];
        }
    }
    return 0; // not found
}


#ifdef COMMTEST
// a main function, for testing
int main (int argc, char * argv[])
{
    fprintf(stderr, "\n\nCOMMTEST\n\n");

    CommandLine lc(argc, argv);

    lc.check('a', eMandatory, eNoArgument);
    lc.check('b', eMandatory, eStringArgument);
    lc.check('c', eOptional, eNoArgument);
    lc.check('d', eOptional, eStringArgument);
    lc.check('e', eOptional, eIntegerArgument);

    if (lc.printError("TEST usage text"))
    {
        fprintf(stderr, "ERROR\n");
    }
    else
    {
        fprintf(stderr, "Option a\n");
        fprintf(stderr, "Option b: string arg (%s)\n", lc.stringArgument('b'));
        if (lc.present('c'))
        {
            fprintf(stderr, "Option c given\n");
        }
        if (lc.present('d'))
        {
            fprintf(stderr, "Option d given with string arg (%s)\n",
                            lc.stringArgument('d'));
        }
        if (lc.present('e'))
        {
            fprintf(stderr, "Option e given with int arg (%d)\n",
                            lc.integerArgument('e'));
        }

        fprintf(stderr, "OK\n");
    }
    return 0;
}
#endif