/* COUNTRY.CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-08-10  created
   96-08-11  checked
*/

#include <assert.h>
#include <string.h>
#include "country.h"
#include "shortlis.h"

static struct
{
    const char * sName;
    const char * sAbbreviation;
    const char * sReprintPrefix;
    const char * sReprintTitlePrefix;
} cCountryStruct[] =
{
    { "Gladstone-Disney", "us",  "rp-us", "Rus" },
    { "Western",          "w",   "rp-w",  "Rw"  },
    { "Netherlands",      "nl",  "rp-nl", "Rnl" },
    { "Sweden",           "se",  "rp-se", "Rse" },
    { "Denmark",          "dk",  "rp-dk", "Rdk" },
    { "Germany",          "de",  "rp-de", "Rde" },
    { "Finland",          "fi",  "rp-fi", "Rfi" },
    { "France",           "fr",  "rp-fr", "Rfr" },
    { "Norway",           "no",  "rp-no", "Rno" },
    { "UK",               "uk",  "rp-uk", "Ruk" },
    { "Various",          "co",  "rp-co", "Rco" }, // used to be Columbia
    { "Italy",            "it",  "rp-it", "Rit" },
    { "??",               "??",  "rp-??", "??"  }, // eLastCountry, not in use
    { "None",             "n",   "rp-n",  "??"  }
};


CountryIndex::CountryIndex(void)
:
    aEnum(eNoCountry)
{
}


CountryIndex::CountryIndex(CountryEnum p)
:
    aEnum(p)
{
    assert(aEnum < eLastCountry || aEnum == eNoCountry);
}


CountryIndex::CountryIndex(const CountryIndex & p)
:
    aEnum(p.aEnum)
{
}


CountryIndex::CountryIndex(const char * p)
:
    aEnum(eNoCountry)
{
    while (next())
    {
        if (strcmp(p, abbreviation()) == 0)
        {
            // aEnum set to the right one
            return;
        }
    }

    aEnum = eNoCountry;
}


CountryIndex::CountryIndex(const PointerIntoInputString & p)
:
    aEnum(eNoCountry)
{
    while (next())
    {
        if (p.equal(abbreviation()))
        {
            // aEnum set to the right one
            return;
        }
    }

    aEnum = eNoCountry;
}


CountryIndex::CountryIndex(const PointerIntoInputString & p, int)
:
    aEnum(eNoCountry)
{
    // constructor from reprint prefix

    while (next())
    {
        if (p.equal(reprintPrefix()))
        {
            // aEnum set to the right one
            return;
        }
    }

    aEnum = eNoCountry;
}


bool CountryIndex::next(void)
{
    if (aEnum == eNoCountry)
    {
        aEnum = eFirstCountry;
    }
    else
    {
        if (aEnum != eLastCountry)
        {
            aEnum = (CountryEnum) (aEnum + 1);
        }
    }
    return aEnum != eLastCountry;
}


const char * CountryIndex::name(void) const
{
    assert(aEnum < eLastCountry);
    return cCountryStruct[aEnum].sName;
}


const char * CountryIndex::abbreviation(void) const
{
    assert(aEnum < eLastCountry || aEnum == eNoCountry);
    return cCountryStruct[aEnum].sAbbreviation;
}


const char * CountryIndex::reprintTitlePrefix(void) const
{
    assert(aEnum < eLastCountry);
    return cCountryStruct[aEnum].sReprintTitlePrefix;
}


const char * CountryIndex::reprintPrefix(void) const
{
    assert(aEnum < eLastCountry);
    return cCountryStruct[aEnum].sReprintPrefix;
}


CountrySet::CountrySet(const char * p)
{
    CountryIndex c;
    while (c.next())
    {
        aPresent[c.number()] = FALSE;
    }

    if (strcmp(p, "all") == 0)
    {
        allCountries();
    }
    else if (strcmp(p, "none") == 0)
    {
        // already initialised correctly
    }
    else
    {
        ShortList lShl;
        lShl.set(p); // will insert from comma-delimited string
        int lNumberOfCountries = lShl.numberOfElements();
        for (int i = 0; i < lNumberOfCountries; i++)
        {
            CountryIndex c3(lShl.element(i));
            aPresent[c3.number()] = TRUE;
        }
    }
}


void CountrySet::clear(void)
{
    CountryIndex c;
    while (c.next())
    {
        aPresent[c.number()] = FALSE;
    }
}


void CountrySet::allCountries(void)
{
    CountryIndex c;
    while (c.next())
    {
        aPresent[c.number()] = TRUE;
    }
}


void CountrySet::insert(const CountryIndex p)
{
    aPresent[p.number()] = TRUE;
}


bool CountrySet::includes(const CountryIndex & p) const
{
    return aPresent[p.number()];
}
