/* DIZNI.CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   Global main function

   95-07-20  created
   97-04-28  "legends" moved to "preformat"
   97-09-01  Preformatter and GatherStoryWithEntries fully objectified
*/

#define PROGRAMID "DIZNI version 971023"

#include <stdio.h>
#include "definiti.h"  // bool
#include "mylib.h"
#include "commandl.h"
#include "preforma.h"
#include "gather.h"
#include "bulkoutp.h"
#include "creatoro.h"
#include "barksweb.h"
#include "accessou.h"
#include "inputche.h"

class LogHeader
{
public:
    LogHeader(const char *);
    LogHeader(const char * p1, const char * p2);
    LogHeader(const char * p1, const char p2);
    ~LogHeader();

private:
    const char * aText;
};


LogHeader::LogHeader(const char * p1)
:
    aText(p1)
{
    MyLib::log("%s", p1);
    printf("%s\n", p1);
}


LogHeader::LogHeader(const char * p1, const char * p2)
:
    aText(p1)
{
    MyLib::log("%s (%s)", p1, p2);
    printf("%s (%s)\n", p1, p2);
}


LogHeader::LogHeader(const char * p1, const char p2)
:
    aText(p1)
{
    MyLib::log("%s (%c)", p1, p2);
    printf("%s (%c)\n", p1, p2);
}


LogHeader::~LogHeader()
{
    MyLib::log("%s ready", aText);
}


static bool doPreformat(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xp [-h]\n\
\n\
       -h          generate header files only\n";

    pCommandLine.check('h', CommandLine::eOptional, CommandLine::eNoArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    LogHeader l("PREFORMAT", pCommandLine.present('h') ? "Headers" : "");

    Preformatter lPreformatter(pCommandLine.present('h'));
    // -h = test option: create header files only
    lPreformatter.doit();

    return TRUE;
}


static bool doGather(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xg -c <countries>\n\
\n\
       -c <countries> allow titles from <countries> in story files\n\
                      (<countries> is a comma-delimited list of 2 letters,\n\
                      e.g. 'us,de,nl,se,dk'; or 'all')\n";

    pCommandLine.check('c', CommandLine::eMandatory, CommandLine::eStringArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    LogHeader l("GATHER");
    CountrySet lAllowedTitleCountries(pCommandLine.stringArgument('c'));
    GatherStoryWithEntries::doit(lAllowedTitleCountries);
    return TRUE;
}


static bool doCreatorOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xc [-u] -a <abbrev> [-s <select>] [-e] [-c <countries>]\n\
             [-r <countries>] -f <file> -l <layout> [-w <width>]\n\
\n\
                 SELECTION options:\n\
\n\
    -u           select on subseries (in stead of creator)\n\
    -a <abbrev>  abbreviation of the creator or subseries\n\
    -s <select>  stories / gags / covers / oils / everything (default)\n\
    -e           list per entrycode (default per storycode) (not with -u)\n\
    -c <countries> (unsolved) entries from countries (default = none)\n\
    -r <countries> list reprints of countries (default = all)\n\
    (<countries> is a comma-delimited list of 2 letters,\n\
     e.g. 'us,de,nl,se,dk'; or 'all' or 'none')\n\
\n\
                 OUTPUT options:\n\
\n\
    -f <file>    name of the output file (without extension)\n\
    -l <layout>  l = reprints on separate line\n\
                 c = reprints in columns\n\
                 r = reprints as comments\n\
                 d = layout for DES-file\n\
                 w = web page output\n\
    -w <width>   maximum line width (default 80)\n";

    pCommandLine.check('u', CommandLine::eOptional, CommandLine::eNoArgument);
    pCommandLine.check('a', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('s', CommandLine::eOptional, CommandLine::eStringArgument);
    if (!pCommandLine.present('u'))
    {
        pCommandLine.check('e', CommandLine::eOptional, CommandLine::eNoArgument);
    }
    pCommandLine.check('c', CommandLine::eOptional,  CommandLine::eStringArgument);
    pCommandLine.check('r', CommandLine::eOptional,  CommandLine::eStringArgument);
    pCommandLine.check('f', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('l', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('w', CommandLine::eOptional,  CommandLine::eIntegerArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    LogHeader l("CREATOR OUTPUT");

    // selection options:
    SelectionOptions lSelectionOptions(
        pCommandLine.stringArgument('a'), // selector
        StoGaCov(pCommandLine.stringArgument('s', "e")[0]),
        CountrySet(pCommandLine.stringArgument('c', "none")));

    // report options:
    ReportOptions lReportOptions(
        eCountryFormat, // may be changed
        CountrySet(pCommandLine.stringArgument('r', "all")),
        pCommandLine.stringArgument('l')[0], // layout
        pCommandLine.integerArgument('w', 80)); // width

    const char * lFileName = pCommandLine.stringArgument('f');
    if (pCommandLine.present('u'))
    {
        SubseriesOutput lSelectedOutput(
            lFileName, lReportOptions, lSelectionOptions);
        lSelectedOutput.doit();
    }
    else if (pCommandLine.present('e'))
    {
        lReportOptions.aFormat = eDBIFormat;
        if (strcmp(lSelectionOptions.aSelector, "CB") == 0)
        {
            lReportOptions.aFormat = eBarksFormat;
        }
        CreatorIssueOutput lSelectedOutput(
            lFileName, lReportOptions, lSelectionOptions);
        lSelectedOutput.doit();
    }
    else
    {
        if (strcmp(lSelectionOptions.aSelector, "CB") == 0)
        {
            lReportOptions.aFormat = eBarksFormat;
        }
        CreatorOutput lSelectedOutput(
            lFileName, lReportOptions, lSelectionOptions);
        lSelectedOutput.doit();
    }

    return TRUE;
}


static bool doBarksWebOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xb -f <name> [-r <countries>]\n\
\n\
       -f <name>      file name (extension is always 'des')\n\
       -r <countries> list reprints of countries (default = all)\n\
                      (<countries> is a comma-delimited list of 2 letters,\n\
                      e.g. 'us,de,nl,se,dk'; or 'all' or 'none')\n";

    pCommandLine.check('f', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('r', CommandLine::eOptional,  CommandLine::eStringArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    LogHeader l("BARKSWEB OUTPUT");

    BarksWebPages lBarksWebPages;

    FileName lDesFileName(pCommandLine.stringArgument('f'), "des");
    PersonLegend lPersonLegend;

    lBarksWebPages.read(lDesFileName);
    lBarksWebPages.addEntries(FileName("stories", "ins"));
    lBarksWebPages.write(FileName(lDesFileName, "htm"),
        lPersonLegend,
        CountrySet(pCommandLine.stringArgument('r', "all")));

    return TRUE;
}


static bool doStoryOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xs [-r <countries>] -l <layout> [-w <width>]\n\
\n\
       -r <countries> list reprints of countries (default = all)\n\
                      (<countries> is a comma-delimited list of 2 letters,\n\
                       e.g. 'us,de,nl,se,dk'; or 'all' or 'none')\n\
       -l <layout>    l = reprints on separate line\n\
                      c = reprints in columns\n\
                      r = reprints as comments\n\
                      d = layout for DES-file\n\
                      w = web page output\n\
       -w <width>     maximum line width (default 80)\n";

    pCommandLine.check('l', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('w', CommandLine::eOptional,  CommandLine::eIntegerArgument);
    pCommandLine.check('r', CommandLine::eOptional,  CommandLine::eStringArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    ReportOptions lOptions(
        eDBIFormat,
        CountrySet(pCommandLine.stringArgument('r', "all")),
        pCommandLine.stringArgument('l')[0], // layout
        pCommandLine.integerArgument('w', 80)); // width

    LogHeader l("STORY OUTPUT");
    BulkOutput::processAllStoryFiles(lOptions);
    return TRUE;
}


static bool doEntryOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xe -c <countries> -l <layout> [-w <width>]\n\
\n\
       -c <countries> entries from countries\n\
                      (<countries> is a comma-delimited list of 2 letters,\n\
                      e.g. 'us,de,nl,se,dk'; or 'all')\n\
       -l <layout>    l = reprints on separate line\n\
                      c = reprints in columns\n\
                      r = reprints as comments\n\
                      d = layout for DES-file\n\
                      w = web page output\n\
       -w <width>     maximum line width (default 80)\n";

    pCommandLine.check('c', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('l', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('w', CommandLine::eOptional,  CommandLine::eIntegerArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    CountrySet lCountrySet(pCommandLine.stringArgument('c'));

    ReportOptions lOptions(
        eDBIFormat,
        CountrySet("none"), // reprint countries, adjusted later
        pCommandLine.stringArgument('l')[0], // layout
        pCommandLine.integerArgument('w', 80)); // width

    LogHeader l("ENTRY OUTPUT");
    BulkOutput::processAllEntryFiles(lOptions, lCountrySet);
    return TRUE;
}


static bool doLegendOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xl [-w <width>]\n\
\n\
       -w <width>     maximum line width (default 80)\n";

    pCommandLine.check('w', CommandLine::eOptional,  CommandLine::eIntegerArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    LogHeader l("LEGEND OUTPUT");
    BulkOutput::processAllLegendFiles(
        pCommandLine.integerArgument('w', 80)); // width
    return TRUE;
}


static bool doAccessOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xa [ -s | -c <countries> ]\n\
\n\
       -c <countries> MS-Access issue file of countries\n\
                      (<countries> is a comma-delimited list of 2 letters,\n\
                      e.g. 'us,de,nl,se,dk'; or 'all')\n\
       -s             MS-Access story file\n";

    pCommandLine.check('s', CommandLine::eOptional, CommandLine::eNoArgument);
    pCommandLine.check('c', CommandLine::eOptional, CommandLine::eStringArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    CountrySet lCountrySet(pCommandLine.stringArgument('c'));

    LogHeader l("ACCESS");

    DBIFileIndex lFileIndex;
    while (lFileIndex.next())
    {
        if (lCountrySet.includes(lFileIndex.country()))
        {
            AccessOutputFile l(FileName(lFileIndex.fileName(), "ine"));
            l.make();
        }
    }

    if (pCommandLine.present('s'))
    {
        AccessOutputFile l(FileName("stories", "ins"));
        l.make();
    }
    return TRUE;
}


static bool doHeaderOutput(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xh [-a <abbrev>] -c <countries> [-w <width>]\n\
\n\
       -a <abbrev>    abbreviation of the indexer (default: all indexers)\n\
       -c <countries> list issues from countries\n\
                      (<countries> is a comma-delimited list of 2 letters,\n\
                      e.g. 'us,de,nl,se,dk'; or 'all')\n\
       -w <width>     maximum line width (default 80)\n";

    pCommandLine.check('a', CommandLine::eOptional,  CommandLine::eStringArgument);
    pCommandLine.check('c', CommandLine::eMandatory, CommandLine::eStringArgument);
    pCommandLine.check('w', CommandLine::eOptional,  CommandLine::eIntegerArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    LogHeader l("HEADER OUTPUT");
    BulkOutput::processAllHeaderFiles(
        pCommandLine.stringArgument('a'),
        CountrySet(pCommandLine.stringArgument('c')),
        pCommandLine.integerArgument('w', 80)); // width

    return TRUE;
}


static bool doInputCheck(CommandLine & pCommandLine)
{
    const char * lUsage = "\
Usage: dizni -xi [-c] [-f <name>]\n\
\n\
       -c           list all comments (default NO)\n\
       -f <name>    file name (extension is always 'dbi')\n\
\n\
If no file name is given, the standard set of dbi/dbs files is checked.\n";

    pCommandLine.check('c', CommandLine::eOptional, CommandLine::eNoArgument);
    pCommandLine.check('f', CommandLine::eOptional, CommandLine::eStringArgument);
    if (pCommandLine.printError(lUsage))
    {
        return FALSE;
    }

    InputChecker i(pCommandLine.present('c'));

    if (pCommandLine.present('f'))
    {
        LogHeader l("INPUT CHECK", pCommandLine.stringArgument('f'));
        i.checkFile(
            FileName(pCommandLine.stringArgument('f'), "dbi"));
    }
    else
    {
        LogHeader l("INPUT CHECK", "all files");
        i.checkAllFiles();
    }
    return TRUE;
}


int main(int argc, char *argv[])
{
    printf("%s\n", PROGRAMID);
    MyExternalOutputFile::setProgramId(PROGRAMID);

    CommandLine lCommandLine(argc, argv);
    lCommandLine.check('x', CommandLine::eMandatory, CommandLine::eStringArgument);

    if (!lCommandLine.present('x'))
    {
        printf("\nUsage: dizni -x <execute> <other options>\n\n");
        printf("\texecute: p = preformat\n");
        printf("\t         g = gather\n");
        printf("\t         s = storyoutput\n");
        printf("\t         e = entryoutput\n");
        printf("\t         l = legendoutput\n");
        printf("\t         c = creatoroutput (also subseries)\n");
        printf("\t         b = Barksweb output\n");
        printf("\t         a = MS-Access output\n");
        printf("\t         h = headeroutput\n");
        printf("\t         i = inputcheck\n\n");
        return 1;
    }

    switch (lCommandLine.stringArgument('x')[0])
    {
    case 'p': if (!doPreformat(lCommandLine)) return 1; break;
    case 'g': if (!doGather(lCommandLine)) return 1; break;
    case 's': if (!doStoryOutput(lCommandLine)) return 1; break;
    case 'e': if (!doEntryOutput(lCommandLine)) return 1; break;
    case 'l': if (!doLegendOutput(lCommandLine)) return 1; break;
    case 'c': if (!doCreatorOutput(lCommandLine)) return 1; break;
    case 'b': if (!doBarksWebOutput(lCommandLine)) return 1; break;
    case 'a': if (!doAccessOutput(lCommandLine)) return 1; break;
    case 'h': if (!doHeaderOutput(lCommandLine)) return 1; break;
    case 'i': if (!doInputCheck(lCommandLine)) return 1; break;
    default: ;
    }

    Progress::finish();
    return 0;
}
