/* ENTRY.H, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   Object: Entry

   An entry in the database that describes one story or
   one publication of a story.

   An Entry has two storage formats in ASCII files: "internal" and
   "external". Functions to read/write these formats are in separate
   objects (in their own source files).

   95-04-10  created
   95-12-08  converted to C++
*/

#ifndef ENTRY_H
#define ENTRY_H

#include "definiti.h"
#include "shortlis.h"
#include "comment.h"
#include "country.h"
#include "pages.h"
#include "pagelayo.h"
#include "fielddef.h"
#include "pointeri.h"

#define STORYCODElength   12
#define STORYCODEsize     13
typedef char STORYCODEtype[STORYCODEsize];

#define ENTRYCODElength   10
#define ENTRYCODEsize     11
typedef char ENTRYCODEtype[ENTRYCODEsize];

#define DATElength        8 // yy-mm-dd
#define DATEsize          9
typedef char DATEtype[DATEsize];

#define PERSONlength      4
#define PERSONsize        5
typedef char PERSONtype[PERSONsize];

#define TITLElength       280  // enough to cope with K-stories
#define TITLEsize         281
typedef char TITLEtype[TITLEsize];

#define PREFIXlength      10
#define PREFIXsize        11
typedef char PREFIXtype[PREFIXsize];
// for field prefixes, NOT for title prefix


class Entry
{
    friend class WebEntry;

public:
    Entry(void);
    virtual ~Entry();

    // access to fields:
    ShortList & field  (const ShortListFieldIndex & p);
    ShortList & field  (const ShortListFieldEnum p);
    ShortList & reprint(const CountryIndex p);
    ShortList & reprint(const CountryEnum p);
    Comment   & comment(void);
    Comment   & storyComment(void);

    CountryIndex country(void) const;
    void setCountry(const CountryIndex p);
    void setCountry(const CountryEnum p);

    bool isUnsolved(void);
    void setUnsolved(const char * p = "@");

    const PointerIntoInputString & title(void) const;
    void clearTitle(void);
    void copyTitle(Entry & pOther);

    bool goodPart(void);
    // A "good" part is the first part of a sequel, or the only part
    // in a 1-part story.
    // "bad" parts are part 2, 3, etc.

    char * cleanedEntryCode(char * pBuffer);
    // Removes too much spaces, and the entry character, e.g.
    //  "DD  228a" -> "DD 228"
    // Also adds characters indicating changes, missing panels, etc.
    // The result can be used as a reprint-entrycode
    // Returns pointer to pBuffer.

public: // should have been protected...
    // public attributes start with 's', protected/private start with 'a'.
    EntryOrStoryCode sStoryCode;
    EntryOrStoryCode sEntryCode;
    Pages sPages;
    PageLayout sPageLayout;
    TitlePrefix sTitlePrefix;

protected:
    char aUnsolved[2];
    PointerIntoInputString aTitle;

private:
    const char * partString(void);
    const char * changesString(void);

    // next used in internal format:
    CountryIndex aCountry; // the country of an issue
    ShortList aShortListField[eLastShortListField];
    ShortList aReprintField[eLastCountry];
    Comment aComment;
    Comment aStoryComment;
};


// inline functions, for speed:

inline ShortList & Entry::field(const ShortListFieldIndex & p)
{
    return aShortListField[p.number()];
}


inline ShortList & Entry::field(const ShortListFieldEnum p)
{
    return aShortListField[p];
}


inline ShortList & Entry::reprint(const CountryIndex p)
{
    return aReprintField[p.number()];
}


inline ShortList & Entry::reprint(const CountryEnum p)
{
    return aReprintField[p];
}


inline Comment & Entry::comment(void)
{
    return aComment;
}


inline Comment & Entry::storyComment(void)
{
    return aStoryComment;
}


inline CountryIndex Entry::country(void) const
{
    return aCountry;
}


inline void Entry::setCountry(const CountryIndex p)
{
    aCountry = p;
}


inline void Entry::setCountry(const CountryEnum p)
{
    aCountry = p;
}


inline bool Entry::isUnsolved(void)
{
    return aUnsolved[0] != '\0';
}


inline void Entry::setUnsolved(const char * p)
{
    strcpy(aUnsolved, p);
}


inline const PointerIntoInputString & Entry::title(void) const
{
    return aTitle;
}


inline void Entry::clearTitle(void)
{
    aTitle.pointTo("");
}

#endif
