/* INDEXEDI(ssueRange).CC, (c) Harry Fluks 1994, 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-07-18  created
   96-08-07  checked
*/

#include "indexedi.h"
#include <stdlib.h>

IndexedIssueRange::IndexedIssueRange(Header & pHeader)
:
    aMinNumber(1),
    aMaxNumber(1),
    aCheckAllListed(TRUE),
    aAllListedGuaranteed(FALSE),
    aLevel(pHeader.level())
{
    for (int i = 0; i < cMaxIssueNumber; i++)
    {
        aOccurs[i] = FALSE;
    }
    pHeader.storyCode().copyToChars(aRangeCode, STORYCODElength);
    pHeader.title().copyToChars(aRangeTitle, TITLElength);

    if (! pHeader.field(eMinNumber).empty())
    {
        aMinNumber = atoi(pHeader.field(eMinNumber).element(0));
        if (aMinNumber > cMaxIssueNumber - 1)
        {
            MyLib::log("In header (%s) minnumber (%d) too large (max %d)",
                       aRangeCode, aMinNumber, cMaxIssueNumber - 1);
            aMinNumber = 1; // header syntax error
        }
    }
    if (pHeader.field(eMaxNumber).empty())
    {
        // MyLib::log("In header (%s) maxnumber not filled in", aRangeCode);
        // this occurs very often; it is one of the issues for InputCheck.
        aCheckAllListed = FALSE;
    }
    else
    {
        const char * c = pHeader.field(eMaxNumber).element(0);
        switch (c[0])
        {
        case '?':
        case '-':
            aMaxNumber = 0;
            aCheckAllListed = FALSE;
            break;
        case '!':
            aMaxNumber = 0;
            aAllListedGuaranteed = TRUE; // *** in fact only if the indexer matches...
            break;
        default:
            aMaxNumber = atoi(c);
            if (aMaxNumber > cMaxIssueNumber - 1)
            {
                MyLib::log("In header (%s) maxnumber (%d) too large (max %d)",
                           aRangeCode, aMaxNumber, cMaxIssueNumber - 1);
                aMaxNumber = 1; // header syntax error
            }
        }
    }
    if (aMaxNumber < aMinNumber)
    {
        aCheckAllListed = FALSE;
    }
}


void IndexedIssueRange::write(MyFormattedOutputFile & pFile)
{
    for (int k = 0; k < cMaxIssueNumber; k++)
    {
        if (aOccurs[k])
        {
            // only write header if any occurrences.
            // maybe we want it **always**
            // maybe headers of different levels go wrong! **

            if (aLevel == '2')
            {
                pFile.printF("|%7s    %s", " ", aRangeTitle);
                // don't write aRangeCode, it was written before
            }
            else
            {
                pFile.printF("%-8s %s", aRangeCode, aRangeTitle);
            }
            pFile.forceNewLine();
            break; // EXIT FOR-LOOP
        }
    }

    bool lVeryFirst = TRUE;
    int lSave = -1; // first number in the current subrange
    int lListed = 0;
    int lNotListed = 0;
    for (int i = 0; i < cMaxIssueNumber; i++)
    {
        if (aOccurs[i])
        {
            lListed++;
            if (lSave == -1)
            {
                lSave = i;
            }
        }
        else
        {
            if (i >= aMinNumber && i <= aMaxNumber)
            {
                lNotListed++;
            }

            // flush previous sequence
            if (lSave != -1)
            {
                if (lVeryFirst)
                {
                    lVeryFirst = FALSE;
                }
                else
                {
                    pFile.putS(", ");
                }

                if (lSave == i - 1)
                {
                    pFile.printF("%d", lSave);
                }
                else
                {
                    pFile.printF("%d-%d", lSave, i - 1);
                }
                lSave = -1;
            }
        }
    }
    // flushing last sequence is not needed; the last
    // sOccurs is always FALSE.

    assert(lSave == -1);

    if (! lVeryFirst)
    {
        pFile.putS(" ");
    }

    if (lListed > 0) // ********************
    {
        // print a verdict on the completeness of the range
        printVerdict(pFile, lListed, lNotListed);
        pFile.printLine();
    }
}


void IndexedIssueRange::printVerdict(
    MyFormattedOutputFile & pFile,
    int pListed,
    int pNotListed)
{
    if (aAllListedGuaranteed)
    {
        pFile.printF("all"); // we didn't count pListed
    }
    else 
    {
        if (pListed == 0)
        {
            pFile.putS("(no issues");
        }
        else
        {
            pFile.printF("(%d issue%s", pListed, (pListed == 1 ? "" : "s"));
        }
        
        if (aCheckAllListed)
        {
            if (pNotListed == 0)
            {
                pFile.printF(" = all)");
            }
            else
            {
                if (aMinNumber == aMaxNumber)
                {
                    // pNotListed == 1, pListed == 0
                    if (aMaxNumber == 1) // most likely incomplete header entry!
                    {
                        pFile.putS(")");
                    }
                    else
                    {
                        pFile.printF("; #%d missing)", aMaxNumber);
                    }
                }
                else
                {
                    pFile.printF("; %d missing in %d-%d)",
                              pNotListed, aMinNumber, aMaxNumber);
                }
            }
        }
        else
        {
            pFile.putS(")"); // only mention the number of listed issues
        }
    }
}


void IndexedIssueRange::addNumber(const PointerIntoInputString & pIssueCode)
{
    int lInt = findNumber(pIssueCode);

    if (lInt >= cMaxIssueNumber - 1)
    {
        MyLib::log("Strange number: %d", lInt);
    }
    else
    {
        if (lInt < aMinNumber || lInt > aMaxNumber)
        {
            if (aCheckAllListed)
            {
                MyLib::log("Number out of range: %d", lInt);
            }
        }
        if (aOccurs[lInt])
        {
// ***** occurs too often with X-numbers... MyLib::log("Number occurs more than once: %s", lIssueCodeBuffer);
        }
        aOccurs[lInt] = TRUE;
    }
}


void IndexedIssueRange::addToUnindexed(const PointerIntoInputString &)
{
    // we don't use this information; every not-indexed issue is "unindexed"
    // by definition
}


int IndexedIssueRange::findNumber(const PointerIntoInputString & pIssueCode)
{
    STORYCODEtype lIssueCodeBuffer;
    pIssueCode.copyToChars(lIssueCodeBuffer, STORYCODElength);

    // check compatibility
    if (! pIssueCode.equalOrLonger(aRangeCode))
    {
        MyLib::log("Incompatible headers: (%s)(%s)",
                   aRangeCode, lIssueCodeBuffer);
        // just a warning, but the output will be wrong
    }

    // find out the number
    const char * lFirstDigit = lIssueCodeBuffer;
    bool lNoDigit = TRUE;
    for (const char * c = lIssueCodeBuffer; *c != '\0'; c++)
    {
        if (*c >= '0' && *c <= '9')
        {
            if (lNoDigit)
            {
                lFirstDigit = c;
                lNoDigit = FALSE;
            }
        }
        else
        {
            lNoDigit = TRUE;
            // found a non-digit after the lFirstDigit
        }
    }

    // convert ascii to integer
    return atoi(lFirstDigit);
}
