/* INPUTCHE(ck).CC, (c) Harry Fluks 1994

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   94-10-06  created
   95-03-27  adjusted to new formats (see entry.i etc.)
   96-01-17  converted to C++, several new checks
   96-07-17  made part of DIZNI main program
   96-08-13  checked
*/

#include "inputche.h"
#include "header.h"
#include "indexedi.h" // for cMaxIssueNumber

InputChecker::InputChecker(bool pListComments)
:
    aListComments(pListComments),
    aHeroList(TRUE),
    aCreatorList(TRUE),  // TRUE: complete persons
    aSubseriesList(TRUE)
{
    aHeroList     .readFromInternalFile(FileName("heroes", "inl"));
    aCreatorList  .readFromInternalFile(FileName("creators", "inl"));
    aSubseriesList.readFromInternalFile(FileName("subserie", "inl"));
}


InputChecker::~InputChecker()
{
}


void InputChecker::checkAllFiles(void)
{
    strcpy(aPreviousStoryCode, "");

    DBSFileIndex lDBSFileIndex;
    while (lDBSFileIndex.next())
    {
        processOneStoryFile(lDBSFileIndex);
    }

    DBIFileIndex lDBIFileIndex;
    while (lDBIFileIndex.next())
    {
        strcpy(aPreviousEntryCode, "");
        processOneEntryFile(
            lDBIFileIndex.fileName(),
            lDBIFileIndex.isWestern());
    }
}


void InputChecker::checkFile(const FileName & p)
{
    processOneEntryFile(p, FALSE);
    // FALSE: not pIsWestern. We only want to check standard DBI files
}


void InputChecker::checkStoryCodeSequence(const PointerIntoInputString & p, bool pInHeader)
{
    int lResult = p.compare(aPreviousStoryCode);
    if (lResult == 0)
    {
        // headers on different levels CAN have the same code
        if (!pInHeader)
        {
            aCheckLog.log(1, "story code", p, "identical to previous");
        }
    }
    else if (lResult < 0)
    {
        aCheckLog.log(2, "story code", p, "smaller than previous");
    }

    p.copyToChars(aPreviousStoryCode, STORYCODElength);
}


void InputChecker::checkEntryCodeSequence(const PointerIntoInputString & p, bool pInHeader)
{
    if (p.empty())
    {
        return;
    }

    int lResult = p.compare(aPreviousEntryCode);
    if (lResult == 0)
    {
        if (!pInHeader)
        {
// *** not of interest at the moment... aCheckLog.log(3, "entry code", p, "identical to previous");
        }
    }
    else if (lResult < 0)
    {
        aCheckLog.log(4, "entry code", p, "smaller than previous");
    }

    p.copyToChars(aPreviousEntryCode, ENTRYCODElength);
}


void InputChecker::checkDate(const char * p)
{
    // all spaces, or "70-01-01".
    //                 01234567
    // No strict cheking on date (yet)

    for (int i = 0; i < DATElength; i++)
    {
        if (p[i] == '\0')
        {
            // date is allowed to be shorter than yy-mm-dd
            return;
        }

        if ((i == 2) || (i == 5))
        {
            if ((p[i] != '-') && (p[i] != ' '))
            {
                aCheckLog.log(5, "date", p, "format should be yy-mm-dd");
                return;
            }
        }
        else
        {
            switch (p[i])
            {
            case '0': case '1': case '2': case '3': case '4':
            case '5': case '6': case '7': case '8': case '9':
            case ' ':
            case '?':
                break;
            default:
                aCheckLog.log(5, "date", p, "non-digit");
                return;
            }
        }
    }
}


void InputChecker::checkStoryCode(const PointerIntoInputString & p, Format pStoryFormat)
{
    if (  (pStoryFormat == eDBIFormat)
       || (pStoryFormat == eWesternDBIFormat)
       )
    {
        return;
        // no checks at all
    }

    // some checks on the 'real' length of the code
    switch (pStoryFormat)
    {
    case eWesternStoryFormat:
        // codes starting with "C" are from the special cb-misc
        // file, and don't follow the rules exactly

        if (p[0] == 'C')
        {
            return;
        }

        // "W WDC 123-01"

        if (p[9] != '-')
        {
            aCheckLog.log(6, "story code", p, "10th char should be '-'");
            return;
        }

        switch (p[10])
        {
        case '0': case '1': case '2': case '3': case '4':
        case '5': case '6': case '7': case '8': case '9':
        case '?':
            break;

        default:
            aCheckLog.log(6, "story code", p, "unexpected value 11th char");
            return;
        }

        // almost the same for the last char:
        switch (p[11])
        {
        case '0': case '1': case '2': case '3': case '4':
        case '5': case '6': case '7': case '8': case '9':
            break;
        default:
            if (p[10] != '?')
            {
                aCheckLog.log(6, "story code", p, "unexpected value 12th char");
                return;
            }
        }
        break;

    case eCountryFormat:
        // H 85005 etc...
        break;

    case eYMorZTFormat:
        {
            for (int i = 6; i < STORYCODElength; i++)
            {
                if (p[i] != '\0')
                {
                    aCheckLog.log(6, "story code", p, "non-space after 6th char");
                    return;
                }
            }
        }
        break;

    case eNewspaperFormat:
        // "XX 40-01-01" - date is part of the code
        if (p[2] != ' ' && p[2] != '\0')
        {
            aCheckLog.log(6, "story code", p, "3rd char should be space");
            return;
        }
        {
            STORYCODEtype lStoryCode;
            p.copyToChars(lStoryCode, STORYCODElength);
            checkDate(lStoryCode + 3);
        }
        break;

    default:
        ; // OK
    }
}


void InputChecker::checkPages( const Pages & p,
                               ShortList & pPart,
                               const ShortList & pExactPages,
                               Format pStoryFormat)
{
    if (p.isBroken())
    {
        if (pExactPages.empty())
        {
            // aCheckLog.log(7, "exactpages", "", "should not be empty if page-part is '+'");
            // we allow this, it occurs too often...
        }
    }

    if (! p.valid())
    {
        aCheckLog.log(7, "page field", p.theString(), "invalid");
        return;
    }

    switch (pStoryFormat)
    {
    case eWesternStoryFormat:
    case eCountryFormat:
        if (p.empty())
        {
            if (pExactPages.empty())
            {
                aCheckLog.log(7, "page field", "", "empty - exactpages should be specified");
                // exactpages then contains pages >= 100
            }
        }
        break;

    case eYMorZTFormat:
    case eNewspaperFormat:
        if (! p.empty())
        {
            aCheckLog.log(7, "page field", p.theString(), "should be empty");
        }
        break;

    case eDBIFormat:
    case eWesternDBIFormat:
        if (  (pPart.delimitedString()[0] != ' ')
           && (! pPart.empty())
           && p.empty()
           && pExactPages.empty()
           )
        {
            aCheckLog.log(8, "page field", "", "should not be empty in multi-part story");
        }
        break;
    default:
        assert(FALSE);
    }
}


void InputChecker::checkPageLayout(const PageLayout p)
{
    if (!p.valid())
    {
        aCheckLog.log(9, "page layout", p.theChar(), "unknown");
    }
}


void InputChecker::checkMissingPanels(const char * p)
{
    if (strlen(p) > 1)
    {
        aCheckLog.log(10, "missing panels field", p, "has more than one char");
    }
    switch (p[0])
    {
    case '1': case '2': case '3': case '4':
    case 'r': // one row missing
    case 'e': // one panel missing
    case ' ':
    case '\0':
        break;
    default:
        aCheckLog.log(10, "missing panels ", p, "not one of 1234re or space");
    }
}


void InputChecker::checkPart(const char * p)
{
    if (strlen(p) > 2)
    {
        aCheckLog.log(11, "part field", p, "has more than two chars");
        return;
    }

    switch (p[0])
    {
    case '1': case '2': case '3': case '4':
    case '5': case '6': case '7': case '8': case '9':
    case ' ': case '\0':
    case 'f': // final part
    case '?': // unknown part
    case 'x': // not a part, overrides part character from other source
        break;

    default:
        aCheckLog.log(11, "part char", p, "not a nonzero number, f, ?, x or space");
    }

    if (strlen(p) == 2)
    {
        switch (p[1])
        {
        case '0': case '1': case '2': case '3': case '4':
        case '5': case '6': case '7': case '8': case '9':
            break;

        default:
            aCheckLog.log(11, "2nd part char", p, "not a number");
        }
    }
}


void InputChecker::checkComment(const Comment & p, const char * pWhat)
{
    if (aListComments && !p.empty())
    {
        aCheckLog.log(12, pWhat, p.element(), "");
    }
}


void InputChecker::checkTitle(const TitlePrefix & pTitlePrefix,
                              const PointerIntoInputString & pTitle)
{
    if (pTitlePrefix.theString()[0] != '\0')
    {
        if (pTitle.empty())
        {
            aCheckLog.log(13, "title prefix", pTitlePrefix.theString(),
                          "without title");
        }

        if (! pTitlePrefix.valid())
        {
            aCheckLog.log(13, "title prefix",
                          pTitlePrefix.theString(), "unknown");
        }
    }
}


void InputChecker::checkPerson(ShortList & pPersons,
                               const char * pErrorText,
                               PersonList & pPersonList)
{
    int num = pPersons.numberOfElements();
    for (int i = 0; i < num; i++)
    {
        bool lDummy;
        PERSONNAMEtype lCleanPerson;
        (void) PersonList::cleanPerson(pPersons.element(i), lDummy, lCleanPerson);
        assert(strlen(lCleanPerson) <= PERSONNAMElength);

        if (lCleanPerson[0] == '?')
        {
            // OK, apparently an unknown person
        }
        else if (lCleanPerson[0] == '\0')
        {
            // empty person after "cleaning"
            aCheckLog.log(14, pErrorText, pPersons.element(i), "empty after cleaning");
        }
        else
        {
            if (pPersonList.find(lCleanPerson, FALSE) == 0) // FALSE: not in Latin-1
            {
                if (pPersonList.findPersonCode(lCleanPerson) == 0)
                {
                    aCheckLog.log(15, pErrorText, lCleanPerson, "not in table");
                }
                else
                {
                    aCheckLog.log(21, pErrorText, lCleanPerson,
                                  "has different official indication");
                }
            }
        }
    }
}


void InputChecker::checkHeaderLevel(char p)
{
    switch (p)
    {
    case '0': // file
    case '1': // series
    case '2': // volume
    case '3': // issue
        break;
    default:
        aCheckLog.log(16, "Level", p, "should be between 0 and 3");
    }
}


void InputChecker::checkEntry(ExternalEntry & pEntry, bool pInEntryFile)
{
    if (pInEntryFile)
    {
        checkEntryCodeSequence (pEntry.sEntryCode, FALSE);
    }
    else
    {
        checkStoryCodeSequence (pEntry.sStoryCode, FALSE);
    }

    checkStoryCode    (pEntry.sStoryCode, pEntry.format());
    checkPages        (pEntry.sPages,
                       pEntry.field(ePart),
                       pEntry.field(eExactPages),
                       pEntry.format());
    checkPageLayout   (pEntry.sPageLayout);
    checkMissingPanels(pEntry.field(eMissingPanels).delimitedString());
    checkPart         (pEntry.field(ePart).delimitedString());
    checkDate         (pEntry.field(eDate).delimitedString());
    checkDate         (pEntry.field(eEndDate).delimitedString());
    checkTitle        (pEntry.sTitlePrefix, pEntry.title());
    checkComment      (pEntry.comment(), "comment");
    checkComment      (pEntry.storyComment(), "story-comm");

    checkPerson (pEntry.field(ePlotter),  "Creator", aCreatorList);
    checkPerson (pEntry.field(eWriter),   "Creator", aCreatorList);
    checkPerson (pEntry.field(eArtist),   "Creator", aCreatorList);
    checkPerson (pEntry.field(eInker),    "Creator", aCreatorList);
    checkPerson (pEntry.field(eHero),     "Hero", aHeroList);
    checkPerson (pEntry.field(eAppearing),"Hero", aHeroList);
    checkPerson (pEntry.field(eSer),      "Subseries", aSubseriesList);
    // eCode, eSequence, eChapter, eAlsoReprint, eExactPages,
    // eChanges have no strict layout, so need not be checked
}


void InputChecker::checkStoryHeader(MyFixedPosInputString & p)
{
    Header lHeader;
    lHeader.scanExternal(p);

    checkHeaderLevel(lHeader.level());
    checkStoryCodeSequence(lHeader.storyCode(), TRUE);
}


void InputChecker::checkIssueHeader(MyFixedPosInputString & p)
{
    Header lHeader;
    lHeader.scanExternal(p);

    checkHeaderLevel(lHeader.level());
    checkEntryCodeSequence(lHeader.storyCode(), TRUE);

    if (lHeader.level() != '1' && lHeader.level() != '2')
    {
        if (! lHeader.field(eMinNumber).empty())
        {
            aCheckLog.log(17, "Minnumber", lHeader.field(eMinNumber).element(0),
                                       "filled in on wrong level");
        }
        if (! lHeader.field(eMaxNumber).empty())
        {
            aCheckLog.log(17, "Maxnumber", lHeader.field(eMaxNumber).element(0),
                                       "filled in on wrong level");
        }
        // minnumber and maxnumber can only occur on levels 1 and 2
        return;
    }

    int lMinNumber = 1;

    if (! lHeader.field(eMinNumber).empty())
    {
        lMinNumber = atoi(lHeader.field(eMinNumber).element(0));
        if (lMinNumber > cMaxIssueNumber - 1)
        {
            aCheckLog.log(17, "Minnumber", lMinNumber, "too large");
        }
    }
    if (lHeader.field(eMaxNumber).empty())
    {
        // aCheckLog.log(18, "Maxnumber", "", "not filled in");
        // can also occur for level-1 headers, when maxnumber is filled
        // in on level 2!
    }
    else
    {
        const char * c = lHeader.field(eMaxNumber).element(0);
        switch (c[0])
        {
        case '?':
        case '-':
        case '!':
            break; // OK
        default:
            {
                int lMaxNumber = atoi(c);
                if (lMaxNumber < lMinNumber)
                {
                    aCheckLog.log(17, "Maxnumber", lMaxNumber, "smaller than minnumber");
                }
                if (lMaxNumber > cMaxIssueNumber - 1)
                {
                    aCheckLog.log(17, "Maxnumber", lMaxNumber, "too large");
                }
            }
        } // switch
    }
}


void InputChecker::processOneStoryFile(DBSFileIndex & pFile)
{
    MyFixedPosInputString lInputString;

    MyInputFile lStoryInputFile(pFile.fileName());
    aCheckLog.newFile(pFile.fileName());

    InputLineType lType;
    while ((lType = lInputString.read(lStoryInputFile)) != eEndOfFile)
    {
        aCheckLog.nextLine();

        switch (lType)
        {
        case eEntryLine:
            {
                ExternalEntry lEntry(pFile.format());
                lEntry.scanExternal(lInputString);
                if (lEntry.aErrorText != 0)
                {
                    aCheckLog.log(19, "Error during scan", "", lEntry.aErrorText);
                }
                checkEntry(lEntry, FALSE);
            }
            break;
        case eHeaderLine:
            checkStoryHeader(lInputString);
            break;
        case eWesternReprintLine:
            if (!pFile.isWestern())
            {
                aCheckLog.log(20, "", "", "reprint line while not in Western file");
            }
            // otherwise: the line will be checked when checking
            // the file as an Entry file
            break;
        case eCommentLine:
        case eFollowUpLine:
            break;
        case eEmptyLine:   // Nothing to check
            break;
        default:
            assert(FALSE);
        }
    }
}


void InputChecker::processOneEntryFile(const FileName & p, bool pIsWestern)
{
    MyInputFile lEntryInputFile(p);
    aCheckLog.newFile(p);

    MyFixedPosInputString lInputString;
    InputLineType lType;
    while ((lType = lInputString.read(lEntryInputFile)) != eEndOfFile)
    {
        aCheckLog.nextLine();

        if (pIsWestern)
        {
            if (lType == eWesternReprintLine)
            {
                ExternalEntry lEntry(eWesternDBIFormat);
                lEntry.scanExternal(lInputString);
                checkEntry(lEntry, TRUE);
            }
            // else: other lines are checked elsewhere
        }
        else
        {
            switch (lType)
            {
            case eEntryLine:
            case eWesternReprintLine: // probably never occurs...
                {
                    ExternalEntry lEntry(eDBIFormat);
                    lEntry.scanExternal(lInputString);
                    checkEntry(lEntry, TRUE);
                }
                break;
            case eHeaderLine:
                checkIssueHeader(lInputString);
                break;
            case eEmptyLine:
            case eCommentLine:
            case eFollowUpLine:
                break;
            default:
                assert(FALSE);
            }
        }
    }
}
