/* MYINPUT.H, (c) Harry Fluks 1994

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   Objects: MyInputFile, MyInputString, MyDelimitedInputString,
            MyFixedPosInputString

   String manipulation functions for reading fields in various
   formats.

   96-07-04  created
   96-08-23  checked
*/

#ifndef MYINPUT_H
#define MYINPUT_H

#include <string.h>
#include <assert.h>
#include "mylib.h"
#include "pointeri.h"
#include "pages.h"
#include "pagelayo.h"
#include "sortorde.h"

// next position indicates a header line, if filled with 'h'
const int cHeaderIndicatorPosition = 10;


typedef enum
{
    eEntryLine,           // normal line containing entry
    eWesternReprintLine,  // 'r'-line in Western file
    eHeaderLine,          // line marked with 'h' on position 10
    eCommentLine,         // line starting with space
    eFollowUpLine,        // line starting with '#', only in DBS files
    eEmptyLine,
    eEndOfFile
} InputLineType;


class MyInputFile
{
public:
    MyInputFile(const FileName &, bool pAppend = FALSE);
    virtual ~MyInputFile();

    FILE * theFile(void);

protected:
    FILE * aFile;
};


class MyInputString
{
    friend class MyInternalOutputFile; // can print my buffer
    friend class MyExternalOutputFile; // can print my buffer

protected:
    MyInputString(int pMaxLength = MAXLINEsize - 1);

    virtual ~MyInputString();

public:
    InputLineType read(MyInputFile &);

    virtual void reset(void) = 0;
    // derived classes can use this to re-initialise e.g. a pointer to
    // the beginning of the string

    bool empty(void) const;

    // next methods look in the string without looking at the structure (!)
    // this is to speed up things
    char start(void) const;
    int compare(const char * p) const;
    bool matchAnyText(const char * pMatchText) const;

    void replace(char, char);
    // replaces all occurrences of parameter 1 by parameter 2.

    void pointIntoBuffer(bool p);
    // indication by another object that it points into our buffer.
    // we keep track of this and check on it in the destructor
    // p = TRUE: use buffer
    // p = FALSE: we don't use the buffer anymore

protected:
    InputLineType theType(void);
    void check(void) const;

    char * aBuffer;
    int    aMaxLength;
    int    aUsedByOthers; // number of other objects pointing into our buffer
};


class MyDelimitedInputString: public MyInputString
{
public:
    MyDelimitedInputString(int pMaxLength = MAXLINEsize - 1);

    virtual ~MyDelimitedInputString();

    bool atEnd(void) const;

    // following functions adjust the buffer position
    virtual void reset(void); // position to the beginning

    void skipSpaces(void);
    void getDelimitedString
        (char * pTarget, int pMaxLength, char pDelimiter = DELIMITER);
    void getDelimitedPointer
        (PointerIntoInputString &, char pDelimiter = DELIMITER);

    bool getPrefix
        (PointerIntoInputString & pPrefix,
         int pMaxLength,
         char pDelimiter = DELIMITER);
    bool getPrefixedComment
        (PointerIntoInputString & pPrefix,
         int pPrefixLength,
         PointerIntoInputString & pComment);

protected:
    void skip(void);

    char * aPosition;
};


class MyFixedPosInputString: public MyDelimitedInputString
// inheritance from MyDelimitedInputString, because the tail
// of the string mostly contains delimited fields.

// aPosition will be NULL as long as the features of MyDelimitedInputString
// should not be used, i.e. as long as "split" has not been called.
// Using in the wrong way results in an assertion failure
{
public:
    MyFixedPosInputString(int pMaxLength = MAXLINEsize - 1);
    virtual ~MyFixedPosInputString();

    void scanFixedPos(
        int pStartPos, int pMaxLength, char * pTarget) const;
    void scanFixedPos(
        int pStartPos, Pages & pTarget) const;
    void scanFixedPos(
        int pStartPos, PageLayout & pTarget) const;
    void scanFixedPosInBuffer(
        int pStartPos, int pMaxLength, PointerIntoInputString & pTarget);
    // scanFixedPosInBuffer is not const: we count references

    void split(char pChar, int pPosition);
    // right side of the split string can be used as a MyDelimitedInputString.
    // left side will be 0-terminated.
    //
    // "right" side is marked by pChar, or on pPosition if pChar not
    // present
};


// INLINE FUNCTIONS

inline FILE * MyInputFile::theFile(void)
{
    assert(aFile != 0);
    return aFile;
}


inline bool MyInputString::empty(void) const
{
    return aBuffer[0] == '\0';
}


inline char MyInputString::start(void) const
{
    return aBuffer[0];
}


inline int MyInputString::compare(const char * p) const
{
    return SortOrder::strncmp(aBuffer, p, strlen(p));
}


inline bool MyDelimitedInputString::atEnd(void) const
{
    return *aPosition == '\0';
}

#endif
