/* MYOUTPUT.CC, (c) Harry Fluks 1994

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   96-07-04  created
   96-08-22  checked
*/

#include <assert.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h> // exit
#include "myoutput.h"
#include "buffers.h"

static const char cForceNewLine = '\n';
// used in MyFormattedOutputFile

static LINEtype gCommonPurposeBuffer;
// static to avoid stack overflow in several functions
// only to be used as a very local buffer!

MyOutputString::MyOutputString(int pBufferSize)
:
    aBuffer(0),
    aMaxLength(pBufferSize - 1)
{
    aBuffer = Buffers::newBuffer(8, pBufferSize);
    clear();
}


MyOutputString::~MyOutputString()
{
    Buffers::deleteBuffer(8, aBuffer);
}


void MyOutputString::clear(void)
{
    aBuffer[0] = '\0';
    aEnd = aBuffer;
}


void MyOutputString::compress(void)
{
    check();
    trimTrail();

    if (strlen(aBuffer) < 48)
    {
        return;
    }

    char * c = &aBuffer[47];
    char * d = &aBuffer[48];

    while (*c == ' ')
    {
        c--;
    }

    if (d - c > 5)  // i.e. hero missing
    {
        c++;
        c++; // one space before the '#' - we want to keep SOME readability
        *c = '#'; // this corresponds to cEarlyTitleChar in 'external.h'
        c++;

        while ((*c++ = *d++) != '\0'); // safe string copy
    }
    check();
}


void MyOutputString::trimTrail(void)
{
    if (aBuffer[0] == '\0')
    {
        return;
    }

    check();
    goToEnd();

    aEnd--; // outside buffer if empty string!
    while ((*aEnd == ' ') && (aEnd >= aBuffer))
    {
        *aEnd = '\0';
        aEnd--;
    }
    aEnd = aBuffer; // to give it a safe value
}


void MyOutputString::fillSpacesToColumn(int pToColumn)
{
    while (strlen(aBuffer) < pToColumn)
    {
        strcat(aBuffer, " ");
    }
    check();
}


void MyOutputString::check(void) const
{
    assert(aBuffer != 0);
    assert(aEnd >= aBuffer);
    assert(aEnd <  aBuffer + aMaxLength);
}


void MyOutputString::goToEnd(void)
{
    while (*aEnd != '\0')
    {
        aEnd++;
    }
}


MyOutputFile::MyOutputFile(const FileName & p)
:
    aFile(0)
{
    // all output files always in subdir 'data'
    // output file names can be unix-style = long
    const char * lName = p.fullNameInDataDir();

    aFile = fopen(lName, "w");

    if (aFile == 0)
    {
        MyLib::log("cannot write file %s!", lName);
        printf("cannot write file %s!\n", lName);
        exit(1);
    }
}


MyOutputFile::~MyOutputFile()
{
    assert(aFile != 0);
    (void) fclose(aFile);
}


MyInternalOutputFile::MyInternalOutputFile(const FileName & p)
:
    MyOutputFile(p)
{
}


MyInternalOutputFile::~MyInternalOutputFile()
{
}


void MyInternalOutputFile::putPrefix(const char * pValue)
{
    fprintf(aFile, "%s:", pValue);
}


void MyInternalOutputFile::putDelimitedString(const char * pValue)
{
    // no need to write trailing spaces
    strcpy(gCommonPurposeBuffer, pValue);
    MyLib::trimtrail(gCommonPurposeBuffer);
    fprintf(aFile, "%s%c", gCommonPurposeBuffer, DELIMITER);
}


void MyInternalOutputFile::putDelimitedString(const PointerIntoInputString & p)
{
    // a PointerIntoInputString is guaranteed to have no trailing spaces!
    p.copyToFile(aFile);
    fputc(DELIMITER, aFile);
}


void MyInternalOutputFile::putDelimitedStringFixedLength(
    const char * pValue,
    int pLength)
{
    fprintf(aFile, "%-*s%c", pLength, pValue, DELIMITER);
}


void MyInternalOutputFile::putDelimitedStringFixedLength(
    const PointerIntoInputString & p,
    int pLength)
{
    p.copyToFileAligned(aFile, pLength);
    fputc(DELIMITER, aFile);
}


void MyInternalOutputFile::putDelimitedStringWithSpace(
    const PointerIntoInputString & p)
{
    p.copyToFile(aFile);
    fputc(' ', aFile);
    fputc(DELIMITER, aFile);
}


void MyInternalOutputFile::putDelimitedInteger(const int pValue)
{
    fprintf(aFile, "%d%c", pValue, DELIMITER);
}


void MyInternalOutputFile::putInputString(const MyInputString & p)
{
    fputs(p.aBuffer, aFile);
}


void MyInternalOutputFile::putNewLine(void)
{
    Progress::addWritten();
    fprintf(aFile, "\n");
}


MyExternalOutputFile::MyExternalOutputFile(const FileName &p)
:
    MyOutputFile(p)
{
}


MyExternalOutputFile::~MyExternalOutputFile()
{
}


void MyExternalOutputFile::putInputString(const MyInputString & p)
{
    putS(p.aBuffer); // calling a virtual function
}


void MyExternalOutputFile::headerPrefix(int)
{
    printLine(); // calling a virtual function
}


void MyExternalOutputFile::headerPostfix(int)
{
    printLine(); // calling a virtual function
    printLine();
}


void MyExternalOutputFile::setProgramId(const char * p)
{
    gProgramId = p;
}


void MyExternalOutputFile::writeExtraText(const FileName & p)
{
    assert (aFile != 0);

    FILE * lTextFile = fopen("dizni.txt", "r");

    if (lTextFile == 0)
    {
        MyLib::log("cannot open file dizni.txt!");
        printf("cannot open file dizni.txt!\n");
        exit(1);
    }

    bool lInText = FALSE;
    char lAnotherBuffer[80];
    sprintf(lAnotherBuffer, "#%s", p.fullNameNoDir());
    assert(strlen(lAnotherBuffer) < 80);

    while (fgets(gCommonPurposeBuffer, MAXLINEsize - 1, lTextFile))
    {
        if (gCommonPurposeBuffer[0] == '#')
        {
            if (lInText)
            {
                break; // exit WHILE loop
            }
            lInText = (strncmp(gCommonPurposeBuffer,
                               lAnotherBuffer,
                               strlen(lAnotherBuffer)) == 0);
        }
        else if (lInText)
        {
            fprintf(aFile, "%s", gCommonPurposeBuffer); // includes newline
            Progress::addWritten();
        }
    }

    if (lInText) // we found text in the file
    {
        fprintf(aFile, "\n");
        Progress::addWritten();
    }
    fclose(lTextFile);
}


const char * MyExternalOutputFile::gProgramId = "DIZNI";


MyBufferedOutputFile::MyBufferedOutputFile(
    const FileName & p,
    int pBufferSize)
:
    MyExternalOutputFile(p),
    aOutputString(pBufferSize)
{
}


MyBufferedOutputFile::~MyBufferedOutputFile()
{
}


void MyBufferedOutputFile::printF(const char * pFormat, ...)
{
    va_list lVaList;
    va_start(lVaList, pFormat);
    vsprintf(aOutputString.end(), pFormat, lVaList);
    va_end(lVaList);
    aOutputString.check();
}


void MyBufferedOutputFile::putS(const char * pString)
{
    strcpy(aOutputString.end(), pString);
    aOutputString.check();
}


void MyBufferedOutputFile::putS(const PointerIntoInputString & p)
{
    p.copyToChars(aOutputString.end(), MAXLINEsize);
    aOutputString.check();
}


void MyBufferedOutputFile::putSAligned(
    const PointerIntoInputString & p, int pWidth)
{
    p.copyToCharsAligned(aOutputString.end(), pWidth);
    aOutputString.check();
}


MyUnformattedOutputFile::MyUnformattedOutputFile(
    const FileName & p,
    int pBufferSize)
:
    MyBufferedOutputFile(p, pBufferSize)
{
}


MyUnformattedOutputFile::~MyUnformattedOutputFile()
{
}


void MyUnformattedOutputFile::printLine(void)
{
    // simply print the buffer
    aOutputString.trimTrail();
    assert(aFile != 0);
    fprintf(aFile, "%s\n", aOutputString.buffer());
    Progress::addWritten();
    aOutputString.clear();
}


void MyUnformattedOutputFile::compress(void)
{
    aOutputString.compress();
}


MyFormattedOutputFile::MyFormattedOutputFile(
    const FileName & p,
    int pMaxLineWidth,
    int pIndentation,
    int pBufferSize)
:
    MyBufferedOutputFile(p, pBufferSize),
    aMaxLineWidth(pMaxLineWidth),
    aIndentation(pIndentation)
{
    assert(aIndentation < aMaxLineWidth);
    writeExtraText(p);
}


MyFormattedOutputFile::~MyFormattedOutputFile()
{
    assert(aFile != 0);
    fprintf(aFile, "\nThis file is part of the Disney comics Database,\n");
    fprintf(aFile, "and generated by %s.\n", gProgramId);
    fprintf(aFile, "For more information contact fluks4@pi.net\n");
    Progress::addWritten();
    Progress::addWritten();
    Progress::addWritten();
    Progress::addWritten();
}


void MyFormattedOutputFile::forceNewLine(void)
{
    sprintf(aOutputString.end(), "%c", cForceNewLine);
    aOutputString.check();
}


void MyFormattedOutputFile::printLine(void)
{
    if (aOutputString.empty())
    {
        fprintf(aFile, "\n");
        Progress::addWritten();
    }
    else
    {
        aOutputString.trimTrail();
        printControlled(aOutputString.buffer());
        aOutputString.clear();
    }
}


void MyFormattedOutputFile::fillSpacesToColumn(int p)
{
    aOutputString.fillSpacesToColumn(p);
}


void MyFormattedOutputFile::printControlled(char * pString)
{
    assert(aFile != 0);

    char * lStart = pString;
    char * lEnd = pString;
    bool lFirst = TRUE;
    char lLostChar = ' ';

    while (*lEnd != '\0')
    {
        lLostChar = ' ';
        int i = aIndentation;
        if (lFirst)
        {
            i = 0;
        }
        char * lLostCharPtr = 0;

        while (i < aMaxLineWidth)
        {
            switch (*lEnd)
            {
            case '\0':
            case cForceNewLine:
                goto ready;
            case ' ':
            case ',':
            case '-':
                lLostCharPtr = lEnd;
                lLostChar = *lEnd;
                break;
            default:
                ;
            }

            lEnd++;
            i++;
        }
ready:
        switch (*lEnd)
        {
        case ' ':
        case cForceNewLine:
            *lEnd = '\0';
            lEnd++;
            lLostChar = ' ';
            break;
        case '\0':
            lLostChar = ' ';
            break;
        default:
            if (lLostCharPtr != 0)
            {
                lEnd = lLostCharPtr;
                *lEnd = '\0';
            }
            else
            {
                lLostChar = *lEnd;
                *lEnd = '\0';
                MyLib::log("Word too long (%s)", lStart);
            }
            lEnd++;
        }

        if (lFirst)
        {
            lFirst = FALSE;
        }
        else
        {
            fputc('|', aFile);
            // will only go wrong if aIndentation < 8, i.e. no tabs are
            // written. Well, who cares...

            int j = aIndentation;
            while (j >= 8)
            {
                fputc('\t', aFile);
                j -= 8;
            }
            while (j > 0)
            {
                fputc(' ', aFile);
                j--;
            }

            // alternative: fprintf(aFile, "|%*s", aIndentation - 1, " ");
            // this would write a lot of spaces
        }

        if (lLostChar == ' ')
        {
            fprintf(aFile, "%s", lStart);
        }
        else
        {
            fprintf(aFile, "%s%c", lStart, lLostChar);
        }
        fprintf(aFile, "\n");
        Progress::addWritten();
        lStart = lEnd;
    } // end while
}


MyWebPageOutputFile::MyWebPageOutputFile(const FileName & p)
:
    MyExternalOutputFile(p)
{
    fprintf(aFile, "<html>\n");
    Progress::addWritten();
}


MyWebPageOutputFile::~MyWebPageOutputFile()
{
    fprintf(aFile, "</html>\n");
    Progress::addWritten();
}


void MyWebPageOutputFile::printF(const char * pFormat, ...)
{
    va_list lVaList;
    va_start(lVaList, pFormat);
    vfprintf(aFile, pFormat, lVaList);
    va_end(lVaList);
}


void MyWebPageOutputFile::putS(const char * pString)
{
    fputs(pString, aFile);
}


void MyWebPageOutputFile::putS(const PointerIntoInputString & p)
{
    p.copyToFile(aFile);
}


void MyWebPageOutputFile::putSAligned(const PointerIntoInputString &, int)
{
    assert(FALSE); // we don't need this
}


void MyWebPageOutputFile::printLine(void)
{
    assert(aFile != 0);
    fprintf(aFile, "<br>\n");
    Progress::addWritten();
}


void MyWebPageOutputFile::headerPrefix(int pLevel)
{
    assert(aFile != 0);
    fprintf(aFile, "<p><h%d>\n", pLevel);
    Progress::addWritten();
}


void MyWebPageOutputFile::headerPostfix(int pLevel)
{
    assert(aFile != 0);
    fprintf(aFile, "</h%d>\n", pLevel);
    Progress::addWritten();
}


MyFluksWebPageOutputFile::MyFluksWebPageOutputFile(const FileName & p)
:
    MyWebPageOutputFile(p)
{
    fprintf(aFile, "<head>\n");
    fprintf(aFile, "<title>Disney comics Database (%s)</title>\n", p.fullNameNoDir());
    fprintf(aFile, "<meta name=\"GENERATOR\" content=\"%s\">\n", gProgramId);
    fprintf(aFile, "</head>\n");
    Progress::addWritten();
    Progress::addWritten();
    Progress::addWritten();
    Progress::addWritten();

    writeExtraText(FileName("fheader", "web"));
    writeExtraText(p);
}


MyFluksWebPageOutputFile::~MyFluksWebPageOutputFile()
{
    printLine();
    fprintf(aFile, "<hr>This page is part of the Disney comics Database, ");
    fprintf(aFile, "and generated by %s.", gProgramId);
    printLine();
    writeExtraText(FileName("ffooter", "web"));
}


MyBarksWebPageOutputFile::MyBarksWebPageOutputFile(const FileName & p)
:
    MyWebPageOutputFile(p)
{
    fprintf(aFile, "<head>\n");
    fprintf(aFile, "<title>Disney comics Database (%s)</title>\n", p.fullNameNoDir()); // ** something else
    fprintf(aFile, "<meta name=\"GENERATOR\" content=\"%s\">\n", gProgramId);
    fprintf(aFile, "</head>\n");
    Progress::addWritten();
    Progress::addWritten();
    Progress::addWritten();
    Progress::addWritten();
    writeExtraText(FileName("bheader", "web"));
}


MyBarksWebPageOutputFile::~MyBarksWebPageOutputFile()
{
    printLine();
    fprintf(aFile, "<hr>This page is generated by %s.", gProgramId);
    printLine();
    writeExtraText(FileName("bfooter", "web"));
}
