/* PERSONLE(gend).CC, (c) Harry Fluks 1994, 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   95-06-22  created
   96-08-13  checked
*/

#include <assert.h>
#include "personle.h"

PersonLegend::PersonLegend(void)
:
    aHeroList(FALSE),
    aCreatorList(FALSE),  // FALSE: we don't need complete persons
    aSubseriesList(FALSE)
{
    aHeroList     .readFromInternalFile(FileName("heroes", "inl"));
    aCreatorList  .readFromInternalFile(FileName("creators", "inl"));
    aSubseriesList.readFromInternalFile(FileName("subserie", "inl"));

    CountryIndex c;
    while (c.next())
    {
        aSeriesList[c.number()] = new SeriesList(c);
        assert (aSeriesList[c.number()] != 0);
    }
}


PersonLegend::~PersonLegend()
{
    CountryIndex c;
    while (c.next())
    {
        delete aSeriesList[c.number()];
    }
}


void PersonLegend::add(Entry & pEntry, const CountrySet & pCountrySet)
{
    aHeroList.usePersons     (pEntry.field(eHero));
    aHeroList.usePersons     (pEntry.field(eAppearing));
    aCreatorList.usePersons  (pEntry.field(ePlotter));
    aCreatorList.usePersons  (pEntry.field(eWriter));
    aCreatorList.usePersons  (pEntry.field(eArtist));
    aCreatorList.usePersons  (pEntry.field(eInker));
    aSubseriesList.usePersons(pEntry.field(eSer));

    CountryIndex c;
    while (c.next())
    {
        if (pCountrySet.includes(c))
        {
            aSeriesList[c.number()]->useReprintIssues(pEntry);
        }
    }
}


void PersonLegend::reInitialise(void)
{
    aHeroList.reInitialise();
    aCreatorList.reInitialise();
    aSubseriesList.reInitialise();

    CountryIndex c;
    while (c.next())
    {
        aSeriesList[c.number()]->reInitialise();
    }
}


const char * PersonLegend::find(
    const char * pKey,
    ShortListFieldEnum pEnum,
    bool pInLatin1)
{
    // returns the full name from the right table, or null if not found
    switch (pEnum)
    {
    case eHero:
    case eAppearing:
        return aHeroList.find(pKey, pInLatin1);
    case ePlotter:
    case eWriter:
    case eArtist:
    case eInker:
        return aCreatorList.find(pKey, pInLatin1);
    case eSer:
        return aSubseriesList.find(pKey, pInLatin1);
    default:
        return 0; // no list for these
    }
}


const char * PersonLegend::find(
    const char * pKey,
    CountryIndex pCountry)
{
    assert (aSeriesList[pCountry.number()] != 0);
    return aSeriesList[pCountry.number()]->find(pKey);
}


void PersonLegend::write(MyExternalOutputFile & pFile, bool pInLatin1)
{
    writeLocal(pFile, FALSE, pInLatin1);
}


void PersonLegend::writeWeb(MyExternalOutputFile & pFile)
{
    writeLocal(pFile, TRUE, TRUE);
}


void PersonLegend::writeLocal(
    MyExternalOutputFile & pFile, bool pWeb, bool pInLatin1)
{
    int lHeaderLevel = 2;

    if (pWeb)
    {
        pFile.headerPrefix(2);
        pFile.printF("Legend");
        pFile.headerPostfix(2);
        lHeaderLevel = 3;
    }
    else
    {
        // not of interest for web pages;
        // Persons/Subseries are written by their full names directly
        aHeroList.writeFileLegend(pFile, "heroes", pInLatin1);
        aCreatorList.writeFileLegend(pFile, "creators", pInLatin1);
        aSubseriesList.writeFileLegend(pFile, "subseries", pInLatin1);

        CountryIndex c;
        while (c.next())
        {
            aSeriesList[c.number()]->writeFileLegend(pFile);
        }
    }

    if (!pWeb)
    {
        // not of interest for web pages;
        // comments are written in a different way
        pFile.headerPrefix(lHeaderLevel);
        pFile.printF("Possible comments between []:");
        pFile.headerPostfix(lHeaderLevel);
        pFile.printF(" plot: plotter");                pFile.printLine();
        pFile.printF(" writ: writer");                 pFile.printLine();
        pFile.printF(" art:  artist");                 pFile.printLine();
        pFile.printF(" ink:  inker");                  pFile.printLine();
        pFile.printF(" app:  appearing characters");   pFile.printLine();
        pFile.printF(" ser:  subseries");              pFile.printLine();
        // other fields occur less often
    }

    pFile.headerPrefix(lHeaderLevel);
    pFile.printF("Possible characters behind a reprint:");
    pFile.headerPostfix(lHeaderLevel);
    pFile.printF(" * changed (redrawn, pages missing, etc.)"); pFile.printLine();
    pFile.printF(" ' a few panels missing");                   pFile.printLine();
    pFile.printF(" + sequence starting in indicated issue");   pFile.printLine();

    if (!pWeb)
    {
        pFile.headerPrefix(lHeaderLevel);
        pFile.printF("Possible tags before a title:");
        pFile.headerPostfix(lHeaderLevel);
        pFile.printF(" T:Real English title");                 pFile.printLine();
        pFile.printF(" D:English description as printed outside the story"); pFile.printLine();
        pFile.printF(" O:Own description");                    pFile.printLine();
        pFile.printF(" Rde:Title as used in a reprint in Germany"); pFile.printLine();
        pFile.printF("     likewise Rdk, Rse, Rnl, etc.");     pFile.printLine();
    }

    pFile.printLine();
}
