/* PERSONLI(st).CC, (c) Harry Fluks 1994, 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   95-06-22  created (from LEGENDS.C)
   96-08-12  checked
   97-03-07  findPersonCode added
*/

#include <assert.h>
#include "personli.h"
#include "myoutput.h"
#include "linklist.h"
#include "buffers.h" // Buffers::usePool()

PersonListElement::PersonListElement(const char * pLastName)
:
    Person(pLastName)
{
}


PersonListElement::~PersonListElement()
{
}


int PersonListElement::compare(const char * pKey)
{
    assert(pKey != 0);
    return SortOrder::strcmp(key(), pKey);
}


PersonList::PersonList(bool pCompletePersons)
:
    aCompletePersons(pCompletePersons)
{
}


PersonList::~PersonList()
{
    Listable * l = aList.first();
    Listable * l2;
    while (l != 0)
    {
        l2 = l;
        l = aList.drop(l);  // remove from list
        delete l2;
    }
}


const char * PersonList::find(const char * pPersonCode, bool pInLatin1)
{
    PersonListElement * lPerson = (PersonListElement *) aList.find(pPersonCode);

    if (lPerson == 0)
    {
        return 0;
    }
    else
    {
        return lPerson->fullName(pInLatin1);
    }
}


const char * PersonList::findPersonCode(const char * pFullName)
{
    assert(aCompletePersons); // we'll be using Person::field()

    // linear search; the list is not sorted by full name!

    PersonListElement * lCurrent = (PersonListElement *) aList.first();

    while (lCurrent != 0)
    {
        if (strcmp(lCurrent->fullName(FALSE), pFullName) == 0)
        {
            return lCurrent->abbreviation();
        }
        else if (strcmp(lCurrent->fullName(TRUE), pFullName) == 0)
        {
            return lCurrent->abbreviation();
        }
        else if (lCurrent->field(eAlternativeNames).contains(pFullName))
        {
            return lCurrent->abbreviation();
        }
        else if (lCurrent->field(ePseudonym).contains(pFullName))
        {
            return lCurrent->abbreviation();
        }
        lCurrent = (PersonListElement *) lCurrent->next();
    }
    return 0;
}


void PersonList::usePersons(ShortList & pShl)
{
    int lNumber = pShl.numberOfElements();
    for (int i = 0; i < lNumber; i++)
    {
        PersonListElement * lPerson = (PersonListElement *)
            findClean(pShl.element(i));
        if (lPerson != 0)
        {
            lPerson->setUsed(TRUE);
        }
    }
}


void PersonList::writeFileLegend(
    MyExternalOutputFile & pOutputFile,
    const char * pHeaderText,
    bool pInLatin1) const
{
    bool lFirst = TRUE;
    PersonListElement * lCurrent = (PersonListElement *) aList.first();

    while (lCurrent != 0)
    {
        if (lCurrent->used())
        {
            if (lCurrent->hasAbbreviation())
            {
                if (lFirst)
                {
                    pOutputFile.headerPrefix(2);
                    pOutputFile.printF("Abbreviations of %s:", pHeaderText);
                    pOutputFile.headerPostfix(2);
                    lFirst = FALSE;
                }
                pOutputFile.printF("%-15s = %s",
                                   lCurrent->abbreviation(),
                                   lCurrent->fullName(pInLatin1));
                pOutputFile.printLine();
            }
        }
        lCurrent = (PersonListElement *) lCurrent->next();
    }
}


void PersonList::reInitialise(void)
{
    PersonListElement * lCurrent = (PersonListElement *) aList.first();

    while (lCurrent != 0)
    {
        lCurrent->setUsed(FALSE);
        lCurrent = (PersonListElement *) lCurrent->next();
    }
}


char * PersonList::cleanPerson(const char * pPerson, bool & pFirst, char * pBuffer)
{
    char * lBufPtr = pBuffer;
    const char * lPersonPtr = pPerson;
    int i = 0;
    pFirst = FALSE;

    while (*lPersonPtr != '\0')
    {
        if (  (  (*lPersonPtr == '?')  // a '?' on the 1st position..
              && (i > 0)               // ..should be listed
              )
           || (  (*lPersonPtr == '+')  // a '+' on the 2nd position..
              && (i > 1)               // ..should be listed
              )
           )
        {
            lPersonPtr++;
        }
        else if (*lPersonPtr == '(')
        {
            if (strncmp(lPersonPtr, "(1st", 4) == 0)
            {
                pFirst = TRUE;
            }
            break;
        }
        else if (*lPersonPtr == ')')
        {
            // comment to the Person, skip the rest
            break; // END WHILE LOOP
        }
        else if (*lPersonPtr == '[')
        {
            // comment to the Person, skip the rest
            break; // END WHILE LOOP
        }
        else
        {
            if (i < PERSONNAMElength)
            {
                *lBufPtr++ = *lPersonPtr;
            }
            lPersonPtr++;
        }
        i++;
    }
    *lBufPtr = '\0';

    MyLib::trimtrail(pBuffer);
    return pBuffer;
}


void PersonList::readFromInternalFile(const FileName & pFileName)
{
    MyFixedPosInputString lInputString;

    Buffers::usePool(FALSE);
    MyInputFile lFile(pFileName);

    while (lInputString.read(lFile) != eEndOfFile)
    {
        PersonListElement * lElement = new PersonListElement;
        assert(lElement != 0);
        lElement->scanInternal(lInputString, aCompletePersons);
        (void) aList.insert(lElement, lElement->key());
    }

    Buffers::usePool(TRUE);
    aList.createIndex();
}


Listable * PersonList::findClean(const char * pPerson)
{
    if ((pPerson[0] == '\0') || (pPerson[0] == ' '))
    {
        return 0; // not a person
    }

    PERSONNAMEtype lBuffer;
    bool lDummy;
    char * lCleanPerson = cleanPerson(pPerson, lDummy, lBuffer);
    assert(strlen(lCleanPerson) <= PERSONNAMElength);
    Listable * lPerson = aList.find(lCleanPerson);

    return lPerson; // may be null
}


void PersonList::readFromExternalFile(
    const FileName & pFileName,
    bool pCreatorLayout)
{
    assert(aCompletePersons);
    MyFixedPosInputString lInputString;

    Buffers::usePool(FALSE);
    MyInputFile lFile(pFileName);
    InputLineType lType;
    while ((lType = lInputString.read(lFile)) != eEndOfFile)
    {
        switch (lType)
        {
        case eCommentLine:
        case eEmptyLine:
            break;
        default:
            {
                PersonListElement * lElement = new PersonListElement;
                assert(lElement != 0);
                lElement->scanExternal(lInputString, pCreatorLayout);

                (void) aList.insert(lElement, lElement->key());
            }
        } // end switch
    }

    Buffers::usePool(TRUE);
    aList.createIndex();
}


void PersonList::addPersons(
    Entry & pEntry,
    ShortListFieldEnum pField,
    RoleFieldEnum pRole)
{
    assert(aCompletePersons);

    int i;
    int lNumber = pEntry.field(pField).numberOfElements();
    if (lNumber != 0)
    {
        ShortList & lShl = pEntry.field(pField);

        for (i = 0; i < lNumber; i++)
        {
            RoleFieldEnum lRole = pRole;
            if (pEntry.sPageLayout.isCoverOrSo())
            {
                lRole = eCoverRole;
            }

            bool lStoryIsFirst = FALSE;
            PersonListElement * lPerson = (PersonListElement *)
                add(lShl.element(i), lStoryIsFirst);
            if (lPerson != 0)
            {
                lPerson->update(lRole,
                                pEntry.country(),
                                1, // pNumberOfTimes
                                &pEntry.sStoryCode,
                                &pEntry.title(),
                                lStoryIsFirst);
            }
        }
    }
}


void PersonList::addPersons(
    Header & pHeader,
    HeaderFieldEnum pField,
    int pNumberOfTimes,
    RoleFieldEnum pRole)
{
    assert(aCompletePersons);
    ShortList & lShl = pHeader.field(pField);

    int lNumber = lShl.numberOfElements();
    for (int i = 0; i < lNumber; i++)
    {
        bool lStoryIsFirst = FALSE;
        PersonListElement * lPerson = (PersonListElement *)
            add(lShl.element(i), lStoryIsFirst);
        if (lPerson != 0)
        {
            lPerson->update(pRole,
                            pHeader.country(),
                            pNumberOfTimes,
                            0, 0, lStoryIsFirst);
        }
    }
}


void PersonList::writeToInternalFile(const FileName & pFileName)
{
    assert(aCompletePersons);
    MyInternalOutputFile lOutputFile(pFileName);

    PersonListElement * lCurrent =
        (PersonListElement *) aList.first();

    while (lCurrent != 0)
    {
        lCurrent->putInternal(lOutputFile);
        lCurrent = (PersonListElement *) lCurrent->next();
    }
}


Listable * PersonList::add(const char * pPerson, bool & pStoryIsFirst)
{
    if ((pPerson[0] == '\0') || (pPerson[0] == ' '))
    {
        return 0; // not a person
    }

    pStoryIsFirst = FALSE;
    char lBuffer[81];
    char * lCleanPerson = cleanPerson(pPerson, pStoryIsFirst, lBuffer);

    assert(strlen(lCleanPerson) < 81);
    Listable * lPerson = aList.find(lCleanPerson);

    if (lPerson == 0)
    {
//static int kl = 0; /*******/
//kl++;
//if (kl < 100)
//{
        // insert new Person with empty abbreviation
// *****        lPerson = new PersonListElement(lCleanPerson);
//        assert(lPerson != 0);
//        (void) aList.insert(lPerson, lCleanPerson);
//printf("  added %d [%s]\n", kl, lCleanPerson); // ******
//}
    }

    return lPerson; // may be null
}
