/* STORYWIT(hentries).CC, (c) Harry Fluks 1995

   Permission is hereby granted for unlimited modification, use, and
   distribution.  This software is made available with no warranty of
   any kind, express or implied.  This copyright notice must remain
   intact in all versions of this software.

   The author would appreciate it if any bug fixes and enhancements were
   to be sent back to him for incorporation into future versions of this
   software.  Please send changes to fluks4@pi.net

   ----

   95-07-21  created (from GATHER)
   96-08-13  checked
   96-11-05  introducing StoryWithEntriesRecord and eXRefEntry
*/

#include <assert.h>
#include "storywit.h"
#include "internal.h"
#include "mylib.h"


void StoryWithEntriesRecord::init(void)
{
    sEntry = 0;
    sInputString = 0;
    sTitleOrigin = eNoTitle;
    sKindOfEntry = eNoEntry;
}


void StoryWithEntriesRecord::fill(
    InternalEntry * pEntry,
    MyDelimitedInputString * pInputString)
{
    sEntry = pEntry;
    sInputString = pInputString;

    if (sEntry->title().empty())
    {
        sTitleOrigin = eNoTitle; // empty title
    }
    else
    {
        sTitleOrigin = eTitleHere;
    }

    // determine sKindOfEntry
    if (sEntry->field(eXRefd).empty())
    {
        if (sEntry->sEntryCode.empty())
        {
            sKindOfEntry = eStoryEntry;
        }
        else
        {
            sKindOfEntry = eReprintEntry;
        }
    }
    else
    {
        // eXRefd field can only be filled by 'preformat' - hence
        // this *must* be an eXRefEntry.
        sKindOfEntry = eXRefEntry;
    }
}


StoryWithEntries::StoryWithEntries(void)
:
    aNumber(0),
    aTheStoryIndex(-1),
    aOverflow(FALSE)
{
    for (int i = 0; i < cMaxEntriesPerStory; i++)
    {
        aRecord[i].init();
    }
}


StoryWithEntries::~StoryWithEntries()
{
    int i;
    for (i = 0; i < aNumber; i++)
    {
        if (aRecord[i].sEntry != 0)
        {
            delete aRecord[i].sEntry;
        }
    }
    // first all entries, then all inputstrings. Several
    // entries can point into the same inputstring!
    for (i = 0; i < aNumber; i++)
    {
        if (aRecord[i].sInputString != 0)
        {
            delete aRecord[i].sInputString;
        }
    }
}


InternalEntry & StoryWithEntries::entry(int pIndex) const
{
    assert(pIndex >= 0 && pIndex < aNumber);
    return *(aRecord[pIndex].sEntry);
}


StoryWithEntriesRecord::TitleOrigin & StoryWithEntries::titleOrigin(int pIndex)
{
    assert(pIndex >= 0 && pIndex < aNumber);
    return aRecord[pIndex].sTitleOrigin;
}


StoryWithEntriesRecord::KindOfEntry & StoryWithEntries::kindOfEntry(int pIndex)
{
    assert(pIndex >= 0 && pIndex < aNumber);
    return aRecord[pIndex].sKindOfEntry;
}


void StoryWithEntries::add(
    InternalEntry * pEntry,
    MyDelimitedInputString * pInputString)
{
    if (aOverflow)
    {
        // overflow, already logged
        delete pEntry;
        delete pInputString;
    }
    else if (aNumber >= cMaxEntriesPerStory)
    {
        // overflow, log this once
        MyLib::log("For code %s too many entries (max %d)",
                   pEntry->sStoryCode.zeroString(), cMaxEntriesPerStory);
        aOverflow = TRUE;
        delete pEntry;
        delete pInputString;
    }
    else
    {
        aRecord[aNumber].fill(pEntry, pInputString);

        if (aRecord[aNumber].sKindOfEntry == StoryWithEntriesRecord::eStoryEntry)
        {
            aTheStoryIndex = aNumber;
        }
        aNumber++;
    }
}


void StoryWithEntries::write( MyInternalOutputFile & pStoriesFile,
                              CountryFileArray & pEntriesFileArray) const
{
    for (int i = 0; i < aNumber; i++)
    {
        switch (aRecord[i].sKindOfEntry)
        {
        default:
            assert(FALSE);
        case StoryWithEntriesRecord::eNoEntry:
            // can be a former xref entry that was used
            break;
        case StoryWithEntriesRecord::eStoryEntry:
            aRecord[i].sEntry->put(pStoriesFile);
            break;
        case StoryWithEntriesRecord::eReprintEntry:
            {
                if (aTheStoryIndex == -1)
                {
                    // entry without corresponding story
                    aRecord[i].sEntry->setUnsolved();
                }

                MyInternalOutputFile & m =
                    pEntriesFileArray.element(aRecord[i].sEntry->country());
                aRecord[i].sEntry->put(m);
            }
            break;
        case StoryWithEntriesRecord::eXRefEntry:
            // xref entry that was not used
            MyLib::log("Unused xref (%s)(%s)",
                       aRecord[i].sEntry->sStoryCode.zeroString(),
                       aRecord[i].sEntry->field(eXRefd).delimitedString());
            break;
        }
    }
}


InternalEntry * StoryWithEntries::theStoryEntry(void)
{
    if (aTheStoryIndex == -1)
    {
        return 0;
    }
    else
    {
        return aRecord[aTheStoryIndex].sEntry;
    }
}
