/*
 *	pdp8/e emulator
 *
 *	Author:
 *		Bill Haygood
 *		41832 Ernest Road
 *		Loon Lake, WA 99148-9607
 *		Internet: billh@comtch.iea.com
 *		Voice/AnsMach/FAX \
 *			or	   509-233-2555
 *		  Cellular/Pager  /
 *
 *	Copyright 1992, 1993 by the author with all rights reserved.
 *
 *	Operate microinstruction (OPR) module for all OPRs except EAE.
 *
 *	Questions:
 *		1. Does every microprogrammed combo of MQL MQA
 *			yield a SWP instruction ? (page 3-17)
 */
#include "pdp8regs.h"

/*
 *	Operate microinstructions
 *
 *		|           |           |           |
 *		|---|---|---|---|---|---|---|---|---|
 *		| 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10| 11|	Group I
 *		| 0 |CLA|CLL|CMA|CML|RAR|RAL|BSW|IAC|
 *		|---|---|---|---|---|---|---|---|---|
 *	Sequence:     1   1   2   2   4   4   4   3
 *
 *		|           |           |           |
 *		|---|---|---|---|---|---|---|---|---|
 *		| 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10| 11|
 *		| 1 |CLA|SMA|SZA|SNL|FL |OSR|HLT| 0 |	Group II
 *		| 1 |CLA|SPA SNA SZL| AG|OSR|HLT| 0 |
 *		|---|---|---|---|---|---|---|---|---|
 *	Sequence:     2   1   1   1   1   3   4
 *
 */
VOID i7000 (VOID)			/* NOP				*/
{}
VOID i7001 (VOID)			/* IAC				*/
{
    AC = ++AC & 017777 ;
}
VOID i7002 (VOID)			/* BSW				*/
{
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7003 (VOID)			/* IAC BSW			*/
{
    AC++ ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7004 (VOID)			/* RAL				*/
{
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7005 (VOID)			/* IAC RAL			*/
{
    AC = ++AC & 017777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7006 (VOID)			/* RTL				*/
{
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7007 (VOID)			/* IAC RTL			*/
{
    AC = ++AC & 017777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7010 (VOID)			/* RAR				*/
{
    AC = ((AC << 12) + (AC >> 1)) & 017777 ;
}
VOID i7011 (VOID)			/* IAC RAR			*/
{
    AC = ++AC & 017777 ;
    AC = ((AC << 12) + (AC >> 1)) & 017777 ;
}
VOID i7012 (VOID)			/* RTR				*/
{
    AC = ((AC << 11) + (AC >> 2)) & 017777 ;
}
VOID i7013 (VOID)			/* IAC RTR			*/
{
    AC = ++AC & 017777 ;
    AC = ((AC << 11) + (AC >> 2)) & 017777 ;
}
VOID i7014 (VOID)			/* RAR RAL (reserved)		*/
{
    AC = (AC & 010000) | (AC & 07014) ;
}
VOID i7015 (VOID)			/* IAC RAR RAL (reserved)	*/
{
    AC = ++AC & 017015 ;
}
VOID i7016 (VOID)			/* RTR RTL (reserved)		*/
{
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7017 (VOID)			/* IAC RTR RTL (reserved)	*/
{
    AC = (++AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7020 (VOID)			/* CML				*/
{
    AC ^= 010000 ;
}
VOID i7021 (VOID)			/* CML IAC			*/
{
    AC = ((AC ^ 010000) + 1) & 017777 ;
}
VOID i7022 (VOID)			/* CML BSW			*/
{
    AC = ((AC ^ 010000) & 010000)
		+ ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7023 (VOID)			/* CML IAC BSW			*/
{
    AC = (AC ^ 010000) + 1 ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7024 (VOID)			/* CML RAL			*/
{
    AC ^= 010000 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7025 (VOID)			/* CML IAC RAL			*/
{
    AC = ((AC ^ 010000) + 1) & 017777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7026 (VOID)			/* CML RTL			*/
{
    AC = (((AC ^ 010000) >> 11) + (AC  << 2)) & 017777 ;
}
VOID i7027 (VOID)			/* CML IAC RTL			*/
{
    AC = ((AC ^ 010000) + 1) & 017777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7030 (VOID)			/* CML RAR			*/
{
    AC = ((AC << 12) + ((AC ^ 010000) >> 1)) & 017777 ;
}
VOID i7031 (VOID)			/* CML IAC RAR			*/
{
    AC = ((AC ^ 010000) + 1) & 017777 ;
    AC = ((AC << 12) + (AC >> 1)) & 017777 ;
}
VOID i7032 (VOID)			/* CML RTR			*/
{
    AC = (((AC ^ 010000) >> 2) + (AC << 11)) & 017777 ;
}
VOID i7033 (VOID)			/* CML IAC RTR			*/
{
    AC = ((AC ^ 010000) + 1) & 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7034 (VOID)			/* CML RAR RAL (reserved)	*/
{
    AC ^= 010000 ;
    AC &= 017034 ;
}
VOID i7035 (VOID)			/* CML IAC RAR RAL (reserved)	*/
{
    AC = ((AC ^ 010000) + 1) & 017035 ;
}
VOID i7036 (VOID)			/* CML RTR RTL (reserved)	*/
{
    AC ^= 010000 ;
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7037 (VOID)			/* CML IAC RTR RTL (reserved)	*/
{
    AC = (AC ^ 010000) + 1 ;
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7040 (VOID)			/* CMA				*/
{
    AC ^= 07777 ;
}
VOID i7041 (VOID)			/* CMA IAC			*/
{
    AC = ((AC ^ 07777) + 1) & 017777 ;
}
VOID i7042 (VOID)			/* CMA BSW			*/
{
    AC ^= 07777 ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7043 (VOID)			/* CMA IAC BSW			*/
{
    AC = (AC ^ 07777) + 1 ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7044 (VOID)			/* CMA RAL			*/
{
    AC ^= 07777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7045 (VOID)			/* CMA IAC RAL			*/
{
    AC = ((AC ^ 07777) + 1) & 017777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7046 (VOID)			/* CMA RTL			*/
{
    AC ^= 07777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7047 (VOID)			/* CMA IAC RTL			*/
{
    AC = ((AC ^ 07777) + 1) & 017777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7050 (VOID)			/* CMA RAR			*/
{
    AC ^= 07777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7051 (VOID)			/* CMA IAC RAR			*/
{
    AC = ((AC ^ 07777) + 1) & 017777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7052 (VOID)			/* CMA RTR			*/
{
    AC ^= 07777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7053 (VOID)			/* CMA IAC RTR			*/
{
    AC = ((AC ^ 07777) + 1) & 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7054 (VOID)			/* CMA RAR RAL (reserved)	*/
{
    AC ^= 07777 ;
    AC &= 017054 ;
}
VOID i7055 (VOID)			/* CMA IAC RAR RAL (reserved)	*/
{
    AC = ((AC ^ 07777) + 1) & 017055 ;
}
VOID i7056 (VOID)			/* CMA RTR RTL (reserved)	*/
{
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7057 (VOID)			/* CMA IAC RTR RTL (reserved)	*/
{
    AC = (((AC ^ 07777) + 1) & 017777) | (PC & 07600) | 016 ;
}
VOID i7060 (VOID)			/* CMA CML			*/
{
    AC ^= 017777 ;
}
VOID i7061 (VOID)			/* CMA CML IAC			*/
{
    AC = (-AC) & 017777 ;
}
VOID i7062 (VOID)			/* CMA CML BSW			*/
{
    AC ^= 017777 ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7063 (VOID)			/* CMA CML IAC BSW		*/
{
    AC = -AC ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7064 (VOID)			/* CMA CML RAL			*/
{
    AC ^= 017777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7065 (VOID)			/* CMA CML IAC RAL		*/
{
    AC = (-AC) & 017777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7066 (VOID)			/* CMA CML RTL			*/
{
    AC ^= 017777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7067 (VOID)			/* CMA CML IAC RTL		*/
{
    AC = (-AC) & 017777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7070 (VOID)			/* CMA CML RAR			*/
{
    AC ^= 017777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7071 (VOID)			/* CMA CML IAC RAR		*/
{
    AC = (-AC) & 017777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7072 (VOID)			/* CMA CML RTR			*/
{
    AC ^= 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7073 (VOID)			/* CMA CML IAC RTR		*/
{
    AC = (-AC) & 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7074 (VOID)			/* CMA CML RAR RAL (reserved)	*/
{
    AC ^= 017777 ;
    AC &= 017074 ;
}
VOID i7075 (VOID)			/* CMA CML IAC RAR RAL		*/
{					/* (reserved)			*/
    AC = (-AC) & 017075 ;
}
VOID i7076 (VOID)			/* CMA CML RTR RTL (reserved)	*/
{
    AC ^= 017777 ;
    AC = ((AC ^ 010000) & 010000) | (PC & 07600) | 016 ;
}
VOID i7077 (VOID)			/* CMA CML IAC RTR RTL		*/
{					/* (reserved)			*/
    AC = ((-AC) & 010000) | (PC & 07600) | 016 ;
}
VOID i7100 (VOID)			/* CLL				*/
{
    AC &= 07777 ;
}
VOID i7101 (VOID)			/* CLL IAC			*/
{
    AC = (AC & 07777) + 1 ;
}
VOID i7102 (VOID)			/* CLL BSW			*/
{
    AC = ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7103 (VOID)			/* CLL IAC BSW			*/
{
    AC = (AC & 07777) + 1 ;
    AC = (AC & 010000) + ((AC & 077) << 6) + ((AC & 07700) >> 6) ;
}
VOID i7104 (VOID)			/* CLL RAL			*/
{
    AC = (AC & 07777) << 1 ;
}
VOID i7105 (VOID)			/* CLL IAC RAL			*/
{
    AC = (AC & 07777) + 1 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7106 (VOID)			/* CLL RTL			*/
{
    AC &= 07777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7107 (VOID)			/* CLL IAC RTL			*/
{
    AC = (AC & 07777) + 1 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7110 (VOID)			/* CLL RAR			*/
{
    AC = ((AC & 07776) >> 1) + ((AC & 1) << 12) ;
}
VOID i7111 (VOID)			/* CLL IAC RAR			*/
{
    AC = (AC & 07777) + 1 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7112 (VOID)			/* CLL RTR			*/
{
    AC &= 07777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7113 (VOID)			/* CLL IAC RTR			*/
{
    AC = (AC & 07777) + 1 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7114 (VOID)			/* CLL RAR RAL (reserved)	*/
{
    AC &= 07114 ;
}
VOID i7115 (VOID)			/* CLL IAC RAR RAL (reserved)	*/
{
    AC = ((AC & 07777) + 1) & 017115 ;
}
VOID i7116 (VOID)			/* CLL RTR RTL (reserved)	*/
{
    AC = (PC & 07600) | 016 ;
}
VOID i7117 (VOID)			/* CLL IAC RTR RTL (reserved)	*/
{
    AC = (((AC & 07777) + 1) & 010000) | (PC & 07600) | 016 ;
}
VOID i7120 (VOID)			/* CLL CML			*/
{
    AC |= 010000 ;
}
VOID i7121 (VOID)			/* CLL CML IAC			*/
{
    AC = ((AC | 010000) + 1) & 017777 ;
}
VOID i7122 (VOID)			/* CLL CML BSW			*/
{
    AC = 010000 + ((AC & 07700) >> 6) + ((AC & 077) << 6) ;
}
VOID i7123 (VOID)			/* CLL CML IAC BSW		*/
{
    AC = (AC | 010000) + 1 ;
    AC = (AC & 010000) + ((AC & 07700) >> 6) + ((AC & 077) << 6) ;
}
VOID i7124 (VOID)			/* CLL CML RAL			*/
{
    AC |= 010000 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7125 (VOID)			/* CLL CML IAC RAL		*/
{
    AC = ((AC | 010000) + 1) & 017777 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7126 (VOID)			/* CLL CML RTL			*/
{
    AC |= 010000 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7127 (VOID)			/* CLL CML IAC RTL		*/
{
    AC = ((AC | 010000) + 1) & 017777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7130 (VOID)			/* CLL CML RAR			*/
{
    AC = ((AC | 010000) >> 1) + ((AC & 1) << 12) ;
}
VOID i7131 (VOID)			/* CLL CML IAC RAR		*/
{
    AC = ((AC | 010000) + 1) & 017777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7132 (VOID)			/* CLL CML RTR			*/
{
    AC = ((AC | 010000) >> 2) + ((AC & 3) << 11) ;
}
VOID i7133 (VOID)			/* CLL CML IAC RTR		*/
{
    AC = ((AC | 010000) + 1) & 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7134 (VOID)			/* CLL CML RAR RAL (reserved)	*/
{
    AC = (AC | 010000) & 017134 ;
}
VOID i7135 (VOID)			/* CLL CML IAC RAR RAL		*/
{					/* (reserved)			*/
    AC = ((AC | 010000) + 1) & 017135 ;
}
VOID i7136 (VOID)			/* CLL CML RTR RTL (reserved)	*/
{
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7137 (VOID)			/* CLL CML IAC RTR RTL		*/
{					/* (reserved)			*/
    AC = (((AC | 010000) + 1) & 010000) | (PC & 07600) | 016 ;
}
VOID i7140 (VOID)			/* CLL CMA			*/
{
    AC = (AC & 07777) ^ 07777 ;
}
VOID i7141 (VOID)			/* CLL CMA IAC			*/
{
    AC = ((AC & 07777) ^ 07777) + 1 ;
}
VOID i7142 (VOID)			/* CLL CMA BSW			*/
{
    AC ^= 07777 ;
    AC = ((AC & 07700) >> 6) + ((AC & 077) << 6) ;
}
VOID i7143 (VOID)			/* CLL CMA IAC BSW		*/
{
    AC = ((AC & 07777) ^ 07777) + 1 ;
    AC = (AC & 010000) + ((AC & 07700) >> 6) + ((AC & 077) << 6) ;
}
VOID i7144 (VOID)			/* CLL CMA RAL			*/
{
    AC = ((AC & 07777) ^ 07777) << 1 ;
}
VOID i7145 (VOID)			/* CLL CMA IAC RAL		*/
{
    AC = ((AC & 07777) ^ 07777) + 1 ;
    AC = ((AC >> 12) + (AC << 1)) & 017777 ;
}
VOID i7146 (VOID)			/* CLL CMA RTL			*/
{
    AC = (AC & 07777) ^ 07777 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7147 (VOID)			/* CLL CMA IAC RTL		*/
{
    AC = ((AC & 07777) ^ 07777) + 1 ;
    AC = ((AC >> 11) + (AC << 2)) & 017777 ;
}
VOID i7150 (VOID)			/* CLL CMA RAR			*/
{
    AC = (AC & 07777) ^ 07777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7151 (VOID)			/* CLL CMA IAC RAR		*/
{
    AC = ((AC & 07777) ^ 07777) + 1 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7152 (VOID)			/* CLL CMA RTR			*/
{
    AC = (AC & 07777) ^ 07777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7153 (VOID)			/* CLL CMA IAC RTR		*/
{
    AC = ((AC & 07777) ^ 07777) + 1 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7154 (VOID)			/* CLL CMA RAR RAL (reserved)	*/
{
    AC = ((AC & 07777) ^ 07777) & 07154 ;
}
VOID i7155 (VOID)			/* CLL CMA IAC RAR RAL		*/
{					/* (reserved)			*/
    AC = (((AC & 07777) ^ 07777) + 1) & 017155 ;
}
VOID i7156 (VOID)			/* CLL CMA RTR RTL (reserved)	*/
{
    AC = (PC & 07600) | 016 ;
}
VOID i7157 (VOID)			/* CLL CMA IAC RTR RTL		*/
{					/* (reserved)			*/
    AC = ((((AC & 07777) ^ 07777) + 1) & 010000) | (PC & 07600) | 016 ;
}
VOID i7160 (VOID)			/* CLL CMA CML			*/
{
    AC = (AC & 07777) ^ 017777 ;
}
VOID i7161 (VOID)			/* CLL CMA CML IAC		*/
{
    AC = (((AC & 07777) ^ 017777) + 1) & 017777 ;
}
VOID i7162 (VOID)			/* CLL CMA CML BSW		*/
{
    AC ^= 07777 ;
    AC = 010000 + ((AC & 07700) >> 6) + ((AC & 077) << 6) ;
}
VOID i7163 (VOID)			/* CLL CMA CML IAC BSW		*/
{
    AC = ((AC & 07777) ^ 017777) + 1 ;
    AC = (AC & 010000) + ((AC & 07700) >> 6) + ((AC & 077) << 6) ;
}
VOID i7164 (VOID)			/* CLL CMA CML RAL		*/
{
    AC = (((AC & 07777) ^ 07777) << 1) | 1 ;
}
VOID i7165 (VOID)			/* CLL CMA CML IAC RAL		*/
{
    AC = (((AC & 07777) ^ 017777) + 1) & 017777 ;
    AC = ((AC >> 12) | (AC << 1)) & 017777 ;
}
VOID i7166 (VOID)			/* CLL CMA CML RTL		*/
{
    AC = (AC & 07777) ^ 017777 ;
    AC = ((AC >> 11) | (AC << 2)) & 017777 ;
}
VOID i7167 (VOID)			/* CLL CMA CML IAC RTL		*/
{
    AC = (((AC & 07777) ^ 017777) + 1) & 017777 ;
    AC = ((AC >> 11) | (AC << 2)) & 017777 ;
}
VOID i7170 (VOID)			/* CLL CMA CML RAR		*/
{
    AC = (AC & 07777) ^ 017777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7171 (VOID)			/* CLL CMA CML IAC RAR		*/
{
    AC = (((AC & 07777) ^ 017777) + 1) & 017777 ;
    AC = ((AC >> 1) + (AC << 12)) & 017777 ;
}
VOID i7172 (VOID)			/* CLL CMA CML RTR		*/
{
    AC = (AC & 07777) ^ 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7173 (VOID)			/* CLL CMA CML IAC RTR		*/
{
    AC = (((AC & 07777) ^ 017777) + 1) & 017777 ;
    AC = ((AC >> 2) + (AC << 11)) & 017777 ;
}
VOID i7174 (VOID)			/* CLL CMA CML RAR RAL		*/
{					/* (reserved)			*/
    AC = ((AC & 07777) ^ 017777) & 017174 ;
}
VOID i7175 (VOID)			/* CLL CMA CML IAC RAR RAL	*/
{					/* (reserved)			*/
    AC = (((AC & 07777) ^ 017777) + 1) & 017175 ;
}
VOID i7176 (VOID)			/* CLL CMA CML RTR RTL		*/
{					/* (reserved)			*/
    AC = (((AC & 07777) ^ 017777) & 010000) | (PC & 07600) | 016 ;
}
VOID i7177 (VOID)			/* CLL CMA CML IAC RTR RTL	*/
{					/* (reserved)			*/
    AC = ((((AC & 07777) ^ 017777) + 1) & 010000) | (PC & 07600) | 016 ;
}
VOID i7200 (VOID)			/* CLA				*/
{
    AC &= 010000 ;
}
VOID i7201 (VOID)			/* CLA IAC			*/
{
    AC = (AC & 010000) + 1 ;
}
VOID i7202 (VOID)			/* CLA BSW			*/
{
    AC &= 010000 ;
}
VOID i7203 (VOID)			/* CLA IAC BSW			*/
{
    AC = (AC & 010000) + 0100 ;
}
VOID i7204 (VOID)			/* CLA RAL			*/
{
    AC >>= 12 ;
}
VOID i7205 (VOID)			/* CLA IAC RAL			*/
{
    AC = (AC >> 12) + 2 ;
}
VOID i7206 (VOID)			/* CLA RTL			*/
{
    AC = (AC & 010000) >> 11 ;
}
VOID i7207 (VOID)			/* CLA IAC RTL			*/
{
    AC = ((AC & 010000) >> 11) + 4 ;
}
VOID i7210 (VOID)			/* CLA RAR			*/
{
    AC = (AC & 010000) >> 1 ;
}
VOID i7211 (VOID)			/* CLA IAC RAR			*/
{
    AC = 010000 + ((AC & 010000) >> 1) ;
}
VOID i7212 (VOID)			/* CLA RTR			*/
{
    AC = (AC & 010000) >> 2 ;
}
VOID i7213 (VOID)			/* CLA IAC RTR			*/
{
    AC = 04000 + ((AC & 010000) >> 2) ;
}
VOID i7214 (VOID)			/* CLA RAR RAL (reserved)	*/
{
    AC &= 010000 ;
}
VOID i7215 (VOID)			/* CLA IAC RAR RAL (reserved)	*/
{
    AC = (AC & 010000) + 1 ;
}
VOID i7216 (VOID)			/* CLA RTR RTL (reserved)	*/
{
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7217 (VOID)			/* CLA IAC RTR RTL (reserved)	*/
{
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7220 (VOID)			/* CLA CML			*/
{
    AC = (AC & 010000) ^ 010000 ;
}
VOID i7221 (VOID)			/* CLA CML IAC			*/
{
    AC = ((AC & 010000) ^ 010000) + 1 ;
}
VOID i7222 (VOID)			/* CLA CML BSW			*/
{
    AC = (AC & 010000) ^ 010000 ;
}
VOID i7223 (VOID)			/* CLA CML IAC BSW		*/
{
    AC = ((AC & 010000) ^ 010000) + 0100 ;
}
VOID i7224 (VOID)			/* CLA CML RAL			*/
{
    AC = ((AC & 010000) ^ 010000) >> 12 ;
}
VOID i7225 (VOID)			/* CLA CML IAC RAL		*/
{
    AC = (((AC & 010000) ^ 010000) >> 12) + 2 ;
}
VOID i7226 (VOID)			/* CLA CML RTL			*/
{
    AC = ((AC & 010000) ^ 010000) >> 11 ;
}
VOID i7227 (VOID)			/* CLA CML IAC RTL		*/
{
    AC = (((AC & 010000) ^ 010000) >> 11) + 4 ;
}
VOID i7230 (VOID)			/* CLA CML RAR			*/
{
    AC = ((AC & 010000) ^ 010000) >> 1 ;
}
VOID i7231 (VOID)			/* CLA CML IAC RAR		*/
{
    AC = 010000 + (((AC & 010000) ^ 010000) >> 1) ;
}
VOID i7232 (VOID)			/* CLA CML RTR			*/
{
    AC = ((AC & 010000) ^ 010000) >> 2 ;
}
VOID i7233 (VOID)			/* CLA CML IAC RTR		*/
{
    AC = 04000 + ((AC & 010000) ^ 010000) >> 2 ;
}
VOID i7234 (VOID)			/* CLA CML RAR RAL (reserved)	*/
{
    AC = (AC & 010000) ^ 010000 ;
}
VOID i7235 (VOID)			/* CLA CML IAC RAR RAL		*/
{					/* (reserved)			*/
    AC = ((AC & 010000) ^ 010000) + 1 ;
}
VOID i7236 (VOID)			/* CLA CML RTR RTL (reserved)	*/
{
    AC = (((AC & 010000) ^ 010000) & 010000) | (PC & 07600) | 016 ;
}
VOID i7237 (VOID)			/* CLA CML IAC RTR RTL		*/
{					/* (reserved)			*/
    AC = (((AC & 010000) ^ 010000) & 010000) | (PC & 07600) | 016 ;
}
VOID i7240 (VOID)			/* CLA CMA			*/
{
    AC = (AC & 010000) + 07777 ;
}
VOID i7241 (VOID)			/* CLA CMA IAC			*/
{
    AC = (AC & 010000) ^ 010000 ;
}
VOID i7242 (VOID)			/* CLA CMA BSW			*/
{
    AC |= 07777 ;
}
VOID i7243 (VOID)			/* CLA CMA IAC BSW		*/
{
    AC = (AC & 010000) ^ 010000 ;
}
VOID i7244 (VOID)			/* CLA CMA RAL			*/
{
    AC = (AC >> 12) | 017776 ;
}
VOID i7245 (VOID)			/* CLA CMA IAC RAL		*/
{
    AC = ((AC & 010000) ^ 010000) >> 12 ;
}
VOID i7246 (VOID)			/* CLA CMA RTL			*/
{
    AC = 017775 + ((AC & 010000) >> 11) ;
}
VOID i7247 (VOID)			/* CLA CMA IAC RTL		*/
{
    AC = ((AC & 010000) ^ 010000) >> 11 ;
}
VOID i7250 (VOID)			/* CLA CMA RAR			*/
{
    AC = 013777 + ((AC & 010000) >> 1) ;
}
VOID i7251 (VOID)			/* CLA CMA IAC RAR		*/
{
    AC = ((AC & 010000) ^ 010000) >> 1 ;
}
VOID i7252 (VOID)			/* CLA CMA RTR			*/
{
    AC = 015777 + ((AC & 010000) >> 2) ;
}
VOID i7253 (VOID)			/* CLA CMA IAC RTR		*/
{
    AC = ((AC & 010000) ^ 010000) >> 2 ;
}
VOID i7254 (VOID)			/* CLA CMA RAR RAL (reserved)	*/
{
    AC = (AC & 010000) | 07254 ;
}
VOID i7255 (VOID)			/* CLA CMA IAC RAR RAL		*/
{					/* (reserved)			*/
    AC = (AC & 010000) ^ 010000 ;
}
VOID i7256 (VOID)			/* CLA CMA RTR RTL (reserved)	*/
{
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7257 (VOID)			/* CLA CMA IAC RTR RTL		*/
{					/* (reserved)			*/
    AC = ((AC & 010000) ^ 010000) | (PC & 07600) | 016 ;
}
VOID i7260 (VOID)			/* CLA CMA CML			*/
{
    AC = (AC | 07777) ^ 010000 ;
}
VOID i7261 (VOID)			/* CLA CMA CML IAC		*/
{
    AC &= 010000 ;
}
VOID i7262 (VOID)			/* CLA CMA CML BSW		*/
{
    AC = (AC | 07777) ^ 010000 ;
}
VOID i7263 (VOID)			/* CLA CMA CML IAC BSW		*/
{
    AC &= 010000 ;
}
VOID i7264 (VOID)			/* CLA CMA CML RAL		*/
{
    AC = 017776 | (AC >> 12) ;
}
VOID i7265 (VOID)			/* CLA CMA CML IAC RAL		*/
{
    AC = (AC & 010000) >> 12 ;
}
VOID i7266 (VOID)			/* CLA CMA CML RTL		*/
{
    AC = 017775 | (AC >> 11) ;
}
VOID i7267 (VOID)			/* CLA CMA CML IAC RTL		*/
{
    AC = (AC & 010000) >> 11 ;
}
VOID i7270 (VOID)			/* CLA CMA CML RAR		*/
{
    AC = 013777 | (AC >> 1) ;
}
VOID i7271 (VOID)			/* CLA CMA CML IAC RAR		*/
{
    AC = (AC & 010000) >> 1 ;
}
VOID i7272 (VOID)			/* CLA CMA CML RTR		*/
{
    AC = 015777 | (AC >> 2) ;
}
VOID i7273 (VOID)			/* CLA CMA CML IAC RTR		*/
{
    AC = (AC & 010000) >> 2 ;
}
VOID i7274 (VOID)			/* CLA CMA CML RAR RAL		*/
{					/* (reserved)			*/
    AC = ((AC & 010000) ^ 010000) | 07274 ;
}
VOID i7275 (VOID)			/* CLA CMA CML IAC RAR RAL	*/
{					/* (reserved)			*/
    AC &= 010000 ;
}
VOID i7276 (VOID)			/* CLA CMA CML RTR RTL		*/
{					/* (reserved)			*/
    AC = ((AC & 010000) ^ 010000) | (PC & 07600) | 016 ;
}
VOID i7277 (VOID)			/* CLA CMA CML IAC RTR RTL	*/
{					/* (reserved)			*/
    AC = (AC & 010000) | (PC & 07600) | 016 ;
}
VOID i7300 (VOID)			/* CLA CLL			*/
{
    AC = 0 ;
}
VOID i7301 (VOID)			/* CLA CLL IAC			*/
{
    AC = 1 ;
}
VOID i7302 (VOID)			/* CLA CLL BSW			*/
{
    AC = 0 ;
}
VOID i7303 (VOID)			/* CLA CLL IAC BSW		*/
{
    AC = 0100 ;
}
VOID i7304 (VOID)			/* CLA CLL RAL			*/
{
    AC = 0 ;
}
VOID i7305 (VOID)			/* CLA CLL IAC RAL		*/
{
    AC = 2 ;
}
VOID i7306 (VOID)			/* CLA CLL RTL			*/
{
    AC = 0 ;
}
VOID i7307 (VOID)			/* CLA CLL IAC RTL		*/
{
    AC = 4 ;
}
VOID i7310 (VOID)			/* CLA CLL RAR			*/
{
    AC = 0 ;
}
VOID i7311 (VOID)			/* CLA CLL IAC RAR		*/
{
    AC = 010000 ;
}
VOID i7312 (VOID)			/* CLA CLL RTR			*/
{
    AC = 0 ;
}
VOID i7313 (VOID)			/* CLA CLL IAC RTR		*/
{
    AC = 04000 ;
}
VOID i7314 (VOID)			/* CLA CLL RAR RAL (reserved)	*/
{
    AC = 0 ;
}
VOID i7315 (VOID)			/* CLA CLL IAC RAR RAL		*/
{					/* (reserved)			*/
    AC = 1 ;
}
VOID i7316 (VOID)			/* CLA CLL RTR RTL (reserved)	*/
{
    AC = (PC & 07600) | 016 ;
}
VOID i7317 (VOID)			/* CLA CLL IAC RTR RTL		*/
{					/* (reserved)			*/
    AC = (PC & 07600) | 016 ;
}
VOID i7320 (VOID)			/* CLA CLL CML			*/
{
    AC = 010000 ;
}
VOID i7321 (VOID)			/* CLA CLL CML IAC		*/
{
    AC = 010001 ;
}
VOID i7322 (VOID)			/* CLA CLL CML BSW		*/
{
    AC = 010000 ;
}
VOID i7323 (VOID)			/* CLA CLL CML IAC BSW		*/
{
    AC = 010100 ;
}
VOID i7324 (VOID)			/* CLA CLL CML RAL		*/
{
    AC = 1 ;
}
VOID i7325 (VOID)			/* CLA CLL CML IAC RAL		*/
{
    AC = 3 ;
}
VOID i7326 (VOID)			/* CLA CLL CML RTL		*/
{
    AC = 2 ;
}
VOID i7327 (VOID)			/* CLA CLL CML IAC RTL		*/
{
    AC = 6 ;
}
VOID i7330 (VOID)			/* CLA CLL CML RAR		*/
{
    AC = 04000 ;
}
VOID i7331 (VOID)			/* CLA CLL CML IAC RAR		*/
{
    AC = 014000 ;
}
VOID i7332 (VOID)			/* CLA CLL CML RTR		*/
{
    AC = 02000 ;
}
VOID i7333 (VOID)			/* CLA CLL CML IAC RTR		*/
{
    AC = 06000 ;
}
VOID i7334 (VOID)			/* CLA CLL CML RAR RAL		*/
{					/* (reserved)			*/
    AC = 010000 ;
}
VOID i7335 (VOID)			/* CLA CLL CML IAC RAR RAL	*/
{					/* (reserved)			*/
    AC = 010001 ;
}
VOID i7336 (VOID)			/* CLA CLL CML RTR RTL		*/
{					/* (reserved)			*/
    AC = 010000 | (PC & 07600) | 016 ;
}
VOID i7337 (VOID)			/* CLA CLL CML IAC RTR RTL	*/
{					/* (reserved)			*/
    AC = 010000 | (PC & 07600) | 016 ;
}
VOID i7340 (VOID)			/* CLA CLL CMA			*/
{
    AC = 07777 ;
}
VOID i7341 (VOID)			/* CLA CLL CMA IAC		*/
{
    AC = 010000 ;
}
VOID i7342 (VOID)			/* CLA CLL CMA BSW		*/
{
    AC = 07777 ;
}
VOID i7343 (VOID)			/* CLA CLL CMA IAC BSW		*/
{
    AC = 010000 ;
}
VOID i7344 (VOID)			/* CLA CLL CMA RAL		*/
{
    AC = 017776 ;
}
VOID i7345 (VOID)			/* CLA CLL CMA IAC RAL		*/
{
    AC = 1 ;
}
VOID i7346 (VOID)			/* CLA CLL CMA RTL		*/
{
    AC = 017775 ;
}
VOID i7347 (VOID)			/* CLA CLL CMA IAC RTL		*/
{
    AC = 02 ;
}
VOID i7350 (VOID)			/* CLA CLL CMA RAR		*/
{
    AC = 013777 ;
}
VOID i7351 (VOID)			/* CLA CLL CMA IAC RAR		*/
{
    AC = 04000 ;
}
VOID i7352 (VOID)			/* CLA CLL CMA RTR		*/
{
    AC = 05777 ;
}
VOID i7353 (VOID)			/* CLA CLL CMA IAC RTR		*/
{
    AC = 02000 ;
}
VOID i7354 (VOID)			/* CLA CLL CMA RAR RAL		*/
{					/* (reserved)			*/
    AC = 07354 ;
}
VOID i7355 (VOID)			/* CLA CLL CMA IAC RAR RAL	*/
{					/* (reserved)			*/
    AC = 010000 ;
}
VOID i7356 (VOID)			/* CLA CLL CMA RTR RTL		*/
{					/* (reserved)			*/
    AC = (PC & 07600) | 016 ;
}
VOID i7357 (VOID)			/* CLA CLL CMA IAC RTR RTL	*/
{					/* (reserved)			*/
    AC = 010000 | (PC & 07600) | 016 ;
}
VOID i7360 (VOID)			/* CLA CLL CMA CML		*/
{
    AC = 017777 ;
}
VOID i7361 (VOID)			/* CLA CLL CMA CML IAC		*/
{
    AC = 0 ;
}
VOID i7362 (VOID)			/* CLA CLL CMA CML BSW		*/
{
    AC = 017777 ;
}
VOID i7363 (VOID)			/* CLA CLL CMA CML IAC BSW	*/
{
    AC = 0 ;
}
VOID i7364 (VOID)			/* CLA CLL CMA CML RAL		*/
{
    AC = 017777 ;
}
VOID i7365 (VOID)			/* CLA CLL CMA CML IAC RAL	*/
{
    AC = 0 ;
}
VOID i7366 (VOID)			/* CLA CLL CMA CML RTL		*/
{
    AC = 017777 ;
}
VOID i7367 (VOID)			/* CLA CLL CMA CML IAC RTL	*/
{
    AC = 0 ;
}
VOID i7370 (VOID)			/* CLA CLL CMA CML RAR		*/
{
    AC = 017777 ;
}
VOID i7371 (VOID)			/* CLA CLL CMA CML IAC RAR	*/
{
    AC = 0 ;
}
VOID i7372 (VOID)			/* CLA CLL CMA CML RTR		*/
{
    AC = 017777 ;
}
VOID i7373 (VOID)			/* CLA CLL CMA CML IAC RTR	*/
{
    AC = 0 ;
}
VOID i7374 (VOID)			/* CLA CLL CMA CML RAR RAL	*/
{					/* (reserved)			*/
    AC = 017374 ;
}
VOID i7375 (VOID)			/* CLA CLL CMA CML IAC RAR RAL	*/
{					/* (reserved)			*/
    AC = 0 ;
}
VOID i7376 (VOID)			/* CLA CLL CMA CML RTR RTL	*/
{					/* (reserved)			*/
    AC = 010000 | (PC & 07600) | 016 ;
}
VOID i7377 (VOID)			/* CLA CLL CMA CML IAC RTR RTL	*/
{					/* (reserved)			*/
    AC = (PC & 07600) | 016 ;
}
VOID i7400 (VOID)			/* NOP	Grp II			*/
{}
VOID i7402 (VOID)			/* HLT				*/
{
    run = FALSE ;
}
VOID i7404 (VOID)			/* OSR				*/
{
    AC |= SR ;
}
VOID i7406 (VOID)			/* OSR HLT			*/
{
    AC |= SR ;
    run = FALSE ;
}
VOID i7410 (VOID)			/* SKP				*/
{
    ++PC ;
}
VOID i7412 (VOID)			/* SKP HLT			*/
{
    ++PC ;
    run = FALSE ;
}
VOID u7412 (VOID)			/* SKP HLT (user mode)		*/
{
    ++PC ;
    io_flags |= userFLAG ;
}
VOID i7414 (VOID)			/* SKP OSR			*/
{
    ++PC ;
    AC |= SR ;
}
VOID i7416 (VOID)			/* SKP OSR HLT			*/
{
    ++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7420 (VOID)			/* SNL				*/
{
    if (AC & 010000)
	++PC ;
}
VOID i7422 (VOID)			/* SNL HLT			*/
{
    if (AC & 010000)
	++PC ;
    run = FALSE ;
}
VOID u7422 (VOID)			/* SNL HLT (user mode)		*/
{
    if (AC & 010000)
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7424 (VOID)			/* SNL OSR			*/
{
    if (AC & 010000)
	++PC ;
    AC |= SR ;
}
VOID i7426 (VOID)			/* SNL OSR HLT			*/
{
    if (AC & 010000)
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7430 (VOID)			/* SZL				*/
{
    if (!(AC & 010000))
	++PC ;
}
VOID i7432 (VOID)			/* SZL HLT			*/
{
    if (!(AC & 010000))
	++PC ;
    run = FALSE ;
}
VOID u7432 (VOID)			/* SZL HLT (user mode)		*/
{
    if (!(AC & 010000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7434 (VOID)			/* SZL OSR			*/
{
    if (!(AC & 010000))
	++PC ;
    AC |= SR ;
}
VOID i7436 (VOID)			/* SZL OSR HLT			*/
{
    if (!(AC & 010000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7440 (VOID)			/* SZA				*/
{
    if (!(AC & 07777))
	++PC ;
}
VOID i7442 (VOID)			/* SZA HLT			*/
{
    if (!(AC & 07777))
	++PC ;
    run = FALSE ;
}
VOID u7442 (VOID)			/* SZA HLT (user mode)		*/
{
    if (!(AC & 07777))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7444 (VOID)			/* SZA OSR			*/
{
    if (!(AC & 07777))
	++PC ;
    AC |= SR ;
}
VOID i7446 (VOID)			/* SZA OSR HLT			*/
{
    if (!(AC & 07777))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7450 (VOID)			/* SNA				*/
{
    if (AC & 07777)
	++PC ;
}
VOID i7452 (VOID)			/* SNA HLT			*/
{
    if (AC & 07777)
	++PC ;
    run = FALSE ;
}
VOID u7452 (VOID)			/* SNA HLT (user mode)		*/
{
    if (AC & 07777)
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7454 (VOID)			/* SNA OSR			*/
{
    if (AC & 07777)
	++PC ;
    AC |= SR ;
}
VOID i7456 (VOID)			/* SNA OSR HLT			*/
{
    if (AC & 07777)
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7460 (VOID)			/* SZA SNL			*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
}
VOID i7462 (VOID)			/* SZA SNL HLT			*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    run = FALSE ;
}
VOID u7462 (VOID)			/* SZA SNL HLT (user mode)	*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7464 (VOID)			/* SZA SNL OSR			*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC |= SR ;
}
VOID i7466 (VOID)			/* SZA SNL OSR HLT		*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7470 (VOID)			/* SNA SZL			*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
}
VOID i7472 (VOID)			/* SNA SZL HLT			*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    run = FALSE ;
}
VOID u7472 (VOID)			/* SNA SZL HLT (user mode)	*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7474 (VOID)			/* SNA SZL OSR			*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC |= SR ;
}
VOID i7476 (VOID)			/* SNA SZL OSR HLT		*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7500 (VOID)			/* SMA				*/
{
    if (AC & 04000)
	++PC ;
}
VOID i7502 (VOID)			/* SMA HLT			*/
{
    if (AC & 04000)
	++PC ;
    run = FALSE ;
}
VOID u7502 (VOID)			/* SMA HLT (user mode)		*/
{
    if (AC & 04000)
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7504 (VOID)			/* SMA OSR			*/
{
    if (AC & 04000)
	++PC ;
    AC |= SR ;
}
VOID i7506 (VOID)			/* SMA OSR HLT			*/
{
    if (AC & 04000)
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7510 (VOID)			/* SPA				*/
{
    if (!(AC & 04000))
	++PC ;
}
VOID i7512 (VOID)			/* SPA HLT			*/
{
    if (!(AC & 04000))
	++PC ;
    run = FALSE ;
}
VOID u7512 (VOID)			/* SPA HLT (user mode)		*/
{
    if (!(AC & 04000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7514 (VOID)			/* SPA OSR			*/
{
    if (!(AC & 04000))
	++PC ;
    AC |= SR ;
}
VOID i7516 (VOID)			/* SPA OSR HLT			*/
{
    if (!(AC & 04000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7520 (VOID)			/* SMA SNL			*/
{
    if (AC & 014000)
	++PC ;
}
VOID i7522 (VOID)			/* SMA SNL HLT			*/
{
    if (AC & 014000)
	++PC ;
    run = FALSE ;
}
VOID u7522 (VOID)			/* SMA SNL HLT (user mode)	*/
{
    if (AC & 014000)
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7524 (VOID)			/* SMA SNL OSR			*/
{
    if (AC & 014000)
	++PC ;
    AC |= SR ;
}
VOID i7526 (VOID)			/* SMA SNL OSR HLT		*/
{
    if (AC & 014000)
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7530 (VOID)			/* SPA SZL			*/
{
    if (!(AC & 014000))
	++PC ;
}
VOID i7532 (VOID)			/* SPA SZL HLT			*/
{
    if (!(AC & 014000))
	++PC ;
    run = FALSE ;
}
VOID u7532 (VOID)			/* SPA SZL HLT (user mode)	*/
{
    if (!(AC & 014000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7534 (VOID)			/* SPA SZL OSR			*/
{
    if (!(AC & 014000))
	++PC ;
    AC |= SR ;
}
VOID i7536 (VOID)			/* SPA SZL OSR HLT		*/
{
    if (!(AC & 014000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7540 (VOID)			/* SMA SZA			*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
}
VOID i7542 (VOID)			/* SMA SZA HLT			*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    run = FALSE ;
}
VOID u7542 (VOID)			/* SMA SZA HLT (user mode)	*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7544 (VOID)			/* SMA SZA OSR			*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC |= SR ;
}
VOID i7546 (VOID)			/* SMA SZA OSR HLT		*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7550 (VOID)			/* SPA SNA			*/
{
    if (!(AC & 04000) && (AC & 03777))
	++PC ;
}
VOID i7552 (VOID)			/* SPA SNA HLT			*/
{
    if ((AC & 04000) == 0 && (AC & 03777))
	++PC ;
    run = FALSE ;
}
VOID u7552 (VOID)			/* SPA SNA HLT (user mode)	*/
{
    if ((AC & 04000) == 0 && (AC & 03777))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7554 (VOID)			/* SPA SNA OSR			*/
{
    if ((AC & 04000) == 0 && (AC & 03777))
	++PC ;
    AC |= SR ;
}
VOID i7556 (VOID)			/* SPA SNA OSR HLT		*/
{
    if ((AC & 04000) == 0 && (AC & 03777))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7560 (VOID)			/* SMA SZA SNL			*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
}
VOID i7562 (VOID)			/* SMA SZA SNL HLT		*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    run = FALSE ;
}
VOID u7562 (VOID)			/* SMA SZA SNL HLT (user mode)	*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7564 (VOID)			/* SMA SZA SNL OSR		*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC |= SR ;
}
VOID i7566 (VOID)			/* SMA SZA SNL OSR HLT		*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7570 (VOID)			/* SPA SNA SZL			*/
{
    if (!(AC & 04000) && (AC & 03777) && !(AC & 010000))
	++PC ;
}
VOID i7572 (VOID)			/* SPA SNA SZL HLT		*/
{
    if (!(AC & 04000) && (AC & 03777) && !(AC & 010000))
	++PC ;
    run = FALSE ;
}
VOID u7572 (VOID)			/* SPA SNA SZL HLT (user mode)	*/
{
    if (!(AC & 04000) && (AC & 03777) && !(AC & 010000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7574 (VOID)			/* SPA SNA SZL OSR		*/
{
    if (!(AC & 04000) && (AC & 03777) && !(AC & 010000))
	++PC ;
    AC |= SR ;
}
VOID i7576 (VOID)			/* SPA SNA SZL OSR HLT		*/
{
    if (!(AC & 04000) && (AC & 03777) && !(AC & 010000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7600 (VOID)			/* CLA Grp II			*/
{
    AC &= 010000 ;
}
VOID i7602 (VOID)			/* CLA HLT			*/
{
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7602 (VOID)			/* CLA HLT (user mode)		*/
{
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7604 (VOID)			/* CLA OSR			*/
{
    AC = (AC & 010000) | SR ;
}
VOID i7606 (VOID)			/* CLA OSR HLT			*/
{
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7610 (VOID)			/* SKP CLA			*/
{
    AC &= 010000 ;
    ++PC ;
}
VOID i7612 (VOID)			/* SKP CLA HLT			*/
{
    AC &= 010000 ;
    ++PC ;
    run = FALSE ;
}
VOID u7612 (VOID)			/* SKP CLA HLT (user mode)	*/
{
    AC &= 010000 ;
    ++PC ;
    io_flags |= userFLAG ;
}
VOID i7614 (VOID)			/* SKP CLA OSR			*/
{
    AC &= 010000 ;
    ++PC ;
    AC |= SR ;
}
VOID i7616 (VOID)			/* SKP CLA OSR HLT		*/
{
    AC &= 010000 ;
    ++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7620 (VOID)			/* SNL CLA			*/
{
    if (AC &= 010000)
	++PC ;
}
VOID i7622 (VOID)			/* SNL CLA HLT			*/
{
    if (AC &= 010000)
	++PC ;
    run = FALSE ;
}
VOID u7622 (VOID)			/* SNL CLA HLT (user mode)	*/
{
    if (AC &= 010000)
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7624 (VOID)			/* SNL CLA OSR			*/
{
    if (AC &= 010000)
	++PC ;
    AC |= SR ;
}
VOID i7626 (VOID)			/* SNL CLA OSR HLT		*/
{
    if (AC &= 010000)
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7630 (VOID)			/* SZL CLA			*/
{
    if (!(AC &= 010000))
	++PC ;
}
VOID i7632 (VOID)			/* SZL CLA HLT			*/
{
    if (!(AC &= 010000))
	++PC ;
    run = FALSE ;
}
VOID u7632 (VOID)			/* SZL CLA HLT (user mode)	*/
{
    if (!(AC &= 010000))
	++PC ;
    io_flags |= userFLAG ;
}
VOID i7634 (VOID)			/* SZL CLA OSR			*/
{
    if (!(AC &= 010000))
	++PC ;
    AC |= SR ;
}
VOID i7636 (VOID)			/* SZL CLA OSR HLT		*/
{
    if (!(AC &= 010000))
	++PC ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7640 (VOID)			/* SZA CLA			*/
{
    if (!(AC & 07777))
	++PC ;
    AC &= 010000 ;
}
VOID i7642 (VOID)			/* SZA CLA HLT			*/
{
    if (!(AC & 07777))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7642 (VOID)			/* SZA CLA HLT (user mode)	*/
{
    if (!(AC & 07777))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7644 (VOID)			/* SZA CLA OSR			*/
{
    if (!(AC & 07777))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7646 (VOID)			/* SZA CLA OSR HLT		*/
{
    if (!(AC & 07777))
	++PC ;
    AC &= 010000 ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7650 (VOID)			/* SNA CLA			*/
{
    if (AC & 07777)
	++PC ;
    AC &= 010000 ;
}
VOID i7652 (VOID)			/* SNA CLA HLT			*/
{
    if (AC & 07777)
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7652 (VOID)			/* SNA CLA HLT (user mode)	*/
{
    if (AC & 07777)
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7654 (VOID)			/* SNA CLA OSR			*/
{
    if (AC & 07777)
	++PC ;
    AC &= 010000 ;
    AC |= SR ;
}
VOID i7656 (VOID)			/* SNA CLA OSR HLT		*/
{
    if (AC & 07777)
	++PC ;
    AC &= 010000 ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7660 (VOID)			/* SZA SNL CLA			*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC &= 010000 ;
}
VOID i7662 (VOID)			/* SZA SNL CLA HLT		*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7662 (VOID)			/* SZA SNL CLA HLT (user mode)	*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7664 (VOID)			/* SZA SNL CLA OSR		*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC &= 010000 ;
    AC |= SR ;
}
VOID i7666 (VOID)			/* SZA SNL CLA OSR HLT		*/
{
    if ((AC & 07777) == 0 || (AC & 010000))
	++PC ;
    AC &= 010000 ;
    AC |= SR ;
    run = FALSE ;
}
VOID i7670 (VOID)			/* SNA SZL CLA			*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC &= 010000 ;
}
VOID i7672 (VOID)			/* SNA SZL CLA HLT		*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7672 (VOID)			/* SNA SZL CLA HLT (user mode)	*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7674 (VOID)			/* SNA SZL CLA OSR		*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7676 (VOID)			/* SNA SZL CLA OSR HLT		*/
{
    if ((AC & 07777) && !(AC & 010000))
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7700 (VOID)			/* SMA CLA			*/
{
    if (AC & 04000)
	++PC ;
    AC &= 010000 ;
}
VOID i7702 (VOID)			/* SMA CLA HLT			*/
{
    if (AC & 04000)
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7702 (VOID)			/* SMA CLA HLT			*/
{
    if (AC & 04000)
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7704 (VOID)			/* SMA CLA OSR			*/
{
    if (AC & 04000)
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7706 (VOID)			/* SMA CLA OSR HLT		*/
{
    if (AC & 04000)
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7710 (VOID)			/* SPA CLA			*/
{
    if (!(AC & 04000))
	++PC ;
    AC &= 010000 ;
}
VOID i7712 (VOID)			/* SPA CLA HLT			*/
{
    if (!(AC & 04000))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7712 (VOID)			/* SPA CLA HLT (user mode)	*/
{
    if (!(AC & 04000))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7714 (VOID)			/* SPA CLA OSR			*/
{
    if (!(AC & 04000))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7716 (VOID)			/* SPA CLA OSR HLT		*/
{
    if (!(AC & 04000))
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7720 (VOID)			/* SMA SNL CLA			*/
{
    if (AC & 014000)
	++PC ;
    AC &= 010000 ;
}
VOID i7722 (VOID)			/* SMA SNL CLA HLT		*/
{
    if (AC & 014000)
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7722 (VOID)			/* SMA SNL CLA HLT (user mode)	*/
{
    if (AC & 014000)
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7724 (VOID)			/* SMA SNL CLA OSR		*/
{
    if (AC & 014000)
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7726 (VOID)			/* SMA SNL CLA OSR HLT		*/
{
    if (AC & 014000)
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7730 (VOID)			/* SPA SZL CLA			*/
{
    if (!(AC & 014000))
	++PC ;
    AC &= 010000 ;
}
VOID i7732 (VOID)			/* SPA SZL CLA HLT		*/
{
    if (!(AC & 014000))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7732 (VOID)			/* SPA SZL CLA HLT (user mode)	*/
{
    if (!(AC & 014000))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7734 (VOID)			/* SPA SZL CLA OSR		*/
{
    if (!(AC & 014000))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7736 (VOID)			/* SPA SZL CLA OSR HLT		*/
{
    if (!(AC & 014000))
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7740 (VOID)			/* SMA SZA CLA			*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC &= 010000 ;
}
VOID i7742 (VOID)			/* SMA SZA CLA HLT		*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7742 (VOID)			/* SMA SZA CLA HLT (user mode)	*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7744 (VOID)			/* SMA SZA CLA OSR		*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7746 (VOID)			/* SMA SZA CLA OSR HLT		*/
{
    if ((AC & 04000) || !(AC & 07777))
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7750 (VOID)			/* SPA SNA CLA			*/
{
    if ((AC & 07777) && (AC & 07777) < 04000)
	++PC ;
    AC &= 010000 ;
}
VOID i7752 (VOID)			/* SPA SNA CLA HLT		*/
{
    if ((AC & 07777) && (AC & 07777) < 04000)
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7752 (VOID)			/* SPA SNA CLA HLT (user mode)	*/
{
    if ((AC & 07777) && (AC & 07777) < 04000)
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7754 (VOID)			/* SPA SNA CLA OSR		*/
{
    if ((AC & 07777) && (AC & 07777) < 04000)
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7756 (VOID)			/* SPA SNA CLA OSR HLT		*/
{
    if ((AC & 07777) && (AC & 07777) < 04000)
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7760 (VOID)			/* SMA SZA SNL CLA		*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC &= 010000 ;
}
VOID i7762 (VOID)			/* SMA SZA SNL CLA HLT		*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7762 (VOID)			/* SMA SZA SNL CLA HLT		*/
{					/*		(user mode)	*/
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7764 (VOID)			/* SMA SZA SNL CLA OSR		*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7766 (VOID)			/* SMA SZA SNL CLA OSR HLT	*/
{
    if ((AC & 07777) == 0 || (AC & 014000))
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
VOID i7770 (VOID)			/* SPA SNA SZL CLA		*/
{
    if ((AC & 07777) && !(AC & 014000))
	++PC ;
    AC &= 010000 ;
}
VOID i7772 (VOID)			/* SPA SNA SZL CLA HLT		*/
{
    if ((AC & 07777) && !(AC & 014000))
	++PC ;
    AC &= 010000 ;
    run = FALSE ;
}
VOID u7772 (VOID)			/* SPA SNA SZL CLA HLT		*/
{					/*		(user mode)	*/
    if ((AC & 07777) && !(AC & 014000))
	++PC ;
    AC &= 010000 ;
    io_flags |= userFLAG ;
}
VOID i7774 (VOID)			/* SPA SNA SZL CLA OSR		*/
{
    if ((AC & 07777) && !(AC & 014000))
	++PC ;
    AC = (AC & 010000) | SR ;
}
VOID i7776 (VOID)			/* SPA SNA SZL CLA OSR HLT	*/
{
    if ((AC & 07777) && !(AC & 014000))
	++PC ;
    AC = (AC & 010000) | SR ;
    run = FALSE ;
}
/* -------------------------------------------------------------------- */
