package de.beagtex.gui;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.FlowLayout;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Properties;
import java.util.StringTokenizer;

import javax.swing.Action;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.text.TextAction;

import de.beagtex.control.FileWorkMethods;
import de.beagtex.control.RedoAction;
import de.beagtex.control.UndoAction;
import de.beagtex.listener.gui.MyWindowListener;

/**
 * Standard-Frame for BeaGTex; all other elements should be simply be integrated
 * in this frame. Every sort of communication between the single elements has to
 * be via BFrame.
 * 
 * @author Benjamin Haag
 * 
 */
public class BFrame extends BWindow {

	/**
	 * serialVersionUID
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Editor-Component
	 */
	Editor edit;

	/**
	 * Button-Bar
	 */
	ButtonBar blButtons;

	/**
	 * Format-Bar
	 */
	FormatBar formate;

	/**
	 * Hashtable for Undo-/Redo
	 */
	Hashtable commands = null;

	/**
	 * Undo-Action
	 */
	public UndoAction undoAction = new UndoAction(edit);

	/**
	 * Redo-Action
	 */
	public RedoAction redoAction = new RedoAction(edit);

	/**
	 * Definition of Undo-/Redo-Actions
	 */
	private Action[] defaultActions = { undoAction, redoAction };

	/**
	 * Language for BeaGTex
	 */
	private static String language = "";

	/**
	 * Default-Properties-File (english texts)
	 */
	private static final String BEAGTEX_DEFAULT_PROP = "data/text_engl.properties";

	/**
	 * Properties for other languages
	 */
	private static final String BEAGTEX_GER_PROP = "data/text_ger.properties";

	/**
	 * Properties for BeaGTex
	 */
	private static Properties _prop = new Properties();

	/**
	 * Constructor; here the layout is defined and finally the other components
	 * (editor, menu ...) are integrated
	 * 
	 */
	public BFrame() {

		setTitle("BeaGTex");
		setSize(850, 625);
		addWindowListener(new MyWindowListener(this));
		try {
			UIManager.setLookAndFeel("javax.swing.plaf.metal.MetalLookAndFeel");
			// UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
			// UIManager.setLookAndFeel("com.sun.java.swing.plaf.windows.WindowsLookAndFeel");
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (UnsupportedLookAndFeelException e) {
			e.printStackTrace();
		}

		// get properties
		setLanguage(this.readLanguageFile());
		String lang = getLanguage();
		if (lang.equalsIgnoreCase("first")) {
			this.loadProperties(BEAGTEX_GER_PROP);
		} else {
			if (lang.equalsIgnoreCase("German")) {
				this.loadProperties(BEAGTEX_GER_PROP);
			} else {
				this.loadProperties(BEAGTEX_DEFAULT_PROP);
			}
		}

		/* Editor-Field */
		edit = new Editor(this);

		// Filling of the Action-Array for Undo/Redo-Actions
		Hashtable commands = new Hashtable();
		Action[] actions = edit.getActions();
		for (int i = 0; i < actions.length; i++) {
			Action a = actions[i];
			commands.put(a.getValue(Action.NAME), a);
		}

		// necessary for Scrollbars
		JScrollPane pEdit = new JScrollPane(edit);
		setContentPane(pEdit);

		// Menu-Bar
		JPanel pan = new JPanel();
		pan.setLayout(new FlowLayout(-1));
		pan.add(new MainMenu(this));
		setContentPane(pan);

		blButtons = new ButtonBar(this, 1); // 1 steht fr vertikal
		setContentPane(blButtons);

		formate = new FormatBar(this, 0);
		setContentPane(formate);

		/* Toolbar for distance to left border */
		JToolBar tbLinks = new JToolBar(1);
		setContentPane(tbLinks);

		Container conty = new Container();
		conty.setLayout(new BorderLayout());
		conty.add(pEdit, BorderLayout.CENTER);
		conty.add(pan, BorderLayout.NORTH);
		conty.add(blButtons, BorderLayout.EAST);
		conty.add(formate, BorderLayout.SOUTH);
		conty.add(tbLinks, BorderLayout.WEST);
		setContentPane(conty);
	}

	/**
	 * Main; for testing only
	 * 
	 * @param args
	 */
	public static void main(String[] args) {

		BFrame test = new BFrame();
		test.setVisible(true);
	}

	/**
	 * Starts a BFrame
	 */
	public static void start(BFrame pStartFrame) {

		StartChoiceWindow starty = new StartChoiceWindow(pStartFrame);
		starty.show(400, 150);
	}

	/**
	 * Starts the langChoice-dialog if needed and loads the properties
	 */
	public static void startLangChoice() {
		BFrame startFrame = new BFrame();
		setLanguage(startFrame.readLanguageFile());
		String lang = getLanguage();
		if (lang.equalsIgnoreCase("first")) {
			LanguageChoiceWindow lcw = new LanguageChoiceWindow(startFrame);
			lcw.show(200, 100);
		} else {
			if (lang.equalsIgnoreCase("German")) {
				startFrame.loadProperties(BEAGTEX_GER_PROP);
				_prop.toString();
				start(startFrame);
			} else {
				startFrame.loadProperties(BEAGTEX_DEFAULT_PROP);
				_prop.toString();
				start(startFrame);
			}
		}
	}

	/**
	 * Reads the language-file (from data/lang.txt)
	 * 
	 * @return String with language
	 */
	private String readLanguageFile() {
		String lang = "";
		try {
			StringBuffer buffy = FileWorkMethods
					.createStringBufferFromFile("data/lang.txt");
			lang = buffy.toString();
			StringTokenizer toky = new StringTokenizer(lang);
			lang = toky.nextToken();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return lang;
	}

	/**
	 * Loads properties for BeaGTex
	 * 
	 * @param pProperties :
	 *            String with properties-file
	 */
	private void loadProperties(String pProperties) {

		try {
			_prop.clear();
			if (pProperties.equals("")) {
				_prop.load(ClassLoader
						.getSystemResourceAsStream(BEAGTEX_DEFAULT_PROP));
			} else {
				_prop.load(new FileInputStream(new File(pProperties)));
			}

		} catch (FileNotFoundException e1) {
			e1.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	// ab hier Getter/Setter
	/**
	 * Loading the Action-Argument-List for Undo-Manager
	 */
	public Action[] getActions() {
		return TextAction.augmentList(edit.getActions(), defaultActions);
	}

	public Editor getEdit() {
		return edit;
	}

	public ButtonBar getBlButtons() {
		return blButtons;
	}

	public FormatBar getFormate() {
		return formate;
	}

	public static String getLanguage() {
		return language;
	}

	public static void setLanguage(String pLanguage) {
		language = pLanguage;
	}

	public static Properties get_prop() {
		return _prop;
	}

}
