/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import static jnpad.util.Utilities.COMMA_STRING;
import static jnpad.util.Utilities.EMPTY_STRING;

import java.io.Serializable;

import jnpad.util.Platform;
import jnpad.util.Utilities;

/**
 * The Class JNPadInput.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class JNPadInput implements Serializable {
  private String            _filePath;
  private boolean           _isReadOnly;
  private int               _caretPosition;
  private boolean           _isCurrent;
  private String            _splitConfig;
  private boolean           _isLineWrapped; // [v0.3]

  /** UID */
  private static final long serialVersionUID = 5444128936316971023L;
  
  /**
   * Instantiates a new jNPad input.
   *
   * @param filePath the file path
   * @param caretPosition the caret position
   * @param isReadOnly is read only
   * @param isCurrent is current
   * @param isLineWrapped is line wrapped
   */
  public JNPadInput(String filePath, int caretPosition, boolean isReadOnly, boolean isCurrent, boolean isLineWrapped) {
    this(filePath, caretPosition, isReadOnly, isCurrent, EMPTY_STRING, isLineWrapped);
  }

  /**
   * Instantiates a new jNPad input.
   *
   * @param filePath the file path
   * @param caretPosition the caret position
   * @param isReadOnly is read only
   * @param isCurrent is current
   * @param splitConfig the split config
   * @param isLineWrapped is line wrapped
   */
  public JNPadInput(String filePath, int caretPosition, boolean isReadOnly, boolean isCurrent, String splitConfig, boolean isLineWrapped) {
    _filePath      = filePath;
    _caretPosition = caretPosition;
    _isReadOnly    = isReadOnly;
    _isCurrent     = isCurrent;
    _splitConfig   = splitConfig == null ? EMPTY_STRING : splitConfig;
    _isLineWrapped = isLineWrapped;
  }

  /**
   * Checks if is read only.
   *
   * @return true, if is read only
   */
  public boolean isReadOnly() {
    return _isReadOnly;
  }

  /**
   * Checks if is current.
   *
   * @return true, if is current
   */
  public boolean isCurrent() {
    return _isCurrent;
  }
  
  /**
   * Gets the path.
   *
   * @return the path
   */
  public String getPath() {
    return _filePath;
  }

  /**
   * Gets the caret position.
   *
   * @return the caret position
   */
  public int getCaretPosition() {
    return _caretPosition;
  }
  
  /**
   * Gets the split config.
   *
   * @return the split config
   */
  public String getSplitConfig() {
    return _splitConfig;
  }

  /**
   * Checks if is line wrapped.
   *
   * @return true, if is line wrapped
   * @since 0.3
   */
  public boolean isLineWrapped() {
    return _isLineWrapped;
  }
  
  /**
   * Equals.
   *
   * @param obj the obj
   * @return true, if successful
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    return (obj == this) ||
        ((obj instanceof JNPadInput) &&
            Utilities.equals(_filePath, ((JNPadInput) obj)._filePath, !Platform.isWindows) &&
            (_caretPosition == ((JNPadInput) obj)._caretPosition) &&
            (_isReadOnly    == ((JNPadInput) obj)._isReadOnly)    &&
            (_isCurrent     == ((JNPadInput) obj)._isCurrent)     &&
            (_isLineWrapped == ((JNPadInput) obj)._isLineWrapped) &&
            Utilities.equals(_splitConfig, ((JNPadInput) obj)._splitConfig));
  }

  /**
   * Hash code.
   *
   * @return the int
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    return Utilities.hashCode(_filePath,
        _caretPosition, _isReadOnly, _isCurrent, _splitConfig, _isLineWrapped);
  }

  /**
   * To string.
   *
   * @return the string
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder();
    
    sb.append(_filePath).append(COMMA_STRING).
        append(_caretPosition).append(COMMA_STRING).
        append(_isReadOnly).append(COMMA_STRING).
        append(_isCurrent).append(COMMA_STRING).
        append(_isLineWrapped);
    
    if(Utilities.isNotEmptyString(_splitConfig)) {
      sb.append(COMMA_STRING).append(_splitConfig);
    }
    
    return sb.toString();
  }

}
