/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import static jnpad.action.JNPadActions.ACTION_NAME_CHECK_FOR_UPDATES;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JOptionPane;

import jnpad.GUIUtilities;
import jnpad.JNPadBundle;
import jnpad.JNPadFrame;
import jnpad.action.JNPadActions.Group;
import jnpad.config.Accelerators;
import jnpad.ui.ReporterUtilities;
import jnpad.ui.status.StatusDisplayable;
import jnpad.ui.status.StatusDisplayable.StatusType;
import jnpad.util.Version;

/**
 * The Class Check4UpdatesAction.
 * 
 * @version 0.3
 * @since jNPad v0.3
 */
public final class Check4UpdatesAction extends JNPadAction {
  /** Logger */
  private static final Logger LOGGER           = Logger.getLogger(Check4UpdatesAction.class.getName());

  /** UID */
  private static final long   serialVersionUID = 6150350042765974110L;

  /**
   * Instantiates a new check for updates action.
   * 
   * @param jNPad the jNPpad's frame
   */
  public Check4UpdatesAction(JNPadFrame jNPad) {
    super(jNPad,
          ACTION_NAME_CHECK_FOR_UPDATES,
          Group.HELP,
          null, "check4updates.png"); //$NON-NLS-1$

    if (Accelerators.isUsingCompositeShortcuts()) {
      setKeyBinding(Accelerators.getShortcutLabel(ACTION_NAME_CHECK_FOR_UPDATES));
    }
  }

  /**
   * Perform action.
   * 
   * @see jnpad.action.JNPadAction#performAction()
   */
  @Override
  public void performAction() {
    URLConnection connection = null;
    BufferedReader in = null;
    try {
      URL url = new URL("http://rgs.freeiz.com/jnpad/version.html"); //$NON-NLS-1$
      connection = url.openConnection();
      connection.connect();
      in = new BufferedReader(new InputStreamReader(connection.getInputStream()));
      
      String lastVersion;
      if ((lastVersion = in.readLine()) != null) {
        jNPad.setStatus(StatusType.INFORMATION, ActionBundle.getString("check-for-updates.latest", lastVersion), StatusDisplayable.TIMEOUT_DEFAULT); //$NON-NLS-1$
      }
      if(lastVersion == null) {
        ReporterUtilities.reportError(jNPad, ActionBundle.getString("check-for-updates.error")); //$NON-NLS-1$
        return;
      }
      
      final String version = Version.getVersion().trim();
      
      if (version.equals(lastVersion)) {
        ReporterUtilities.reportInfo(jNPad, 
                                     ActionBundle.getString("check-for-updates.noUpdates.message", version), //$NON-NLS-1$
                                     ActionBundle.getString("check-for-updates.noUpdates.title")); //$NON-NLS-1$
      }
      else {
        String[] options = { JNPadBundle.getYesOptionText(), JNPadBundle.getNoOptionText() };
        int option = JOptionPane.showOptionDialog(jNPad,
                                                  ActionBundle.getString("check-for-updates.updates.message", lastVersion), //$NON-NLS-1$
                                                  ActionBundle.getString("check-for-updates.updates.title"), //$NON-NLS-1$
                                                  JOptionPane.DEFAULT_OPTION,
                                                  JOptionPane.QUESTION_MESSAGE,
                                                  null,
                                                  options,
                                                  options[0]);
        
        if (option == 1 || option == JOptionPane.CLOSED_OPTION)
          return;
        
        GUIUtilities.openBrowser("http://sourceforge.net/projects/jnpad"); //$NON-NLS-1$
      }
    }
    catch (Exception ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
      ReporterUtilities.reportError(jNPad, ActionBundle.getString("check-for-updates.error"), null, ex); //$NON-NLS-1$
    }
    finally {
      try {
        if (in != null) {
          in.close();
        }
      }
      catch (IOException ex) {
        LOGGER.log(Level.WARNING, ex.getMessage(), ex);
      }
    }
  }

}
