/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.print;

import java.awt.Font;
import java.awt.print.PageFormat;

/**
 * Stores the print options for printing using the PrintText class.
 * 
 * @version 0.3
 * @since jNPad 0.1
 */
public class PrintingOptions {
  
  /** The _print line numbers. */
  private boolean    _printLineNumbers = false;
  
  /** The _wrap text. */
  private boolean    _wrapText         = false;
  
  /** The _print header. */
  private boolean    _printHeader      = false;
  
  /** The _print syntax. */
  private boolean    _printSyntax      = false;
  
  /** The _page font. */
  private Font       _pageFont         = new Font("Courier", Font.PLAIN, 10); // the page font (default) //$NON-NLS-1$
  
  /** The _header font. */
  private Font       _headerFont       = new Font("Courier", Font.BOLD, 10); // the page font (default) //$NON-NLS-1$
  
  /** The _page format. */
  private PageFormat _pageFormat       = null;

  /**
   * Instantiates a new printing options.
   */
  public PrintingOptions() {
    _pageFormat = new PageFormat();
  }

  /**
   * Instantiates a new printing options.
   *
   * @param printLineNumbers boolean
   * @param wrapText boolean
   * @param printHeader boolean
   * @param printSyntax boolean
   * @param pageFont Font
   * @param pageFormat PageFormat
   */
  public PrintingOptions(boolean printLineNumbers, boolean wrapText, boolean printHeader, boolean printSyntax,
                         Font pageFont, PageFormat pageFormat) {
    _printLineNumbers = printLineNumbers;
    _wrapText         = wrapText;
    _printHeader      = printHeader;
    _printSyntax      = printSyntax;
    _pageFont         = pageFont;
    _pageFormat       = pageFormat;

    // Build the header font;
    _headerFont = new Font(_pageFont.getName(), Font.BOLD, _pageFont.getSize());
  }

  /**
   * Sets the prints the line numbers.
   *
   * @param printLineNumbers boolean
   */
  public void setPrintLineNumbers(boolean printLineNumbers) {
    _printLineNumbers = printLineNumbers;
  }

  /**
   * Gets the prints the line numbers.
   *
   * @return the prints the line numbers
   */
  public boolean getPrintLineNumbers() {
    return _printLineNumbers;
  }

  /**
   * Sets the wrap text.
   *
   * @param wrapText the new wrap text
   */
  public void setWrapText(boolean wrapText) {
    _wrapText = wrapText;
  }

  /**
   * Gets the wrap text.
   *
   * @return the wrap text
   */
  public boolean getWrapText() {
    return _wrapText;
  }

  /**
   * Sets the prints the header.
   *
   * @param printHeader the new prints the header
   */
  public void setPrintHeader(boolean printHeader) {
    _printHeader = printHeader;
  }

  /**
   * Gets the prints the header.
   *
   * @return the prints the header
   */
  public boolean getPrintHeader() {
    return _printHeader;
  }

  /**
   * Sets the prints the syntax.
   *
   * @param printSyntax the new prints the syntax
   */
  public void setPrintSyntax(boolean printSyntax) {
    _printSyntax = printSyntax;
  }

  /**
   * Gets the prints the syntax.
   *
   * @return the prints the syntax
   */
  public boolean getPrintSyntax() {
    return _printSyntax;
  }

  /**
   * Sets the page font.
   *
   * @param pageFont the new page font
   */
  public void setPageFont(Font pageFont) {
    _pageFont = pageFont;
    // Build the header font;
    _headerFont = new Font(_pageFont.getName(), Font.BOLD, _pageFont.getSize());
  }

  /**
   * Gets the page font.
   *
   * @return the page font
   */
  public Font getPageFont() {
    return _pageFont;
  }

  /**
   * Gets the header font.
   *
   * @return the header font
   */
  public Font getHeaderFont() {
    return _headerFont;
  }

  /**
   * Sets the page format.
   *
   * @param pageFormat the new page format
   */
  public void setPageFormat(PageFormat pageFormat) {
    _pageFormat = pageFormat;
  }

  /**
   * Gets the page format.
   *
   * @return the page format
   */
  public PageFormat getPageFormat() {
    return _pageFormat;
  }
}
