/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.syntax;

import java.awt.Font;
import java.awt.Graphics;

import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.Segment;
import javax.swing.text.Utilities;

import jnpad.text.JNPadDocument;
import jnpad.text.syntax.ASN1Scheme.WordType;

/**
 * The Class ASN1View.
 * 
 * @version 0.3
 * @since jNPad 0.1
 */
public class ASN1View extends SyntaxView {
  ASN1Scheme scheme;

  /**
   * Instantiates a new aS n1 view.
   *
   * @param elem the elem
   * @param mini the mini
   */
  public ASN1View(Element elem, boolean mini) {
    super(elem, mini);
    scheme = !mini ? (ASN1Scheme) ((JNPadDocument) getDocument()).getScheme() :
        (ASN1Scheme) ((JNPadDocument) getDocument()).getMiniScheme();
  }

  /**
   * Do draw unselected text.
   * 
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawUnselectedText(java.awt.Graphics,
   *      int, int, int, int)
   */
  @Override
  int doDrawUnselectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (isASN1IdentifierStart(indexedChar)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex);
        int scannedIdentifierLength = scannedIdentifier.length();

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType(scannedIdentifier);
        g.setColor(scheme.getWordColor(type));
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (Character.isDigit(indexedChar)) {
        String scannedNumericLiteral = scanNumericLiteral(content, wordIndex);
        int scannedNumericLiteralLength = scannedNumericLiteral.length();

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedNumericLiteralLength, text);
        g.setColor(scheme.getNumberColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        
        wordIndex += scannedNumericLiteralLength;

        continue;
      }
      else if (isStringDelimiter(indexedChar)) {
        int scannedStringLength = scanStringLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedStringLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getStringColor());
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedStringLength;

        continue;
      }
      else if (isCharDelimiter(indexedChar)) {
        int scannedCharLength = scanCharLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCharLength, text);
        g.setColor(scheme.getStringColor());
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        
        wordIndex += scannedCharLength;

        continue;
      }
      else if (isSingleLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getCommentColor());
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;

        continue;
      }
      else if (isBracket(indexedChar)) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getBracketFont());
        g.setColor(scheme.getBracketColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex++;

        continue;
      }
      else if (isOperator(indexedChar)) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        g.setColor(scheme.getOperatorColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex++;

        continue;
      }

      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      g.setColor(scheme.getTextColor());
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      
      wordIndex++;
    }

    return x;
  }

  /**
   * Do draw selected text.
   * 
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawSelectedText(java.awt.Graphics,
   *      int, int, int, int)
   */
  @Override
  int doDrawSelectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    g.setColor(scheme.getSelectedTextColor());

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (isASN1IdentifierStart(indexedChar)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex);
        int scannedIdentifierLength = scannedIdentifier.length();

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType(scannedIdentifier);
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
      
        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (isStringDelimiter(indexedChar)) {
        int scannedStringLength = scanStringLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedStringLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedStringLength;

        continue;
      }
      else if (isCharDelimiter(indexedChar)) {
        int scannedCharLength = scanCharLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCharLength, text);
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        
        wordIndex += scannedCharLength;

        continue;
      }
      else if (isSingleLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;

        continue;
      }
      else if (isBracket(indexedChar)) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getBracketFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;

        continue;
      }
      else if (isOperator(indexedChar)) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex++;

        continue;
      }

      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);

      wordIndex++;
    }

    return x;
  }

  /**
   * Checks if is asn1 identifier start.
   *
   * @param ch the ch
   * @return true, if is asn1 identifier start
   */
  boolean isASN1IdentifierStart(char ch) {
    return Character.isLetter(ch);
  }

  /**
   * Checks if is asn1 identifier part.
   *
   * @param ch the ch
   * @return true, if is asn1 identifier part
   */
  boolean isASN1IdentifierPart(char ch) {
    return Character.isLetterOrDigit(ch) || ch == '-' || ch == '.';
  }
  
  /**
   * Scan identifier.
   *
   * @param s the s
   * @param index the index
   * @return the string
   * @see jnpad.text.syntax.SyntaxView#scanIdentifier(javax.swing.text.Segment, int)
   */
  @Override
  String scanIdentifier(Segment s, int index) {
    String string = new String(s.array, s.offset, s.count);
    int identifierLength = 0;
    if (!isASN1IdentifierStart(string.charAt(index))) 
      return jnpad.util.Utilities.EMPTY_STRING;
    for (identifierLength = 1; identifierLength < (string.length() - index); identifierLength++) {
      if (!isASN1IdentifierPart(string.charAt(index + identifierLength)))
        break;
    }
    return string.substring(index, index + identifierLength);
  }
  
  /**
   * Checks if is single line comment start.
   *
   * @param s the s
   * @param index the index
   * @return true, if is single line comment start
   * @see jnpad.text.syntax.SyntaxView#isSingleLineCommentStart(javax.swing.text.Segment, int)
   */
  @Override
  boolean isSingleLineCommentStart(Segment s, int index) {
    return s.charAt(index) == '-' && index + 1 < s.length() && s.charAt(index + 1) == '-';
  }

  /**
   * Scan single line comment.
   *
   * @param s the s
   * @param index the index
   * @return the int
   * @see jnpad.text.syntax.SyntaxView#scanSingleLineComment(javax.swing.text.Segment, int)
   */
  @Override
  int scanSingleLineComment(Segment s, int index) {
    index += 2;
    int commentLength;
    for (commentLength = 0; commentLength < (s.length() - index); commentLength++) {
      int idx = index + commentLength;
      char commentChar = s.charAt(idx);
      if (commentChar == jnpad.util.Utilities.LF)
        break;
      if (commentChar == '-' && idx - 1 >= 0 && s.charAt(idx - 1) == '-') {
        commentLength++;
        break;
      }
    }
    return commentLength + 2;
  }
  
  /**
   * Gets the scheme.
   * 
   * @return the scheme
   * @see jnpad.text.syntax.SyntaxView#getScheme()
   */
  @Override
  SyntaxScheme getScheme() {
    return scheme;
  }

}
