/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui;

import java.awt.Window;

import javax.swing.JOptionPane;

import jnpad.JNPadBundle;

/**
 * 
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class ReporterUtilities {
  /** Used for error messages. */
  public static final int     ERROR_MESSAGE       = JOptionPane.ERROR_MESSAGE;
  /** Used for information messages. */
  public static final int     INFORMATION_MESSAGE = JOptionPane.INFORMATION_MESSAGE;
  /** Used for warning messages. */
  public static final int     WARNING_MESSAGE     = JOptionPane.WARNING_MESSAGE;

  private static final String ERROR_TITLE         = JNPadBundle.getErrorTitle();
  private static final String INFORMATION_TITLE   = JNPadBundle.getInformationTitle();
  private static final String WARNING_TITLE       = JNPadBundle.getWarningTitle();

  private static final String OK_OPTION           = JNPadBundle.getOkOptionText();
  private static final String DETAILS_OPTION      = JNPadBundle.getDetailsOptionText();

  /** no instances */
  private ReporterUtilities() {
    super();
  }

  /**
   * Report error.
   *
   * @param window the window
   * @param message the message
   * @param title the title
   */
  public static void reportError(Window window, String message, String title) {
    reportMsg(window, ERROR_MESSAGE, message, title, null);
  }

  /**
   * Report error.
   *
   * @param window the window
   * @param message the message
   */
  public static void reportError(Window window, String message) {
    reportMsg(window, ERROR_MESSAGE, message, null, null);
  }
  
  /**
   * Report error.
   *
   * @param window the window
   * @param message the message
   * @param title the title
   * @param ex the throwable
   */
  public static void reportError(Window window, String message, String title, Throwable ex) {
    reportMsg(window, ERROR_MESSAGE, message, title, ex);
  }

  /**
   * Report info.
   *
   * @param window the window
   * @param message the message
   * @param title the title
   */
  public static void reportInfo(Window window, String message, String title) {
    reportMsg(window, INFORMATION_MESSAGE, message, title, null);
  }

  /**
   * Report info.
   *
   * @param window the window
   * @param message the message
   */
  public static void reportInfo(Window window, String message) {
    reportMsg(window, INFORMATION_MESSAGE, message, null, null);
  }

  /**
   * Report warning.
   *
   * @param window the window
   * @param message the message
   * @param title the title
   */
  public static void reportWarning(Window window, String message, String title) {
    reportMsg(window, WARNING_MESSAGE, message, title, null);
  }

  /**
   * Report warning.
   *
   * @param window the window
   * @param message the message
   */
  public static void reportWarning(Window window, String message) {
    reportMsg(window, WARNING_MESSAGE, message, null, null);
  }

  /**
   * Report msg.
   *
   * @param window the window
   * @param messageType the message type
   * @param message the message
   */
  public static void reportMsg(Window window, int messageType, String message) {
    reportMsg(window, messageType, message, null, null);
  }

  /**
   * Report msg.
   *
   * @param window the window
   * @param messageType the message type
   * @param message the message
   * @param title the title
   */
  public static void reportMsg(Window window, int messageType, String message, String title) {
    reportMsg(window, messageType, message, title, null);
  }

  /**
   * Report msg.
   *
   * @param window the window
   * @param messageType the message type
   * @param message the message
   * @param title the title
   * @param ex the throwable
   */
  public static void reportMsg(Window window, int messageType, String message, String title, Throwable ex) {
    if (messageType == ERROR_MESSAGE) {
      if (ex == null) {
        JOptionPane.showMessageDialog(window, message,
                                      (title == null) ? ERROR_TITLE : title,
                                      JOptionPane.ERROR_MESSAGE);
      }
      else {
        String[] options = { OK_OPTION, DETAILS_OPTION };
        int option = JOptionPane.showOptionDialog(window,
                                                  message,
                                                  (title == null) ? ERROR_TITLE : title,
                                                  JOptionPane.DEFAULT_OPTION,
                                                  JOptionPane.ERROR_MESSAGE,
                                                  null,
                                                  options,
                                                  options[0]);
        if (option == 1) {
          new ExceptionDialog(window, (title == null) ? ERROR_TITLE : title, message, ex, true);
        }
      }
    }
    else if (messageType == INFORMATION_MESSAGE) {
      JOptionPane.showMessageDialog(window, message,
                                    (title == null) ? INFORMATION_TITLE : title,
                                    JOptionPane.INFORMATION_MESSAGE);
    }
    else if (messageType == WARNING_MESSAGE) {
      JOptionPane.showMessageDialog(window, message,
                                    (title == null) ? WARNING_TITLE : title,
                                    JOptionPane.WARNING_MESSAGE);
    }
    else {
      JOptionPane.showMessageDialog(window, 
                                    JNPadBundle.getString("ReporterUtils.message.error"),  //$NON-NLS-1$
                                    ERROR_TITLE,
                                    JOptionPane.ERROR_MESSAGE);
    }
  }

}
