/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.table;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;

import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.table.TableCellRenderer;

import jnpad.ui.ColorUtilities;
import jnpad.ui.plaf.LAFUtils;

/**
 * The Class JNPadTableCellRenderer.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public abstract class JNPadTableCellRenderer extends JPanel implements TableCellRenderer {
  /** The supports focus border. */
  protected boolean           supportsFocusBorder;
  
  /** The tooltip enabled. */
  protected boolean           tooltipEnabled;
  
  /** The alternate row color enabled. */
  protected boolean           alternateRowColorEnabled = true;
  
  /** The alternate row color. */
  protected Color             alternateRowColor;

  /** The Constant DEFAULT_NO_FOCUS_BORDER. */
  private static final Border DEFAULT_NO_FOCUS_BORDER  = new EmptyBorder(1, 1, 1, 1);

  /** UID. */
  private static final long   serialVersionUID         = -6184651210115838365L;

  /**
   * Instantiates a new jNPad's table cell renderer.
   */
  protected JNPadTableCellRenderer() {
    this(true);
  }

  /**
   * Instantiates a new jNPad's table cell renderer.
   *
   * @param tooltipEnabled the tooltip enabled
   */
  protected JNPadTableCellRenderer(boolean tooltipEnabled) {
    this.tooltipEnabled = tooltipEnabled;
    setOpaque(true);
    setSupportsFocusBorder(true);
  }

  /**
   * Instantiates a new jNPad's table cell renderer.
   *
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColorEnabled the alternate row color enabled
   */
  public JNPadTableCellRenderer(boolean tooltipEnabled, boolean alternateRowColorEnabled) {
    this(tooltipEnabled);
    setAlternateRowColorEnabled(alternateRowColorEnabled);
  }

  /**
   * Instantiates a new jNPad's table cell renderer.
   *
   * @param alternateRowColor the alternate row color
   */
  public JNPadTableCellRenderer(Color alternateRowColor) {
    this(true);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Instantiates a new jNPad's table cell renderer.
   *
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColor the alternate row color
   */
  public JNPadTableCellRenderer(boolean tooltipEnabled, Color alternateRowColor) {
    this(tooltipEnabled);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Sets the supports focus border.
   *
   * @param supportsFocusBorder the new supports focus border
   */
  public void setSupportsFocusBorder(boolean supportsFocusBorder) {
    this.supportsFocusBorder = supportsFocusBorder;
    if (supportsFocusBorder) {
      setBorder(getNoFocusBorder());
    }
  }

  /**
   * Gets the supports focus border.
   *
   * @return the supports focus border
   */
  public boolean getSupportsFocusBorder() {return supportsFocusBorder;}

  /**
   * Gets the no focus border.
   *
   * @return the no focus border
   */
  protected static Border getNoFocusBorder() {
    Border border = UIManager.getBorder("Table.cellNoFocusBorder"); //$NON-NLS-1$
    if (border != null)
      return border;
    return DEFAULT_NO_FOCUS_BORDER;
  }

  /**
   * Checks if is tooltip enabled.
   *
   * @return true, if is tooltip enabled
   */
  public boolean isTooltipEnabled() {return tooltipEnabled;}

  /**
   * Sets the tooltip enabled.
   *
   * @param tooltipEnabled the new tooltip enabled
   */
  public void setTooltipEnabled(boolean tooltipEnabled) {
    this.tooltipEnabled = tooltipEnabled;
  }

  /**
   * Checks if is alternate row color enabled.
   *
   * @return true, if is alternate row color enabled
   */
  public boolean isAlternateRowColorEnabled() {return alternateRowColorEnabled;}

  /**
   * Sets the alternate row color enabled.
   *
   * @param alternateRowColorEnabled the new alternate row color enabled
   */
  public void setAlternateRowColorEnabled(boolean alternateRowColorEnabled) {
    this.alternateRowColorEnabled = alternateRowColorEnabled;
  }

  /**
   * Gets the alternate row color.
   *
   * @return the alternate row color
   */
  public Color getAlternateRowColor() { return alternateRowColor;}

  /**
   * Sets the alternate row color.
   *
   * @param alternateRowColor the new alternate row color
   */
  public void setAlternateRowColor(Color alternateRowColor) {
    this.alternateRowColor = alternateRowColor;
    this.alternateRowColorEnabled = alternateRowColor != null;
  }

  /**
   * Sets the row background.
   *
   * @param c the new row background
   */
  protected void setRowBackground(Color c) {
    setBackground(c);
  }

  /**
   * Sets the row foreground.
   *
   * @param c the new row foreground
   */
  protected void setRowForeground(Color c) {
    setForeground(c);
  }

  /**
   * Sets the row font.
   *
   * @param font the new row font
   */
  protected void setRowFont(Font font) {
    setFont(font);
  }

  /**
   * Sets the colors.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   */
  protected void setColors(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    if (isSelected && table.isEnabled()) {
      setRowForeground(table.getSelectionForeground());
      setRowBackground(table.getSelectionBackground());
    }
    else if (!table.isEnabled()) {
      setRowForeground(LAFUtils.getTextFieldInactiveForeground());
      setRowBackground(LAFUtils.getTextFieldInactiveBackground());
    }
    else {
      Color bg;
      Color fg;
      if (alternateRowColorEnabled && (row % 2 == 0)) {
        if (alternateRowColor != null) {
          bg = alternateRowColor;
          fg = ColorUtilities.computeForeground(bg);
        }
        else {
          Color alternateColor = UIManager.getColor("Table.alternateRowColor"); //$NON-NLS-1$
          if (alternateColor != null) {
            bg = alternateColor;
          }
          else {
            bg = table.getBackground();
            if (LAFUtils.isNimbusLAF() && bg instanceof javax.swing.plaf.UIResource) // 4Nimbus
              bg = ColorUtilities.createPureColor(bg);
          }
          fg = table.getForeground();
        }
      }
      else {
        bg = table.getBackground();
        if (LAFUtils.isNimbusLAF() && bg instanceof javax.swing.plaf.UIResource) // 4Nimbus
          bg = ColorUtilities.createPureColor(bg);
        fg = table.getForeground();
      }
      setRowForeground(fg);
      setRowBackground(bg);
    }
  }

  /**
   * Sets the value.
   *
   * @param table the table
   * @param value the value
   * @param row the row
   * @param column the column
   */
  protected abstract void setValue(JTable table, Object value, int row, int column);

  /**
   * Sets the font.
   *
   * @param table the table
   * @param value the value
   * @param row the row
   * @param column the column
   */
  protected void setFont(JTable table, Object value, int row, int column) {
    setRowFont(table.getFont());
  }

  /**
   * Sets the state.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   */
  protected void setState(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    //empty
  }

  /**
   * Sets the border.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   */
  protected void setBorder(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    if (supportsFocusBorder) {
      if (hasFocus) {
        Border border = null;
        if (isSelected) {
          border = UIManager.getBorder("Table.focusSelectedCellHighlightBorder"); //$NON-NLS-1$
        }
        if (border == null) {
          border = LAFUtils.getTableFocusCellHighlightBorder();
        }
        setBorder(border);
        if (!isSelected && table.isCellEditable(row, column)) {
          setRowForeground(LAFUtils.getTableFocusCellForeground());
          setRowBackground(LAFUtils.getTableFocusCellBackground());
        }
      }
      else {
        setBorder(getNoFocusBorder());
      }
    }
  }

  /**
   * Gets the table cell renderer component.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   * @return the table cell renderer component
   * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
   */
  @Override
  public Component getTableCellRendererComponent(JTable table,
                                                 Object value,
                                                 boolean isSelected,
                                                 boolean hasFocus,
                                                 int row,
                                                 int column) {

    setColors(table, value, isSelected, hasFocus, row, column);

    setFont(table, value, row, column);

    setBorder(table, value, isSelected, hasFocus, row, column);

    setState(table, value, isSelected, hasFocus, row, column);

    setValue(table, value, row, column);

    return this;
  }

}
